/*
 *
 *    OPEN-XCHANGE legal information *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.fileitem.impl;

import static org.apache.commons.lang.StringUtils.isEmpty;
import static org.apache.commons.lang.StringUtils.isNotBlank;
import static org.apache.commons.lang.StringUtils.isNotEmpty;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import com.google.common.base.Throwables;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.config.ConfigurationService;
import com.openexchange.config.WildcardNamePropertyFilter;
import com.openexchange.exception.OXException;
import com.openexchange.imageconverter.api.FileItemException;
import com.openexchange.imageconverter.api.IFileItemService;
import com.openexchange.imageconverter.api.ISubGroup;
import com.zaxxer.hikari.pool.HikariPool;
import liquibase.Liquibase;
import liquibase.changelog.ChangeSet;
import liquibase.changelog.ChangeSet.ExecType;
import liquibase.changelog.ChangeSet.RunStatus;
import liquibase.changelog.DatabaseChangeLog;
import liquibase.changelog.visitor.ChangeExecListener;
import liquibase.database.Database;
import liquibase.database.jvm.JdbcConnection;
import liquibase.exception.LiquibaseException;
import liquibase.resource.ResourceAccessor;

/**
 * {@link FileItemDatabase}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public class FileItemDatabase implements ChangeExecListener {

    // Predefined properties
    /**
     * KEY_CREATEATE
     */
    final public static String KEY_CREATEDATE = "CreateDate";

    /**
     * KEY_MODIFICATIONDATE
     */
    final public static String KEY_MODIFICATIONDATE = "ModificationDate";

    /**
     * KEY_LENGTH
     */
    final public static String KEY_LENGTH = "Length";

    /**
     * DEFAULT_KEYS
     */
    final public static String[] DEFAULT_KEYS = { KEY_CREATEDATE, KEY_MODIFICATIONDATE, KEY_LENGTH };

    /**
     * VIRTKEY_DIFF_LENGTH
     */
    final public static String VIRTKEY_DIFF_LENGTH = "DiffLength";

    /**
     * VIRTKEY_DIFF_LENGTH
     */
    final public static String DB_VARCHAR_TYPE = "VARCHAR(190)";

    /**
     * @param key
     * @return
     */
    public static boolean implIsDefaultKey(@NonNull final Object key) {
        return ArrayUtils.contains(DEFAULT_KEYS, key);
    }

    /**
     * Initializes a new {@link FileItemDatabase}.
     */
    public FileItemDatabase(@NonNull ConfigurationService configService) throws FileItemException {
        super();

        m_configService = configService;

        implCreateConnectionPools();
        implUpdateDatabase();

        synchronized (m_groupMap) {
            implReadGroups();
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see liquibase.changelog.visitor.ChangeExecListener#willRun(liquibase.changelog.ChangeSet, liquibase.changelog.DatabaseChangeLog, liquibase.database.Database, liquibase.changelog.ChangeSet.RunStatus)
     */
    @Override
    public void willRun(ChangeSet changeSet, DatabaseChangeLog databaseChangeLog, Database database, RunStatus runStatus) {
        if (null != changeSet) {
            implLogChangeEvent(runStatus, changeSet.getDescription());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see liquibase.changelog.visitor.ChangeExecListener#ran(liquibase.changelog.ChangeSet, liquibase.changelog.DatabaseChangeLog, liquibase.database.Database, liquibase.changelog.ChangeSet.ExecType)
     */
    @Override
    public void ran(ChangeSet changeSet, DatabaseChangeLog databaseChangeLog, Database database, ExecType execType) {
        if (null != changeSet) {
            implLogChangeEvent(execType, changeSet.getDescription());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see liquibase.changelog.visitor.ChangeExecListener#rolledBack(liquibase.changelog.ChangeSet, liquibase.changelog.DatabaseChangeLog, liquibase.database.Database)
     */
    @Override
    public void rolledBack(ChangeSet changeSet, DatabaseChangeLog databaseChangeLog, Database database) {
        if (null != changeSet) {
            implLogChangeEvent(null, changeSet.getDescription());
        }
    }

    // - Database access methods -----------------------------------------------

    /**
     * @return The type of JDBC database read and write connections,
     *  allowing to optimize SQL queries
     */
    public IFileItemService.DatabaseType getDatabaseType() {
        return m_databaseType;
    }

    /**
     * @param groupId
     * @param customKeyN
     * @throws FileItemException
     */
    public void registerGroup(@NonNull final String groupId, @Nullable String... customKeyN) throws FileItemException {
        Set<String> propertyKeySet = null;

        synchronized (m_groupMap) {
            if (!m_groupMap.containsKey(groupId)) {
                // create Content and Property tables for new group
                implCreateGroupTables(groupId);

                synchronized (m_groupMap) {
                     propertyKeySet = implGetKeySet(groupId, false);

                     // insert the default property keys, that are always available
                     try {
                         implInsertPropertyKey(groupId, KEY_CREATEDATE, "BIGINT(20)", "NOT NULL");
                     } catch (@SuppressWarnings("unused") FileItemException e) {
                         FileItemUtils.logError("FileItemService received exception when trying to register 'CreateData' default key for new group: " + groupId);
                     }

                     try {
                         implInsertPropertyKey(groupId, KEY_MODIFICATIONDATE, "BIGINT(20)", "NOT NULL");
                     } catch (@SuppressWarnings("unused") FileItemException e) {
                         FileItemUtils.logError("FileItemService received exception when trying to register 'ModificationData' default key for new group: " + groupId);
                     }

                     try {
                         implInsertPropertyKey(groupId, KEY_LENGTH, "BIGINT(20)", "NOT NULL");
                     } catch (@SuppressWarnings("unused") FileItemException e) {
                         FileItemUtils.logError("FileItemService received exception when trying to register 'Length' default key for new group: " + groupId);
                     }
                }
            } else {
                propertyKeySet = implGetKeySet(groupId, false);
            }
        }

        if (FileItemUtils.isValid(customKeyN)) {
            final Set<String> customPropertyKeySet = implGetKeySet(groupId, true);

            synchronized (m_groupMap) {
                for (final String curKey : customKeyN) {
                    if (FileItemUtils.isValid(curKey) && !propertyKeySet.contains(curKey)) {
                        try {
                            implInsertPropertyKey(groupId, curKey, "TEXT", "NULL");

                            if (!implIsDefaultKey(curKey)) {
                                customPropertyKeySet.add(curKey);
                            }

                        } catch (FileItemException e) {
                            FileItemUtils.logError(STR_BUILDER("FileItemService received exception when trying to register customKey for groupId: ").
                                append(groupId).append(" => ").append(curKey).
                                append(" (").append(Throwables.getRootCause(e).getMessage()).append(')').toString());
                        }
                    }
                }
            }
        }
    }

    /**
     * @param groupId
     * @return
     */
    public Set<String> getPropertyKeys(@NonNull final String groupId) {
        synchronized (m_groupMap) {
            return implGetKeySet(groupId, false);
        }
    }

    /**
     * @param groupId
     * @return
     */
    public Set<String> getCustomPropertyKeys(@NonNull final String groupId) {
        synchronized (m_groupMap) {
            return implGetKeySet(groupId, true);
        }
    }

    /**
     * @return
     */
    public boolean hasCustomPropertyKey(@NonNull final String groupId, @NonNull final String customKey) {
        synchronized (m_groupMap) {
            return implGetKeySet(groupId, true).contains(customKey);
        }
    }

    /**
     * @param fileStoreId
     * @param fileItem
     * @param fileItemProperties
     * @throws FileItemException
     */
    public boolean createEntry(@NonNull final FileItem fileItem, @NonNull final FileStoreData fileStoreData, @NonNull final FileItemProperties fileItemProperties) throws FileItemException {
        // check FileItem
        if (fileItem.isInvalid()) {
            throw new FileItemException("FileItem is not valid: " + fileItem.toString());
        }

        // check FileStoreData
        if (fileStoreData.isInvalid()) {
            throw new FileItemException("FileStoreData is not valid: " + fileStoreData.toString());
        }

        try (final Connection con = getWriteConnection()) {
            final String groupId = fileItem.getGroupId();
            final String subGroupId = fileItem.getSubGroupId();
            final String fileId = fileItem.getFileId();
            final Properties customProperties = fileItemProperties.getCustomProperties();
            final Set<Object> customKeys = customProperties.keySet();

            final String insertContentSql = STR_BUILDER("INSERT INTO ").append(TBL_CONTENT(groupId, null)).
                append(" (FileStoreNumber,FileStoreId,SubGroupId,FileId) VALUES (?,?,?,?)").toString();

            final StringBuilder insertPropertiesBuilderStart = STR_BUILDER("INSERT INTO ").append(TBL_PROPERTIES(groupId, null)).
                append(" (FileStoreNumber,FileStoreId,CreateDate,ModificationDate,Length");

            final StringBuilder insertPropertiesBuilderEnd = STR_BUILDER(" VALUES (?,?,?,?,?");

            if (customKeys.size() > 0) {
                for (final Object curCustomKey : customKeys) {
                    insertPropertiesBuilderStart.append(',').append(curCustomKey);
                    insertPropertiesBuilderEnd.append(",?");
                }
            }

            final String insertPropertiesSql = insertPropertiesBuilderStart.append(") ").append(insertPropertiesBuilderEnd).append(')').toString();

            // Insert new data into Content table if
            // fileElement and FileStoreData are valid
            try (final PreparedStatement fcStmt = con.prepareStatement(insertContentSql);
                 final PreparedStatement fpStmt = con.prepareStatement(insertPropertiesSql)) {

                int index = 0;

                // create entry in Content table
                fcStmt.setInt(++index, fileStoreData.getFileStoreNumber());
                fcStmt.setString(++index, fileStoreData.getFileStoreId());
                fcStmt.setString(++index, subGroupId);
                fcStmt.setString(++index, fileId);

                // create properties
                index = 0;

                fpStmt.setInt(++index, fileStoreData.getFileStoreNumber());
                fpStmt.setString(++index, fileStoreData.getFileStoreId());

                fpStmt.setLong(++index, fileItemProperties.getCreateDateMillis());
                fpStmt.setLong(++index, fileItemProperties.getModificationDateMillis());
                fpStmt.setLong(++index, fileItemProperties.getLength());

                // set given key values
                if (customKeys.size() > 0) {
                    for (final Object curCustomKey : customKeys) {
                        fpStmt.setString(++index, fileItemProperties.getCustomKeyValue(curCustomKey.toString()));
                    }
                }

                // execute both statements and commit
                fcStmt.executeUpdate();
                fpStmt.executeUpdate();

                con.commit();

                // if inserting of new entries into both
                // tables went fine, set fileStoreData
                // at fileElement and set return value to true
                fileItem.setFileStoreData(fileStoreData);
                return true;
            }
        } catch (Exception e) {
            throw new FileItemException(e);
        }
    }

    /**
     * @param groupId
     * @param subGroupId
     * @return
     */
    public FileItem[] getFileItems(@NonNull final String groupId, @NonNull final String subGroupId) throws FileItemException {
        final String querySql = STR_BUILDER("SELECT fc.FileStoreNumber, fc.FileStoreId, fc.SubGroupId, fc.FileId").
            append(" FROM ").append(TBL_CONTENT(groupId, "fc")).
            append(" WHERE fc.SubGroupId=?").toString();

        try (final Connection con = getReadConnection();
             final PreparedStatement queryStmt = con.prepareStatement(querySql)) {

            queryStmt.setString(1, subGroupId);

            return implExecuteQuery(con, groupId, queryStmt);
        } catch (Exception e) {
            throw new FileItemException(e);
        }
    }

    /**
     * @param groupId
     * @param subGroupId
     * @return
     */
    public FileItem[] getFileItems(@NonNull final String groupId, @NonNull final Properties properties) throws FileItemException {
        final Properties customProperties = new Properties();

        for (final Object curKey : properties.keySet()) {
            customProperties.put("fp." + curKey, properties.getProperty(curKey.toString()));
        }

        final String querySql = FileItemUtils.createSqlWithPropertyVariables(
            STR_BUILDER("SELECT fc.FileStoreNumber, fc.FileStoreId, fc.SubGroupId, fc.FileId FROM ").append(TBL_CONTENT(groupId, "fc")).
            append(" INNER JOIN ").append(TBL_PROPERTIES(groupId, "fp")).
            append(" ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId ").
            append(" WHERE ").toString(),
            customProperties,
            " AND ");

        try (final Connection con = getReadConnection();
             final PreparedStatement queryStmt = con.prepareStatement(querySql)) {

            FileItemUtils.setStatementValues(queryStmt, customProperties.values().toArray(), 1);

            return implExecuteQuery(con, groupId, queryStmt);
        } catch (Exception e) {
            throw new FileItemException(e);
        }
    }

    /**
     * @param groupId
     * @param subGroupId
     * @return
     */
    public FileItem[] getFileItemsByCustomQuerySQL(@NonNull final String groupId, @NonNull final String customQuerySQL, Object... returnValues) throws FileItemException {
        try (final Connection con = getReadConnection();
             final PreparedStatement customQueryStmt = con.prepareStatement(customQuerySQL)) {

            return implExecuteQuery(con, groupId, customQueryStmt, returnValues);
        } catch (Exception e) {
            throw new FileItemException(e);
        }
    }

    /**
     * @param groupId
     * @param subGroupId
     * @return
     */
    public long getGroupCount() {
        synchronized (m_groupMap) {
            return m_groupMap.size();
        }
    }

    /**
     * @param groupId
     * @param subGroupId
     * @return
     */
    public long getSubGroupCount(@NonNull final String groupId) throws FileItemException {
        try {
            return implExecuteGetLong(STR_BUILDER("SELECT COUNT(DISTINCT fc.SubGroupId) FROM ").append(TBL_CONTENT(groupId, "fc")).toString());
        } catch (SQLException e) {
            throw new FileItemException("Exception caught when getting SubGroupIds", e);
        }
    }

    /**
     * @param groupId
     * @return
     */
    public boolean contains(@NonNull final String groupId) {
        synchronized (m_groupMap) {
            return m_groupMap.containsKey(groupId);
        }
    }

    /**
     * @param groupId
     * @return
     */
    public boolean contains(@NonNull final String groupId, @NonNull final String subGroupId) throws FileItemException {
        return implExecuteContains(groupId, STR_BUILDER("fc.SubGroupId='").append(subGroupId).append("'").toString());
    }

    /**
     * @param groupId
     * @return
     */
    public boolean contains(@NonNull final String groupId, @NonNull final String subGroupId,@NonNull final String fileId) throws FileItemException {
        return implExecuteContains(groupId,
            STR_BUILDER("fc.SubGroupId='").append(subGroupId).append("'").
                append(" AND fc.FileId='").append(fileId).append("'").toString());
    }

    /**
     * @param groupId
     * @param subGroupId
     * @return
     */
    public String[] getSubGroupIds(@NonNull final String groupId) throws FileItemException {
        try {
            return implExecuteGetIds(STR_BUILDER("SELECT DISTINCT fc.SubGroupId FROM ").append(TBL_CONTENT(groupId, "fc")).toString());
        } catch (SQLException e) {
            throw new FileItemException("Exception caught when getting SubGroupIds", e);
        }
    }

    /**
     * @param groupId
     * @param subGroupId
     * @return
     */
    public String[] getSubGroupIdsBy(@NonNull final String groupId, @Nullable final String whereClause, @Nullable String limitClause) throws FileItemException {
        final StringBuilder querySql = STR_BUILDER("SELECT fc.SubGroupId, SUM(fp.Length) AS SubGroupLength, MIN(fp.ModificationDate) AS SubGroupDate FROM ").
            append(TBL_CONTENT(groupId, "fc")).append(" INNER JOIN ").append(TBL_PROPERTIES(groupId, "fp")).
            append(" ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId ");

        if (isNotBlank(whereClause)) {
            querySql.append(" WHERE ").append(whereClause);
        }

        querySql.append(" GROUP BY fc.SubGroupId ORDER BY SubGroupDate ASC, SubGroupLength DESC");

        if (isNotBlank(limitClause)) {
            querySql.append(" LIMIT ").append(limitClause);
        }

        try {
            return implExecuteGetIds(querySql.toString());
        } catch (SQLException e) {
            throw new FileItemException("Exception caught when getting SubGroupIds by WHERE and/or LIMIT clause", e);
        }
    }

    /**
     * @param groupId
     * @param whereClause
     * @param limitClause
     * @param subGroupConsumer
     * @return
     * @throws FileItemException
     */
    public long getSubGroupsBy(String groupId, String whereClause, String limitClause, Consumer<ISubGroup> subGroupConsumer) throws FileItemException {
        final StringBuilder querySql = STR_BUILDER("SELECT fc.SubGroupId,").
            append(" SUM(fp.Length) AS SubGroupLength,").
            append(" MIN(fp.CreateDate) AS SubGroupCreateDate,").
            append(" MAX(fp.ModificationDate) AS SubGroupModificationDate").
            append(" FROM ").append(TBL_CONTENT(groupId, "fc")).append(" INNER JOIN ").append(TBL_PROPERTIES(groupId, "fp")).
            append(" ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId ");

        if (isNotBlank(whereClause)) {
            querySql.append(" WHERE ").append(whereClause);
        }

        querySql.append(" GROUP BY fc.SubGroupId ORDER BY SubGroupModificationDate ASC, SubGroupLength DESC");

        if (isNotBlank(limitClause)) {
            querySql.append(" LIMIT ").append(limitClause);
        }

        try (final Connection con = getReadConnection();
            final PreparedStatement stmt = con.prepareStatement(querySql.toString())) {

            long ret = 0;

            try (final ResultSet resultSet = stmt.executeQuery()) {
                if (null != resultSet) {
                    while (resultSet.next()) {
                        final String subGroupId = resultSet.getString(1);
                        final long subGroupLength = resultSet.getLong(2);
                        final long subGroupCreateDateMillis = resultSet.getLong(3);
                        final long subGroupModificationDateMillis = resultSet.getLong(4);

                        ++ret;

                        if (null != subGroupConsumer) {
                            subGroupConsumer.accept(new ISubGroup() {

                                @Override
                                public String getSubGroupId() {
                                    return subGroupId;
                                }

                                @Override
                                public long getLength() {
                                    return subGroupLength;
                                }

                                @Override
                                public Date getCreateDate() {
                                    return new Date(subGroupCreateDateMillis);
                                }

                                @Override
                                public Date getModificationDate() {
                                    return new Date(subGroupModificationDateMillis);
                                }
                            });
                        }
                    }
                }
            } finally {
                // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
                con.commit();
            }

            return ret;
        } catch (SQLException e) {
            throw new FileItemException("Exception caught when getting SubGroupIds by WHERE and/or LIMIT clause", e);
        }
    }

    /**
     * @param groupId
     * @param subGroupId
     * @param properties
     * @return
     * @throws FileItemException
     */
    public long getTotalLength(@Nullable final String groupId, @Nullable final String subGroupId, @Nullable final Properties properties) throws FileItemException {
        final Properties whereProperties = new Properties();
        StringBuilder selectSumSql = STR_BUILDER("SELECT SUM(fp.Length) from ").append(TBL_PROPERTIES(groupId, "fp"));
        Object[] whereValues = null;

        if (FileItemUtils.isValid(subGroupId)) {
            whereProperties.put("fc.SubGroupId", subGroupId);
        }

        if (null != properties) {
            for (final Object curKey : properties.keySet()) {
                whereProperties.put("fp." + curKey, properties.getProperty(curKey.toString()));
            }
        }

        if (whereProperties.size() > 0) {
            selectSumSql = STR_BUILDER(FileItemUtils.createSqlWithPropertyVariables(selectSumSql.
                    append(" INNER JOIN ").append(TBL_CONTENT(groupId, "fc")).
                    append(" ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId WHERE").toString(),
                whereProperties,
                " AND "));

            whereValues = whereProperties.values().toArray(new Object[whereProperties.size()]);
        }

        try {
            return implExecuteGetLong(selectSumSql.toString(), whereValues);
        } catch (SQLException e) {
            throw new FileItemException("Exception caught when getting summed up length of FileItems", e);
        }
    }

    /**
     * @param groupId
     * @param subGroupId
     * @return
     */
    public String[] getGroupIds() {
        synchronized (m_groupMap) {
            return m_groupMap.keySet().toArray(new String[m_groupMap.size()]);
        }
    }

    /**
     * @param fileItem
     * @return
     * @throws FileItemException
     */
    public FileStoreData getFileStoreData(@NonNull final FileItem fileItem) throws FileItemException {
        final String selectSql = STR_BUILDER("SELECT fc.FileStoreNumber, fc.FileStoreId FROM ").append(TBL_CONTENT(fileItem.getGroupId(), "fc")).
            append(" WHERE fc.SubGroupId=? AND fc.FileId=?").toString();

        try (final Connection con = getReadConnection();
             final PreparedStatement stmt = con.prepareStatement(selectSql)) {

            stmt.setString(1, fileItem.getSubGroupId());
            stmt.setString(2, fileItem.getFileId());

            try (final ResultSet resultSet = stmt.executeQuery()) {
                if ((null != resultSet) && resultSet.first()) {
                    return new FileStoreData(resultSet.getInt(1), resultSet.getString(2));
                }
            } finally {
                // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
                con.commit();
            }
        } catch (SQLException e) {
            throw new FileItemException(e);
        }

        return null;
    }

    /**
     * @param fileStoreId.get
     * @param fileProperties
     * @throws FileItemException
     */
    public FileItemProperties getFileItemProperties(@NonNull FileItem fileItem) throws FileItemException {
        final String groupId = fileItem.getGroupId();
        final Set<String> customKeys = getCustomPropertyKeys(groupId);
        final StringBuilder selectSqlBuilderStart = STR_BUILDER("SELECT fp.FileStoreNumber,fp.FileStoreId,fp.CreateDate,fp.ModificationDate,fp.Length");
        final String selectSqlEnd = STR_BUILDER(" FROM ").append(TBL_CONTENT(groupId, "fc")).
            append(" INNER JOIN ").append(TBL_PROPERTIES(groupId, " fp ")).
            append(" ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId").
            append(" WHERE fc.SubGroupId=? AND fc.FileId=?").toString();

        // append all property keys for given groupId to Sql statement
        for (final String curCustomKey : customKeys) {
            selectSqlBuilderStart.append(",fp." + curCustomKey);
        }

        final String selectSql = selectSqlBuilderStart.append(selectSqlEnd).toString();

        try (final Connection con = getReadConnection(); final PreparedStatement stmt = con.prepareStatement(selectSql)) {

            stmt.setString(1, fileItem.getSubGroupId());
            stmt.setString(2, fileItem.getFileId());

            try (final ResultSet resultSet = stmt.executeQuery()) {

                if ((null != resultSet) && resultSet.first()) {
                    int index = 0;

                    final int fileStoreNumber = resultSet.getInt(++index);
                    final String fileStoreId = resultSet.getString(++index);

                    fileItem.setFileStoreData(new FileStoreData(fileStoreNumber, fileStoreId));

                    final FileItemProperties ret = new FileItemProperties(customKeys);

                    ret.setCreateDateMillis(resultSet.getLong(++index));
                    ret.setModificationDateMillis(resultSet.getLong(++index));
                    ret.setLength(resultSet.getLong(++index));

                    // read all key property values for the
                    // given groupId and set at result FileItemProperties
                    for (final String curCustomKey : customKeys) {
                        ret.setCustomKeyValue(curCustomKey, resultSet.getString(++index));
                    }

                    return ret;
                }
            } finally {
                // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
                con.commit();
            }
        } catch(SQLException e) {
           throw new FileItemException(e);
        }

        return null;
    }

    /**
     * @param fileStoreData
     * @param fileProperties
     * @throws FileItemException
     */
    public void updateProperties(@NonNull final FileStoreData fileStoreData, @NonNull final String groupId, @NonNull final Properties properties) throws FileItemException {
        if (properties.size() > 0) {
            final Properties setProperties = new Properties();

            for (final Object curKey : properties.keySet()) {
                setProperties.put("fp." + curKey, properties.get(curKey));
            }

            final String sqlStmt = FileItemUtils.createSqlWithPropertyVariables(
                STR_BUILDER("UPDATE ").append(TBL_PROPERTIES(groupId, "fp")).append(" SET").toString(),
                    setProperties,
                    ",",
                    " WHERE fp.FileStoreNumber=? AND fp.FileStoreId=?");

            try (final Connection con = getWriteConnection();
                 final PreparedStatement stmt = con.prepareStatement(sqlStmt)) {

                int curIndex = FileItemUtils.setStatementValues(stmt, setProperties.values().toArray(), 1);

                stmt.setInt(curIndex++, fileStoreData.getFileStoreNumber());
                stmt.setString(curIndex++, fileStoreData.getFileStoreId());

                stmt.executeUpdate();
                con.commit();
            } catch (SQLException e) {
                throw new FileItemException(e);
            }
        }
    }

    /**
     * @param fileStoreId
     * @throws FileItemException
     */
    public FileStoreData[] deleteEntry(@NonNull final String groupId, @NonNull final String subGroupId, @NonNull final String fileId) throws FileItemException {
        final Properties whereProperties = new Properties();

        whereProperties.put("fc.SubGroupId", subGroupId);
        whereProperties.put("fc.FileId", fileId);

        return implExecuteDelete(groupId, whereProperties);
    }

    /**
     * @param fileStoreId
     * @throws FileItemException
     */
    public FileStoreData[] deleteByGroupSubgroup(@NonNull final String groupId, final String subGroupId) throws FileItemException {

        if (FileItemUtils.isValid(subGroupId)) {
            final Properties whereProperties = new Properties();

            whereProperties.put("fc.SubGroupId", subGroupId);

            return implExecuteDelete(groupId, whereProperties);
        }

        return implExecuteDelete(groupId);
    }

    /**
     * @param groupId
     * @param subGroupIds
     * @return
     * @throws FileItemException
     */
    public FileStoreData[] deleteByGroupSubgroups(@NonNull final String groupId, @NonNull final String[] subGroupIds) throws FileItemException {
        final int count = subGroupIds.length;

        if (count > 0) {
            final List<FileStoreData> ret = new ArrayList<>();
            final String whereBeginStr = STR_BUILDER("fc.SubGroupId IN (").toString();
            final int blockCount = 1024;
            int curCount = 0;

            // remove keys in blocks with a maximum of blockCount keys
            do {
                final StringBuilder whereBuilder = STR_BUILDER(whereBeginStr);

                for (int i = 0, loopEnd = Math.min(blockCount, count - curCount); i < loopEnd; ++i, ++curCount) {
                    if (i > 0) {
                        whereBuilder.append(',');
                    }

                    whereBuilder.append("'").append(subGroupIds[curCount]).append("'");
                }

                ret.addAll(Arrays.asList(implExecuteDelete(groupId, whereBuilder.append(')').toString())));
            } while (curCount < count);

            return ret.toArray(new FileStoreData[ret.size()]);
        }

        return new FileStoreData[0];
    }

    /**
     * @param fileStoreId
     * @throws FileItemException
     */
    public FileStoreData[] deleteByProperties(@NonNull final String groupId, @NonNull final Properties properties) throws FileItemException {
        final Properties whereProperties = new Properties();

        for (final Object curKey : properties.keySet()) {
            whereProperties.put("fp." + curKey, properties.getProperty(curKey.toString()));
        }

        return implExecuteDelete(groupId, whereProperties);
    }

    //  - Public interface to retrieve connections -------------------------

    /**
     * getReadConnection
     */
    public Connection getReadConnection() throws FileItemException {
        if (null == m_readConnectionPool) {
            throw new FileItemException("No read connection pool available. Please check logging output and setup!");
        }

        try {
            return m_readConnectionPool.getConnection();
        } catch (SQLException e) {
            throw new FileItemException("Exception caught when getting read connection", e);
        }
    }

    /**
     * getWriteConnection
     */
    public Connection getWriteConnection() throws FileItemException {
        if (null == m_writeConnectionPool) {
            throw new FileItemException("No write connection pool available. Please check logging output and setup!");
        }

        try {
            return m_writeConnectionPool.getConnection();
        } catch (SQLException e) {
            throw new FileItemException("Exception caught when getting write connection", e);
        }
    }

    // - Implementation --------------------------------------------------------

    /**
     * @param selectSql
     * @param stmtValues
     * @return
     * @throws FileItemException
     * @throws SQLException
     */
    final String[] implExecuteGetIds(@NonNull final String selectSql, final Object... stmtValues) throws FileItemException, SQLException {
        try (final Connection con = getReadConnection();
             final PreparedStatement stmt = con.prepareStatement(selectSql)) {

            FileItemUtils.setStatementValues(stmt, stmtValues, 1);

            try (final ResultSet resultSet = stmt.executeQuery()) {
                final List<String> idList = new ArrayList<>();

                if (null != resultSet) {
                    while (resultSet.next()) {
                        idList.add(resultSet.getString(1));
                    }
                }

                return idList.toArray(new String[idList.size()]);
            } finally {
                // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
                con.commit();
            }
        }
    }

    /**
     * @param selectSql
     * @param stmtValues
     * @return
     * @throws FileItemException
     * @throws SQLException
     */
    final long implExecuteGetLong(@NonNull final String selectLongReturnSql, final Object... stmtValues) throws FileItemException, SQLException {
        try (final Connection con = getReadConnection();
             final PreparedStatement stmt = con.prepareStatement(selectLongReturnSql)) {

            FileItemUtils.setStatementValues(stmt, stmtValues, 1);

            try (final ResultSet resultSet = stmt.executeQuery()) {
                if ((null != resultSet) && resultSet.first()) {
                    return resultSet.getLong(1);
                }
            } finally {
                // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
                con.commit();
            }
        }

        return 0;
    }

    /**
     * @param whereProperties
     * @return
     * @throws FileItemException
     */
    private boolean implExecuteContains(@NonNull final String groupId, @NonNull final String whereStr) throws FileItemException {
        boolean groupContained = false;

        synchronized (m_groupMap) {
            if (m_groupMap.containsKey(groupId)) {
                groupContained = true;
            }
        }

        if (groupContained) {
            try (final Connection con = getReadConnection()) {
                final StringBuilder existsBuilder = STR_BUILDER("SELECT EXISTS (SELECT * FROM ").append(TBL_CONTENT(groupId, "fc")).
                    append(" WHERE ").append(whereStr).
                    append(" LIMIT 1)");

                try (final PreparedStatement existsStmt = con.prepareStatement(existsBuilder.toString());
                     final ResultSet resultSet = existsStmt.executeQuery()) {

                    return ((null != resultSet) && resultSet.first()) ?
                        resultSet.getBoolean(1) :
                            false;
                } finally {
                    // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
                    con.commit();
                }
            } catch (SQLException e) {
                throw new FileItemException("Error while deleting entry(ies))", e);
            }
        }

        return false;
    }

    /**
     * @param whereProperties
     * @return
     * @throws FileItemException
     */
    private FileStoreData[] implExecuteDelete(@NonNull final String groupId) throws FileItemException {
        try (final Connection con = getWriteConnection()) {
            final String selectSql = STR_BUILDER("SELECT fc.FileStoreNumber, fc.FileStoreId FROM ").append(TBL_CONTENT(groupId, "fc")).
                append(" INNER JOIN ").append(TBL_PROPERTIES(groupId, "fp")).
                append(" ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId").toString();

            final String deleteContentSql = STR_BUILDER("DELETE FROM ").append(TBL_CONTENT(groupId, null)).toString();
            final String deleteProperiesSql = STR_BUILDER("DELETE FROM ").append(TBL_PROPERTIES(groupId, null)).toString();

            try (final PreparedStatement selectStmt = con.prepareStatement(selectSql);
                 final PreparedStatement deleteContentStmt = con.prepareStatement(deleteContentSql);
                 final PreparedStatement deletePropertiesStmt = con.prepareStatement(deleteProperiesSql)) {

                return implExecuteQueryAndUpdate(con, selectStmt, deleteContentStmt, deletePropertiesStmt);
            }
        } catch (SQLException e) {
            throw new FileItemException("Error while deleting entry(ies))", e);
        }
    }

    /**
     * @param whereProperties
     * @return
     * @throws FileItemException
     */
    private FileStoreData[] implExecuteDelete(@NonNull final String groupId, @NonNull final String whereStr) throws FileItemException {
        try (final Connection con = getWriteConnection()) {
            final String selectStr = STR_BUILDER("SELECT fc.FileStoreNumber, fc.FileStoreId FROM ").append(TBL_CONTENT(groupId, "fc")).
                append(" INNER JOIN ").append(TBL_PROPERTIES(groupId, "fp")).
                append(" ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId WHERE ").toString();

            final String deleteStr = STR_BUILDER("DELETE fc, fp FROM ").append(TBL_CONTENT(groupId, "fc")).
                append(" INNER JOIN ").append(TBL_PROPERTIES(groupId, "fp")).
                append(" ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId WHERE ").toString();

            try (final PreparedStatement selectStmt = con.prepareStatement(selectStr + whereStr);
                 final PreparedStatement deleteStmt = con.prepareStatement(deleteStr + whereStr)) {

                return implExecuteQueryAndUpdate(con, selectStmt, deleteStmt);
            }
        } catch (SQLException e) {
            throw new FileItemException("Error while deleting entry(ies))", e);
        }
    }

    /**
     * @param whereProperties
     * @return
     * @throws FileItemException
     */
    private FileStoreData[] implExecuteDelete(@NonNull final String groupId, @NonNull final Properties whereProperties) throws FileItemException {
        try (final Connection con = getWriteConnection()) {
            final String selectStr = STR_BUILDER("SELECT fc.FileStoreNumber, fc.FileStoreId FROM ").append(TBL_CONTENT(groupId, "fc")).
                append(" INNER JOIN ").append(TBL_PROPERTIES(groupId, "fp")).
                append(" ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId WHERE ").toString();

            final String deleteStr = STR_BUILDER("DELETE fc, fp FROM ").append(TBL_CONTENT(groupId, "fc")).
                append(" INNER JOIN ").append(TBL_PROPERTIES(groupId, "fp")).
                append(" ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId WHERE ").toString();

            final String sqlSelect = FileItemUtils.createSqlWithPropertyVariables(selectStr, whereProperties, " AND ");
            final String sqlDelete = FileItemUtils.createSqlWithPropertyVariables(deleteStr, whereProperties, " AND ");

            try (final PreparedStatement selectStmt = con.prepareStatement(sqlSelect);
                 final PreparedStatement deleteStmt = con.prepareStatement(sqlDelete)) {

                final Object[] propertyValues = whereProperties.values().toArray();

                FileItemUtils.setStatementValues(selectStmt, propertyValues, 1);
                FileItemUtils.setStatementValues(deleteStmt, propertyValues, 1);

                return implExecuteQueryAndUpdate(con, selectStmt, deleteStmt);
            }
        } catch (SQLException e) {
            throw new FileItemException("Error while deleting entry(ies))", e);
        }
    }

    /**
     * @param con
     * @param selectStmt
     * @param deleteStmt
     * @return
     * @throws SQLException
     * @throws FileItemException
     */
    private static FileStoreData[] implExecuteQueryAndUpdate(@NonNull final Connection con, @NonNull final PreparedStatement selectStmt, @NonNull final PreparedStatement... deleteStmts) throws SQLException, FileItemException {
        selectStmt.executeQuery();

        if (null != deleteStmts) {
            for (final PreparedStatement curDeleteStmt : deleteStmts) {
                curDeleteStmt.executeUpdate();
            }
        }

        try (final ResultSet resultSet = selectStmt.getResultSet()) {
            final ArrayList<FileStoreData> fileStoreDataList = new ArrayList<>();

            if (null != resultSet) {
                while (resultSet.next()) {
                    fileStoreDataList.add(new FileStoreData(resultSet.getInt(1), resultSet.getString(2)));
                }
            }

            return FileItemUtils.listToArray(fileStoreDataList, FileStoreData.class);
        } catch (Exception e) {
            throw new FileItemException(e);
        } finally {
            // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
            con.commit();
        }
    }

    /**
     * @param con
     * @param selectStmt
     * @return
     */
    private static FileItem[] implExecuteQuery(@NonNull final Connection con, @NonNull final String groupId, @NonNull final PreparedStatement queryStmt, Object... returnValues) throws Exception {
        try (final ResultSet resultSet = queryStmt.executeQuery()) {
            final ArrayList<FileItem> fileItemList = new ArrayList<>();

            if (null != resultSet) {
                while (resultSet.next()) {
                    final int fileStoreNumber = resultSet.getInt(1);
                    final String fileStoreId = resultSet.getString(2);
                    final String subGroupId = resultSet.getString(3);
                    final String fileId = resultSet.getString(4);

                    final FileItem fileItem = new FileItem(groupId, subGroupId, fileId);

                    fileItem.setFileStoreData(new FileStoreData(fileStoreNumber, fileStoreId));
                    fileItemList.add(fileItem);

                    if (ArrayUtils.isNotEmpty(returnValues)) {
                        for (int i = 0, count = returnValues.length; i < count; ++i) {
                            final Object curReturnObj = returnValues[i];

                            if (curReturnObj instanceof Collection<?>) {
                                @SuppressWarnings("unchecked") final Collection<Object> curReturnCollection = (Collection<Object>) curReturnObj;

                                switch (i) {
                                    case 0: curReturnCollection.add(Integer.valueOf(fileStoreNumber)); break;
                                    case 1: curReturnCollection.add(fileStoreId); break;
                                    case 2: curReturnCollection.add(groupId); break;
                                    case 3: curReturnCollection.add(subGroupId); break;
                                    case 4: curReturnCollection.add(fileId); break;

                                    default: {
                                        // GroupId is not part of the SQL query itself,
                                        // so that the index of the return paramater equals
                                        // the current return array position instead
                                        // of (position + 1)
                                        curReturnCollection.add(resultSet.getObject(i));
                                        break;
                                    }
                                }
                            }
                        }
                    }
                }
            }

            return FileItemUtils.listToArray(fileItemList, FileItem.class);
        } catch (Exception e) {
            throw new FileItemException(e);
        } finally {
            // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
            con.commit();
        }
    }


    /**
     * @throws FileItemException
     */
    private void implCreateConnectionPools() throws FileItemException {
        if (null != m_configService) {
            final String readDriverClass = m_configService.getProperty("com.openexchange.fileItem.readDriverClass");
            final String writeDriverClass = m_configService.getProperty("com.openexchange.fileItem.writeDriverClass");
            final String jdbcReadUrl = m_configService.getProperty("com.openexchange.fileItem.readUrl");
            final String jdbcWriteUrl = m_configService.getProperty("com.openexchange.fileItem.writeUrl");

            try {
                Class.forName(readDriverClass);
                Class.forName(writeDriverClass);
            } catch (ClassNotFoundException e) {
                throw new FileItemException(e);
            }

            if (isEmpty(jdbcReadUrl) || isEmpty(jdbcWriteUrl)) {
                throw new FileItemException("Property readUrl and writeUrl must not be empty (Example Url: jdbc:mysql://host:port/databasename)");
            }

            try {
                final int poolMaxSize = m_configService.getIntProperty("com.openexchange.fileItem.connectionpool.maxPoolSize", HikariSetup.DEFAULT_POOL_MAXSIZE);
                final int poolConnectTimeoutMillis = m_configService.getIntProperty("com.openexchange.fileItem.connectionpool.connectTimeout", (int) HikariSetup.DEFAULT_POOL_CONNECTTIMEOUTMILLIS);
                final int poolIdleTimeoutMillis = m_configService.getIntProperty("com.openexchange.fileItem.connectionpool.idleTimeout", (int) HikariSetup.DEFAULT_POOL_IDLETIMEOUTMILLIS);
                Properties dbProperties = null;

                // DB read properties and DataSource
                implFillDBConnectionPropertiesFromConfigItems(dbProperties = new Properties(), m_configService.getProperties(new WildcardNamePropertyFilter("com.openexchange.fileItem.readProperty.*")));

                m_readConnectionPool = new HikariSetup(readDriverClass, jdbcReadUrl,
                    dbProperties.getProperty("user"), dbProperties.getProperty("password"),
                    dbProperties,
                    poolMaxSize, poolConnectTimeoutMillis, poolIdleTimeoutMillis).getConnectionPool(true);

                // DB write properties and DataSource
                implFillDBConnectionPropertiesFromConfigItems(dbProperties = new Properties(), m_configService.getProperties(new WildcardNamePropertyFilter("com.openexchange.fileItem.writeProperty.*")));

                m_writeConnectionPool = new HikariSetup(writeDriverClass, jdbcWriteUrl,
                    dbProperties.getProperty("user"), dbProperties.getProperty("password"),
                    dbProperties,
                    poolMaxSize, poolConnectTimeoutMillis, poolIdleTimeoutMillis).getConnectionPool(false);
            } catch (OXException e) {
                throw new FileItemException(e);
            }

            // determine database type
            if ((StringUtils.containsIgnoreCase(jdbcReadUrl, "mysql") &&
                 StringUtils.containsIgnoreCase(jdbcWriteUrl, "mysql")) ||
                (StringUtils.containsIgnoreCase(jdbcReadUrl, "maria") &&
                    StringUtils.containsIgnoreCase(jdbcWriteUrl, "maria"))) {

                m_databaseType = IFileItemService.DatabaseType.MYSQL;
            }
        }
    }

    /**
     * @throws FileItemException
     * @throws LiquibaseException
     */
    private void implUpdateDatabase() throws FileItemException {
        try (final Connection con = getWriteConnection()) {
            final Liquibase liquibase = new Liquibase("fileItemChangeLog.xml", new FileItemDatabase.ImplLiquibaseResourceAccessor(), new JdbcConnection(con));

            liquibase.setChangeExecListener(this);
            liquibase.update(null);
        } catch (Exception e) {
            throw new FileItemException(e);
        }
    }

    /**
     * @param fileItem
     * @return
     * @throws FileItemException
     */
    private void implReadGroups() throws FileItemException {
        final String selectSql = STR_BUILDER("SELECT fg.GroupId, fg.PropertyKeys, fg.SubGroupCount FROM ").append(TBL_GROUPADMIN("fg")).toString();

        try (final Connection con = getReadConnection();
             final PreparedStatement stmt = con.prepareStatement(selectSql)) {

            try (final ResultSet resultSet = stmt.executeQuery()) {
                if (null != resultSet) {
                    while (resultSet.next()) {
                        final String curGroupId = resultSet.getString(1);

                        if (FileItemUtils.isValid(curGroupId)) {
                            final Set<String> propertyKeySet = implGetKeySet(curGroupId, false);
                            final Set<String> customKeySet = implGetKeySet(curGroupId, true);
                            final String curPropertyKeys = resultSet.getString(2);
                            final long subGroupCount = resultSet.getLong(3);

                            // add to set for given groupId, if not already present
                            if (isNotEmpty(curPropertyKeys)) {
                                final String[] keys  = new String(curPropertyKeys).split(",", 0);

                                if (null != keys) {
                                    for (final String curKey : keys) {
                                        if (isNotBlank(curKey)) {
                                            final String targetKey = curKey.trim();

                                            propertyKeySet.add(targetKey);

                                            if (!implIsDefaultKey(curKey)) {
                                                customKeySet.add(targetKey);
                                            }
                                        }
                                    }
                                }
                            }

                            // update GroupCount
                            implSetSubGroupCount(curGroupId, subGroupCount);
                        }
                    }
                }
            } finally {
                // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
                con.commit();
            }
        } catch (SQLException e) {
            throw new FileItemException(e);
        }
    }

    /**
     * @param status
     * @param description
     */
    private static void implLogChangeEvent(final Object status, final String description) {
        if (FileItemUtils.isLogInfo()) {
            if (null != status) {
                FileItemUtils.logInfo("FileItemService database update received status: " + status.toString());
            }

            if (isNotEmpty(description)) {
                FileItemUtils.logInfo("FileItemService database update description: " + description);
            }
        }
    }

    /**
     * @param dbProperties
     * @param configItems
     */
    private static void implFillDBConnectionPropertiesFromConfigItems(@NonNull final Properties dbProperties, @NonNull Map<String, String> configItems) {
        for (final String curKey : configItems.keySet()) {
            final String dbConfigPair = configItems.get(curKey);

            if (isNotEmpty(dbConfigPair)) {
                final int assignPos = dbConfigPair.indexOf('=');

                if ((assignPos > 0) && (assignPos < (dbConfigPair.length() - 1))) {
                    dbProperties.put(dbConfigPair.substring(0, assignPos), dbConfigPair.substring(assignPos + 1));
                }
            }
        }
    }

    /**
     * @param groupId
     * @return
     */
    private boolean implCreateGroupTables(@NonNull final String groupId) throws FileItemException {
        try (final Connection con = getWriteConnection()) {
            // TODO (KA): create tables and indices

            // create Content_GroupId table
            final String createContentTableSql = STR_BUILDER("CREATE TABLE ").append(TBL_CONTENT(groupId, null)).
                append(" (").
                append(" FileStoreNumber INT(10) NOT NULL,").
                append(" FileStoreId ").append(DB_VARCHAR_TYPE).append(" NOT NULL,").
                append(" SubGroupId ").append(DB_VARCHAR_TYPE).append(" NOT NULL,").
                append(" FileId ").append(DB_VARCHAR_TYPE).append(" NOT NULL").
                append(" )").toString();

            final String alterContentTableSql = STR_BUILDER("ALTER TABLE ").append(TBL_CONTENT(groupId, null)).
                append(" ADD PRIMARY KEY (SubGroupId, FileId)").toString();

            final String createSecondaryContentIndexSql = STR_BUILDER("CREATE INDEX ").append("idx_FileStore").
                append(" ON ").append(TBL_CONTENT(groupId, " ")).
                append(" (").
                append(" FileStoreNumber,").
                append(" FileStoreId (128)").
                append(" )").toString();


            try (final PreparedStatement createStmt = con.prepareStatement(createContentTableSql);
                 final PreparedStatement alterStmt = con.prepareStatement(alterContentTableSql);
                 final PreparedStatement createSecondaryIndexStmt = con.prepareStatement(createSecondaryContentIndexSql)) {

                createStmt.executeUpdate();
                alterStmt.executeUpdate();
                createSecondaryIndexStmt.executeUpdate();
            } catch (@SuppressWarnings("unused") SQLException e) {
                // ignore, errors will be handled in outer block
            }

            // create Properties_GroupId table
            final String createPropertiesTableSql = STR_BUILDER("CREATE TABLE ").append(TBL_PROPERTIES(groupId, null)).
                append(" (").
                append(" FileStoreNumber INT(10) NOT NULL,").
                append(" FileStoreId ").append(DB_VARCHAR_TYPE).append(" NOT NULL").
                append(" )").toString();

            final String alterPropertiesTableSql = STR_BUILDER("ALTER TABLE ").append(TBL_PROPERTIES(groupId, null)).
                append(" ADD PRIMARY KEY (FileStoreNumber, FileStoreId)").toString();

            try (final PreparedStatement createStmt = con.prepareStatement(createPropertiesTableSql);
                 final PreparedStatement alterStmt = con.prepareStatement(alterPropertiesTableSql)) {

                createStmt.executeUpdate();
                alterStmt.executeUpdate();
            } catch (@SuppressWarnings("unused") SQLException e) {
                // ignore, errors will be handled in outer block
            }

            // insert new group into Group table
            final String insertGroupSql = STR_BUILDER("INSERT INTO ").append(TBL_GROUPADMIN(null)).
                append(" (GroupId,PropertyKeys,SubGroupCount) VALUES (?,?,?)").toString();

            try (final PreparedStatement insertStmt = con.prepareStatement(insertGroupSql)) {
                insertStmt.setString(1, groupId);
                insertStmt.setString(2, "");
                insertStmt.setLong(3, 0);

                insertStmt.executeUpdate();
            } catch (@SuppressWarnings("unused") SQLException e) {
                // ignore, errors will be handled in outer block
            }

            con.commit();

            return true;
        } catch (Exception e) {
            throw new FileItemException(e);
        }

    }

    /**
     * @param groupId
     * @param key
     * @param dbType
     * @return
     * @throws FileItemException
     */
    private void implInsertPropertyKey(@NonNull final String groupId, @NonNull final String key, @NonNull final String dbType, @Nullable final String dbColumnConstraints) throws FileItemException {
        final Set<String> propertyKeySet = implGetKeySet(groupId, false);

        propertyKeySet.add(key);

        final String updateGroupPropertyKeysSql = STR_BUILDER("UPDATE ").append(TBL_GROUPADMIN("fg")).
                append(" SET fg.PropertyKeys=?").
                append(" WHERE fg.GroupId=?").toString();


        // create update string value for set of property keys
        final StringBuilder propertyKeysBuilder = STR_BUILDER("");

        for (final String curKey : propertyKeySet) {
            if (isNotEmpty(curKey)) {
                if (propertyKeysBuilder.length() > 0) {
                    propertyKeysBuilder.append(',');
                }

                propertyKeysBuilder.append(curKey);
            }
        }

        try (final Connection con = getWriteConnection();
             final PreparedStatement fgPropertyKeysUpdateStmt = con.prepareStatement(updateGroupPropertyKeysSql)) {

            // delete/insert groupId in Groups table
            fgPropertyKeysUpdateStmt.setString(1, propertyKeysBuilder.toString());
            fgPropertyKeysUpdateStmt.setString(2, groupId);

            // execute all statements and commit
            fgPropertyKeysUpdateStmt.executeUpdate();

            con.commit();
        } catch (Exception e) {
            if (FileItemUtils.isLogTrace()) {
                FileItemUtils.logTrace(new StringBuilder(128).
                    append("Exception occured when trying to update PropertyKeys column for GroupId: ").append(groupId).
                    append(" (").append(Throwables.getRootCause(e).getMessage()).append(')').toString());
            }
        }

        // Add new customKey column to Properties table
        final StringBuilder insertPropertiesColSql = STR_BUILDER("ALTER TABLE ").append(TBL_PROPERTIES(groupId, null)).
            append(" ADD COLUMN ").append(key).append(' ').append(dbType);

        if (null != dbColumnConstraints) {
            insertPropertiesColSql.append(' ').append(dbColumnConstraints);
        }

        try (final Connection con = getWriteConnection();
             final PreparedStatement fpInsertColStmt = con.prepareStatement(insertPropertiesColSql.toString())) {

            fpInsertColStmt.executeUpdate();
            con.commit();
        } catch (Exception e) {
            if (FileItemUtils.isLogTrace()) {
                FileItemUtils.logTrace(new StringBuilder(128).
                    append("Exception occured when trying to add new column for GroupId: ").append(groupId).
                    append(" (").append(Throwables.getRootCause(e).getMessage()).append(')').toString());
            }
        }

        // create index for new column
        final String indexName = "idx_" + key;
        final String createPropertiesColIndexSql = STR_BUILDER("CREATE INDEX ").append(indexName).
            append(" ON ").append(TBL_PROPERTIES(groupId, " ")).
            append(" (").
            append(key).append(dbType.equalsIgnoreCase("text") ? " (128)" : "").
            append(" )").toString();

        try (final Connection con = getWriteConnection();
             final PreparedStatement fpCreateIndexStmt = con.prepareStatement(createPropertiesColIndexSql)) {

            fpCreateIndexStmt.executeUpdate();
            con.commit();
        } catch (Exception e) {
            if (FileItemUtils.isLogTrace()) {
                FileItemUtils.logTrace(new StringBuilder(128).
                    append("Exception occured when trying to create new index for GroupId: ").append(groupId).
                    append(" (").append(Throwables.getRootCause(e).getMessage()).append(')').toString());
            }
        }
    }

    /**
     * @param groupId
     * @return
     */
    private @NonNull Set<String> implGetKeySet(@NonNull final String groupId, final boolean customKeysOnly) {
        implEnsureValidGroup(groupId);
        return (customKeysOnly ? m_customKeysMap : m_groupMap).get(groupId);
    }

    /**
     * @param groupId
     * @return
     */
    private long implGetSubGroupCount(@NonNull final String groupId) {
        implEnsureValidGroup(groupId);
        return m_subGroupCountMap.get(groupId).get();
    }

    /**
     * @param groupId
     * @param subGroupCount
     */
    private void implSetSubGroupCount(@NonNull final String groupId, long subGroupCount) {
        implEnsureValidGroup(groupId);
        m_subGroupCountMap.get(groupId).set(subGroupCount);
    }

    /**
     * @param groupId
     * @return
     */
    private long implIncrementSubGroupCount(@NonNull final String groupId) {
        implEnsureValidGroup(groupId);
        return m_subGroupCountMap.get(groupId).incrementAndGet();
    }

    /**
     * @param groupId
     * @return
     */
    private long implDecrementSubGroupCount(@NonNull final String groupId) {
        implEnsureValidGroup(groupId);
        return m_subGroupCountMap.get(groupId).decrementAndGet();
    }

    /**
     * @param groupId
     */
    private void implEnsureValidGroup(@NonNull final String groupId) {
        if (!m_groupMap.containsKey(groupId)) {
            m_groupMap.put(groupId, new LinkedHashSet<>());
            m_customKeysMap.put(groupId, new LinkedHashSet<>());
            m_subGroupCountMap.put(groupId, new AtomicLong(0));
        }
    }

    /**
     * @param groupId
     * @param tableAlias
     * @return
     */
    final static StringBuilder TBL_GROUPADMIN(@Nullable String tableAlias) {
        final StringBuilder groupAdminTableNameBuilder = new StringBuilder(32).append("GroupAdmin");

        if (null != tableAlias) {
            groupAdminTableNameBuilder.append(' ').append(tableAlias);
        }

        return groupAdminTableNameBuilder;
    }

    /**
     * @param groupId
     * @param tableAlias
     * @return
     */
    final static StringBuilder TBL_CONTENT(@NonNull final String groupId, @Nullable String tableAlias) {
        final StringBuilder contentTableNameBuilder = new StringBuilder(64).append("Content").append('_').append(groupId);

        if (null != tableAlias) {
            contentTableNameBuilder.append(' ').append(tableAlias);
        }

        return contentTableNameBuilder;
    }

    /**
     * @param groupId
     * @param tableAlias
     * @return
     */
    final private static StringBuilder TBL_PROPERTIES(@NonNull final String groupId, @Nullable String tableAlias) {
        final StringBuilder propertiesTableNameBuilder = new StringBuilder(64).append("Properties").append('_').append(groupId);

        if (null != tableAlias) {
            propertiesTableNameBuilder.append(' ').append(tableAlias);
        }

        return propertiesTableNameBuilder;
    }

    /**
     * @param initStr
     * @return
     */
    final private static StringBuilder STR_BUILDER(final String initStr) {
        final StringBuilder ret = new StringBuilder(STR_BUILDER_DEFAULT_CAPACITY);

        return ((null != initStr) ? ret.append(initStr) : ret);
    }

    /**
     * {@link ImplLiquibaseResourceAccessor}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.10.0
     */
    protected class ImplLiquibaseResourceAccessor implements ResourceAccessor {

        /**
         * Initializes a new {@link ImplLiquibaseResourceAccessor}.
         */
        protected ImplLiquibaseResourceAccessor() {}

        /*
         * (non-Javadoc)
         *
         * @see liquibase.resource.ResourceAccessor#getResourceAsStream(java.lang.String)
         */
        @Override
        public InputStream getResourceAsStream(String file) throws IOException {
            ClassLoader loader = toClassLoader();

            final URL url = ((null != loader) ? loader : FileItemDatabase.class.getClassLoader()).getResource(file);

            if (null != url) {
                try {
                    final URLConnection con = url.openConnection();

                    if (null != con) {
                        con.connect();

                        try (final InputStream resourceInputStm = con.getInputStream()) {
                            if (null != resourceInputStm) {
                                return new ByteArrayInputStream(org.apache.commons.io.IOUtils.toByteArray(resourceInputStm));
                            }
                        }
                    }
                } catch (Exception e) {
                    FileItemUtils.logExcp(e);
                }
            }

            return null;
        }

        /*
         * (non-Javadoc)
         *
         * @see liquibase.resource.ResourceAccessor#getResources(java.lang.String)
         */
        @Override
        public Enumeration<URL> getResources(String packageName) throws IOException {
            return Collections.emptyEnumeration();
        }

        /*
         * (non-Javadoc)
         *
         * @see liquibase.resource.ResourceAccessor#toClassLoader()
         */
        @Override
        public ClassLoader toClassLoader() {
            return Thread.currentThread().getContextClassLoader();
        }
    }

    // - Static members --------------------------------------------------------

    final private static int STR_BUILDER_DEFAULT_CAPACITY = 128;

    // - Members ---------------------------------------------------------------

    private HikariPool m_readConnectionPool = null;

    private HikariPool m_writeConnectionPool = null;

    private ConfigurationService m_configService = null;

    private Map<String, Set<String>> m_groupMap = new HashMap<>();

    private Map<String, Set<String>> m_customKeysMap = new HashMap<>();

    private Map<String, AtomicLong> m_subGroupCountMap = new HashMap<>();

    private IFileItemService.DatabaseType m_databaseType = IFileItemService.DatabaseType.STANDARD_SQL;
}
