/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.fileitem.impl;

import com.openexchange.annotation.NonNull;
import com.openexchange.imageconverter.api.IFileItem;

/**
 * {@link FileItem}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
/**
 * {@link FileItem}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public class FileItem implements IFileItem {

    /**
     * serialVersionUID
     */
    private static final long serialVersionUID = 1203862060096934396L;

    /**
     * Initializes a new {@link FileItem}.
     */
    @SuppressWarnings("unused")
    private FileItem() {
        // no empty Ctor
    }

    /**
     * Initializes a new {@link FileItem}.
     * @param groupId
     * @param subGroupId
     * @param fileId
     */
    public FileItem(@NonNull String groupId, @NonNull String subGroupId, @NonNull String fileId) {
        super();

        m_groupId = FileItemUtils.isValid(groupId) ?  groupId : FileItemService.GROUP_ID_DEFAULT;
        m_subGroupId = FileItemUtils.isValid(subGroupId) ? subGroupId : FileItemService.SUBGROUP_ID_DEFAULT;
        m_fileId = fileId;

        if (FileItemUtils.isInvalid(m_fileId) && FileItemUtils.isLogError()) {
            FileItemUtils.logError("Invalid fileId is not allowed! Set fileId to unique and not empty value.");
        }
    }

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return new StringBuilder("FileItem").
            append(" [").
            append((null != m_fileStoreData) ? m_fileStoreData.toString() : "FileStoreData [null]").
            append(" , groupId: ").append(m_groupId).
            append(" , subGroupId: ").append(m_subGroupId).
            append(" , fileId: ").append(m_fileId).
            append(']').toString();
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItem#getGroupId()
     */
    @Override
    public String getGroupId() {
        return m_groupId;
    }

    /**
     * Gets the subGroupId
     *
     * @return The m_subGroupId
     */
    @Override
    public String getSubGroupId() {
        return m_subGroupId;
    }

    /**
     * Gets the fileId
     *
     * @return The m_fileId
     */
    @Override
    public String getFileId() {
        return m_fileId;
    }

    // - Package internal API  -------------------------------------------------

    /**
     * @return
     */
    protected String getKey() {
        return new StringBuilder(m_groupId.length() + m_subGroupId.length() + m_fileId.length() + 2).
            append(m_groupId).append('_').
            append(m_subGroupId).append('_').
            append(m_fileId).toString();
    }

    /**
     * @return
     */
    protected boolean isValid() {
        return FileItemUtils.isValid(m_groupId) && FileItemUtils.isValid(m_subGroupId) && FileItemUtils.isValid(m_fileId);
    }

    /**
     * @return
     */
    protected boolean isInvalid() {
        return FileItemUtils.isInvalid(m_groupId) || FileItemUtils.isInvalid(m_subGroupId) || FileItemUtils.isInvalid(m_fileId);
    }

    /**
     * Gets the storeId
     *
     * @return The m_storeId
     */
    protected FileStoreData getFileStoreData() {
        return m_fileStoreData;
    }

    /**
     * Gets the storeId
     *
     * @return The m_storeId
     */
    protected void setFileStoreData(final FileStoreData fileStoreData) {
        m_fileStoreData = fileStoreData;
    }

    // - Members ---------------------------------------------------------------

    protected String m_groupId = null;

    protected String m_subGroupId = null;

    protected String m_fileId = null;

    protected FileStoreData m_fileStoreData = null;
}
