/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.imageconverter.client.osgi;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Throwables;
import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.OXException;
import com.openexchange.imageconverter.api.IImageClient;
import com.openexchange.imageconverter.api.IMetadataReader;
import com.openexchange.imageconverter.client.ImageConverterClient;
import com.openexchange.imageconverter.client.Services;
import com.openexchange.osgi.HousekeepingActivator;

/**
 * {@link ImageConverterClientActivator}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public class ImageConverterClientActivator extends HousekeepingActivator {

    /**
     * Initializes a new {@link ImageConverterClientActivator}.
     */
    public ImageConverterClientActivator() {
        super();
    }

    //-------------------------------------------------------------------------

    /* (non-Javadoc)
     * @see com.openexchange.osgi.DeferredActivator#getNeededServices()
     */
    @Override
    protected Class<?>[] getNeededServices() {
        return new Class<?>[] {
            ConfigurationService.class
        };
    }

    @Override
    protected Class<?>[] getOptionalServices() {
        return new Class<?>[] {
            IMetadataReader.class
        };
    }

    //-------------------------------------------------------------------------

    /* (non-Javadoc)
     * @see com.openexchange.osgi.DeferredActivator#startBundle()
     */
    @Override
    public void startBundle() throws Exception {
        LOG.info("starting bundle: " + SERVICE_NAME);

        Services.setServiceLookup(this);

        try {
            final ConfigurationService configService = getService(ConfigurationService.class);

            if (null == configService) {
                throw new OXException(new Throwable("Configuration service not available"));
            }

            m_imageConverterClient = new ImageConverterClient(configService);
            registerService(IImageClient.class, m_imageConverterClient);

            openTrackers();

            LOG.info("Successfully started bundle: " + SERVICE_NAME);
        } catch (Throwable e) {
            LOG.error("... starting bundle: " + SERVICE_NAME + " failed", Throwables.getRootCause(e));
            throw new RuntimeException(e);
        }
    }

    //-------------------------------------------------------------------------

    /* (non-Javadoc)
     * @see com.openexchange.osgi.HousekeepingActivator#stopBundle()
     */
    @Override
    public void stopBundle() throws Exception {
        LOG.info("stopping bundle: " + SERVICE_NAME);

        closeTrackers();
        unregisterServices();
        Services.setServiceLookup(null);

        IOUtils.closeQuietly(m_imageConverterClient);
        m_imageConverterClient = null;

        LOG.info("successfully stopped bundle: " + SERVICE_NAME);
    }

    // - Members ---------------------------------------------------------------

    protected ImageConverterClient m_imageConverterClient = null;

    // - Static Members --------------------------------------------------------

    final private static String SERVICE_NAME = "Open-Xchange ImageConverter Client";

    final private static Logger LOG = LoggerFactory.getLogger(ImageConverterClientActivator.class);
}
