/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.imageconverter.client;

import static org.apache.commons.lang.ArrayUtils.isNotEmpty;
import static org.apache.commons.lang.StringUtils.isNotEmpty;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import javax.ws.rs.core.Response.Status;
import javax.xml.ws.WebServiceException;
import org.apache.commons.fileupload.MultipartStream;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Throwables;
import com.openexchange.annotation.NonNull;
import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.OXException;
import com.openexchange.imageconverter.api.IImageClient;
import com.openexchange.imageconverter.api.IMetadata;
import com.openexchange.imageconverter.api.IMetadataReader;
import com.openexchange.imageconverter.api.ImageConverterException;
import com.openexchange.imageconverter.api.MetadataImage;
import com.openexchange.imageconverter.client.generated.invoker.ApiException;
import com.openexchange.imageconverter.client.generated.modules.ImageConverterApi;

/**
 * {@link ImageConverterClient}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public class ImageConverterClient implements IImageClient {

    /**
     * Private Ctor: not to be used!
     * Initializes a new {@link ImageConverterClient}.
     */
    @SuppressWarnings("unused")
    private ImageConverterClient() {
        super();

        m_clientConfig = null;
        m_remoteValidator = null;
        m_apiProvider = null;
    }

    /**
     * Initializes a new {@link ImageConverterClient}.
     * @param imageConverterClientConfig
     * @throws OXException
     */
    public ImageConverterClient(@NonNull final ConfigurationService configurationService) throws OXException {
        super();

        m_clientConfig = new ImageConverterClientConfig(configurationService);

        if (!m_clientConfig.isValid()) {
        	throw OXException.general("Configuration service is available, but config property is not valid");
        }

        m_apiProvider = new ImageConverterApiProvider(m_clientConfig);
        m_remoteValidator = new ImageConverterRemoteValidator(m_clientConfig, m_apiProvider);
    }

    /* (non-Javadoc)
     * @see java.io.Closeable#close()
     */
    @Override
    public void close() {
        m_clientConfig.shutdown();
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IImageConverter#getImage(java.lang.String, java.lang.String, java.lang.String)
     */
    @Override
    public InputStream getImage(final String imageKey, final String requestFormat, final String... context) throws ImageConverterException {
        final ImageConverterApi api = m_apiProvider.acquire();
        InputStream ret = null;
        
        LOG.trace("IC client #getImage called (imageKey/requestFormat): {} / {}", imageKey, requestFormat);

        if (null != api) {
            try {
                if (isNotEmpty(imageKey) && isNotEmpty(requestFormat)) {
                    File tmpResponseFile = null;

                    try {
                        if (null != (tmpResponseFile = api.getImage(imageKey, requestFormat, implGetContext(context)))) {
                            ret = new ByteArrayInputStream(FileUtils.readFileToByteArray(tmpResponseFile));
                        }
                    } catch (ApiException e) {
                        implHandleAPIException("Error in remote calling ImageConverter#getImage", e);
                    } catch (Exception e) {
                        throw new ImageConverterException("Error in remote calling ImageConverter#getImage", e);
                    } finally {
                        FileUtils.deleteQuietly(tmpResponseFile);
                    }
                }
            } finally {
                m_apiProvider.release(api);
            }
        }

        return ret;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IImageConverter#getMetadata(java.lang.String, java.lang.String[])
     */
    @Override
    public IMetadata getMetadata(String imageKey, String... context) throws ImageConverterException {
        final ImageConverterApi api = m_apiProvider.acquire();
        IMetadata ret = null;

        LOG.trace("IC client #getMetadata called (imageKey): {}", imageKey);

        if (null != api) {
            try {
                if (isNotEmpty(imageKey)) {
                    File tmpResponseFile = null;

                    try {
                        if (null != (tmpResponseFile = api.getMetadata(imageKey, implGetContext(context)))) {
                            try (final InputStream fileInputStm = FileUtils.openInputStream(tmpResponseFile);
                                 final MetadataImage metadataImage = implCreateMetadataImage(api, fileInputStm)) {

                                if (null != metadataImage) {
                                    ret = metadataImage.getMetadata();
                                }
                            }
                        }
                    } catch (ApiException e) {
                        implHandleAPIException("Error in remote calling ImageConverter#getMetadata", e);
                    } catch (Exception e) {
                        throw new ImageConverterException("Error in remote calling ImageConverter#getMetadata", e);
                    } finally {
                        FileUtils.deleteQuietly(tmpResponseFile);
                    }
                }
            } finally {
                m_apiProvider.release(api);
            }
        }

        return ret;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IImageConverter#getImageAndMetadata(java.lang.String, java.lang.String, java.lang.String[])
     */
    @Override
    public MetadataImage getImageAndMetadata(String imageKey, String requestFormat, String... context) throws ImageConverterException {
        final ImageConverterApi api = m_apiProvider.acquire();
        MetadataImage ret = null;

        LOG.trace("IC client #getImageAndMetadata called (imageKey/requestFormat): {} / {}", imageKey, requestFormat);

        if (null != api) {
            try {
                if (isNotEmpty(imageKey) && isNotEmpty(requestFormat)) {
                    File tmpResponseFile = null;

                    try {
                        if (null != (tmpResponseFile = api.getImageAndMetadata(imageKey, requestFormat, implGetContext(context)))) {
                            try (final InputStream fileInputStm = FileUtils.openInputStream(tmpResponseFile)) {
                                ret = implCreateMetadataImage(api, fileInputStm);
                            }
                        }
                    } catch (ApiException e) {
                        implHandleAPIException("Error in remote calling ImageConverter#getImageAndMetadata", e);
                    } catch (Exception e) {
                        throw new ImageConverterException("Error in remote calling ImageConverter#getImageAndMetadata", e);
                    } finally {
                        FileUtils.deleteQuietly(tmpResponseFile);
                    }
                }
            } finally {
                m_apiProvider.release(api);
            }
        }

        return ret;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IImageConverter#cacheImage(java.lang.String, java.lang.String, java.io.InputStream)
     */
    @Override
    public void cacheImage(final String imageKey, final InputStream srcImageStm, final String... context) throws ImageConverterException {
        final ImageConverterApi api = m_apiProvider.acquire();

        LOG.trace("IC client #cacheImage called (imageKey): {}", imageKey);

        if (null != api) {
            try {
                if (isNotEmpty(imageKey) && (null != srcImageStm)) {
                    final File tmpInputFile = implCreateTempFile(srcImageStm);

                    if (null != tmpInputFile) {
                        try {
                            api.cacheImage(imageKey, tmpInputFile, implGetContext(context));
                        } catch (ApiException e) {
                            implHandleAPIException("Error in remote calling ImageConverter#cacheImage", e);
                        } finally {
                            FileUtils.deleteQuietly(tmpInputFile);
                        }
                    } else if (LOG.isWarnEnabled()) {
                        LOG.warn("InputStream in call to ImageConverter#cacheImage is empty (imageKey): " + imageKey);
                    }
                }
            } finally {
                m_apiProvider.release(api);
            }
        }
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IImageConverter#cacheAndGetImage(java.lang.String, java.lang.String, java.lang.String, java.io.InputStream)
     */
    @Override
    public InputStream cacheAndGetImage(final String imageKey, final String requestFormat, final InputStream srcImageStm, final String... context) throws ImageConverterException {
        final ImageConverterApi api = m_apiProvider.acquire();
        InputStream ret = null;

        LOG.trace("IC client #cacheAndGetImage called (imageKey/requestFormat): {} / {}", imageKey, requestFormat);

        if (null != api) {
            try {
                if (isNotEmpty(imageKey) && isNotEmpty(requestFormat)) {
                    if (null != srcImageStm) {
                        final File tmpInputFile = implCreateTempFile(srcImageStm);

                        if (null != tmpInputFile) {
                            File tmpResponseFile = null;

                            try {
                                if (null != (tmpResponseFile = api.cacheAndGetImage(imageKey, requestFormat, tmpInputFile, implGetContext(context)))) {
                                    ret = new ByteArrayInputStream(FileUtils.readFileToByteArray(tmpResponseFile));
                                }
                            } catch (ApiException e) {
                                implHandleAPIException("Error in remote calling ImageConverter#cacheAndGetImage", e);
                            } catch (Exception e) {
                                throw new ImageConverterException("Error in remote calling ImageConverter#cacheAndGetImage", e);
                            } finally {
                                FileUtils.deleteQuietly(tmpInputFile);
                                FileUtils.deleteQuietly(tmpResponseFile);
                            }
                        } else if (LOG.isWarnEnabled()) {
                            LOG.warn("InputStream in call to ImageConverter#cacheAndGetImage is empty (imageKey): " + imageKey);
                        }
                    } else {
                        ret = getImage(imageKey, requestFormat, context);
                    }
                }
            } finally {
                m_apiProvider.release(api);
            }
        }

        return ret;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IImageConverter#cacheAndGetImageAndMetadata(java.lang.String, java.lang.String, java.io.InputStream, java.lang.String[])
     */
    @Override
    public MetadataImage cacheAndGetImageAndMetadata(String imageKey, String requestFormat, InputStream srcImageStm, String... context) throws ImageConverterException {
        final ImageConverterApi api = m_apiProvider.acquire();
        MetadataImage ret = null;

        LOG.trace("IC client #cacheAndGetImageAndMetadata called (imageKey/requestFormat): {} / {}", imageKey, requestFormat);

        if (null != api) {
            try {
                if (isNotEmpty(imageKey) && isNotEmpty(requestFormat)) {
                    if (null != srcImageStm) {
                        final File tmpInputFile = implCreateTempFile(srcImageStm);

                        if (null != tmpInputFile) {
                            File tmpResponseFile = null;

                            try {
                                if (null != (tmpResponseFile = api.cacheAndGetImageAndMetadata(imageKey, requestFormat, tmpInputFile, implGetContext(context)))) {
                                    try (final InputStream fileInputStm = FileUtils.openInputStream(tmpResponseFile)) {
                                        ret = implCreateMetadataImage(api, fileInputStm);
                                    }
                                }
                            } catch (ApiException e) {
                                implHandleAPIException("Error in remote calling ImageConverter#cacheAndGetImageAndMetadata", e);
                            } catch (Exception e) {
                                throw new ImageConverterException("Error in remote calling ImageConverter#cacheAndGetImageAndMetadata", e);
                            } finally {
                                FileUtils.deleteQuietly(tmpInputFile);
                                FileUtils.deleteQuietly(tmpResponseFile);
                            }
                        } else if (LOG.isWarnEnabled()) {
                            LOG.warn("InputStream in call to ImageConverter#cacheAndGetImageAndMetadata is empty (imageKey): " + imageKey);
                        }
                    }
                }
            } finally {
                m_apiProvider.release(api);
            }
        }

        return ret;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IImageConverter#clearImagesByContext(java.lang.String)
     */
    @Override
    public void clearImages(final String... context) throws ImageConverterException {
        final ImageConverterApi api = m_apiProvider.acquire();

        LOG.trace("IC client #clearImages called");

        if (null != api) {
            try {
                api.clearImages(implGetContext(context));
            } catch (ApiException e) {
                implHandleAPIException("Error in remote calling ImageConverter#clearImages with context: " + context, e);
            } finally {
                m_apiProvider.release(api);
            }
        }
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IImageConverter#clearImagesByKey(java.lang.String)
     */

    @Override
    public void clearImagesByKey(final String imageKey) throws ImageConverterException {
        final ImageConverterApi api = m_apiProvider.acquire();

        LOG.trace("IC client #clearImagesByKey called (imageKey): {}", imageKey);

        if (null != api) {
            try {
                api.clearImagesByKey(imageKey);
            } catch (ApiException e) {
                implHandleAPIException("Error in remote calling ImageConverter#clearImagesByKey", e);
            } finally {
                m_apiProvider.release(api);
            }
        }
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IImageConverter#getKeyCountByContext(java.lang.String)
     */
    @Override
    public long getKeyCount(final String... context) throws ImageConverterException {
        final ImageConverterApi api = m_apiProvider.acquire();
        long ret = 0;

        LOG.trace("IC client #getKeyCount called");

        if (null != api) {
            try {
                ret = api.getKeyCount(implGetContext(context)).getCount().longValue();
            } catch (ApiException e) {
                implHandleAPIException("Error in remote calling ImageConverter#getKeyCount with context: " + context, e);
            } finally {
                m_apiProvider.release(api);
            }
        }

        return ret;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IImageConverter#getKeys()
     */
    @Override
    public String[] getKeys(final String... context) throws ImageConverterException {
        final ImageConverterApi api = m_apiProvider.acquire();
        String [] ret = {};

        LOG.trace("IC client #getKeys called");

        if (null != api) {
            try {
                final List<String> keyList = api.getKeys(implGetContext(context)).getKeys();
                ret = keyList.toArray(new String[keyList.size()]);
            } catch (ApiException e) {
                implHandleAPIException("Error in remote calling ImageConverter#getKeys with context: " + context, e);
            } finally {
                m_apiProvider.release(api);
            }
        }

        return ret;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IImageConverter#getTotalImagesSize(java.lang.String[])
     */
    @Override
    public long getTotalImagesSize(String... context) throws ImageConverterException {
        // Dummy implementation, no HTTP Api method available and needed
        return 0;
    }

    /**
     * @return
     * @throws WebServiceException
     */
    @Override
    public String status() throws OXException {
        final ImageConverterApi api = m_apiProvider.acquire();
        String ret = null;

        LOG.trace("IC client #status called");

        if (null != api) {
            try {
                ret = api.status();
            } catch (ApiException e) {
                ImageConverterException ice = implWrapAPIException("Error in remote calling ImageConverter#status", e);
                if (ice != null) {
                	throw OXException.general(ice.getMessage(), ice);
                }
            } finally {
                m_apiProvider.release(api);
            }
        }

        return ret;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IImageClient#isConnected()
     */
    @Override
    public boolean isConnected() throws OXException {
        return m_remoteValidator.isConnected();
    }

    // - public interface ------------------------------------------------------

    /**
     * @return
     */
    public boolean isTest() {
        return m_clientConfig.isTest();
    }

    // - Implementation --------------------------------------------------------

    private void implHandleAPIException(final String msg, final ApiException e) throws ImageConverterException {
    	final ImageConverterException ice = implWrapAPIException(msg, e);
    	if (ice != null) {
    		throw ice;
    	}
    }
    
    private ImageConverterException implWrapAPIException(final String msg, final ApiException e) {
        final StringBuilder throwMessageBuilder = new StringBuilder(StringUtils.isNotEmpty(msg) ? msg : "IC client received unknown API exception");
        boolean throwException = true;

        if (null != e) {
            final Throwable cause = Throwables.getRootCause(e);
            final String causeMessage = (null != cause) ? cause.getMessage() : null;
            final int  statusCode = e.getCode();

            if (Status.NOT_FOUND.getStatusCode() == statusCode) {
                throwException = false;

                LOG.trace("IC client received a resource NOT_FOUND error from server => proceeding normally");
            } else {
                if ((e.getCode() == 503) || (cause instanceof IOException) || StringUtils.containsIgnoreCase(causeMessage, "terminated")) {
                    m_remoteValidator.resetConnection();

                    LOG.warn(new StringBuilder("IC client lost remote connection (cause: ").
                        append(StringUtils.isBlank(causeMessage) ? "n/a" : causeMessage).
                        append(")").toString());
                }

                return new ImageConverterException(throwMessageBuilder.toString(), e);
            }
        }

        if (throwException) {
            return new ImageConverterException(throwMessageBuilder.toString());
        }
        
        return null;
    }

    /**
     * @param context
     * @return
     */
    private static String implGetContext(final String... context) {
        return ((isNotEmpty(context) && isNotEmpty(context[0])) ?
            context[0] :
                null);
    }

    /**
     * @param inputStm
     * @return
     * @throws ImageConverterException
     */
    private File implCreateTempFile(@NonNull final InputStream inputStm) throws ImageConverterException {
        File ret = null;

        try {
            ret = File.createTempFile("oxiccli", ".tmp", m_clientConfig.getSpoolDir());

            if (null != ret) {
                FileUtils.copyInputStreamToFile(inputStm, ret);

                if (ret.length() <= 0) {
                    FileUtils.deleteQuietly(ret);
                    ret = null;
                }
            }
        } catch (IOException e) {
            throw new ImageConverterException("Error whilee creating temp. file", e);
        }

        return ret;
    }

    /**
     * @param metadataImageMultipartFile
     */
    private static MetadataImage implCreateMetadataImage(@NonNull final ImageConverterApi api, @NonNull final InputStream metadataImageMultipartInputStm) throws IOException, IndexOutOfBoundsException {
        final String contentType = api.getApiClient().getResponseHeaders().get( "Content-Type").get(0);
        int boundaryStartPos = -1;
        MetadataImage ret = null;

        if (isNotEmpty(contentType) && contentType.startsWith("multipart/") &&
            ((boundaryStartPos = contentType.lastIndexOf("boundary")) > -1)) {

            final String boundary = (contentType.substring(boundaryStartPos).split("="))[1].trim();
            final MultipartStream multipartStream = new MultipartStream(metadataImageMultipartInputStm, boundary.getBytes(), 8192, null);
            InputStream imageInputStm = null;
            IMetadata metadata = null;

            if (multipartStream.skipPreamble()) {
                do {
                    final String headers = multipartStream.readHeaders();

                    try (final ByteArrayOutputStream bodyOutputStm = new ByteArrayOutputStream()) {
                        if (multipartStream.readBodyData(bodyOutputStm) > 0) {
                            if (headers.contains("application/octet-stream")) {
                                imageInputStm = new ByteArrayInputStream(bodyOutputStm.toByteArray());
                            } else if (headers.contains("application/json")) {
                                final IMetadataReader metadataReader = Services.optService(IMetadataReader.class);

                                if (null != metadataReader) {
                                    try {
                                        metadata = metadataReader.readMetadata(new JSONObject(IOUtils.toString(new ByteArrayInputStream(bodyOutputStm.toByteArray()))));
                                    } catch (Exception e) {
                                        LOG.warn(Throwables.getRootCause(e).getMessage());
                                    }
                                }
                            }
                        }
                    }
                 } while (multipartStream.readBoundary());

                 ret = new MetadataImage(imageInputStm, metadata);
            }
        }

        return ret;
    }

    // -- Statics --------------------------------------------------------------

    /**
     * @return
     */
    public static Logger getLogger() {
        return LOG;
    }

    // - Members ---------------------------------------------------------------

    final protected static Logger LOG = LoggerFactory.getLogger(ImageConverterClient.class);

    final private ImageConverterClientConfig m_clientConfig;

    final private ImageConverterRemoteValidator m_remoteValidator;

    final private ImageConverterApiProvider m_apiProvider;
}
