
package com.openexchange.guard.support;

import java.util.ArrayList;
import java.util.List;
import com.openexchange.exception.OXException;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.keymanagement.commons.DeletedKey;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.storage.DeletedKeysStorage;
import com.openexchange.guard.keymanagement.storage.KeyTableStorage;
import com.openexchange.guard.keymanagement.storage.PGPKeysStorage;
import com.openexchange.guard.osgi.Services;

/**
 * {@link UserDeleter} can be used for user or context based key deletion
 */
//TODO: maybe solve with a com.openexchange.groupware.delete.DeleteListener?
public class UserDeleter {

    private void backupKeys(List<GuardKeys> keys) throws OXException {
        //Backup keys to deleted key table / backup table
        ArrayList<DeletedKey> deletedKeys = new ArrayList<DeletedKey>();
        for (GuardKeys key : keys) {
            deletedKeys.add(new DeletedKey(key));
        }
        if (deletedKeys.size() > 0) {
            DeletedKeysStorage deletedKeysStorage = Services.getService(DeletedKeysStorage.class);
            deletedKeysStorage.insert(deletedKeys.toArray(new DeletedKey[deletedKeys.size()]));
        }
    }

    private void removeEmailMappings(int contextId) throws OXException {
        EmailStorage ogEmailStorage = Services.getService(EmailStorage.class);
        ogEmailStorage.deleteAllForContext(contextId);
    }

    private void removeEmailMappings(int userId, int contextId) throws OXException {
        EmailStorage ogEmailStorage = Services.getService(EmailStorage.class);
        ogEmailStorage.deleteAllForUser(contextId, userId);
    }

    private void removeKeys(int contextId) throws OXException {
        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        ogKeyTableStorage.deleteAllForContext(contextId);
    }

    private void removeKeys(int userId, int contextId) throws OXException {
        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        ogKeyTableStorage.deleteAllForUser(userId, contextId);
    }

    private void removeKeyMappings(List<GuardKeys> keys) throws OXException {
        for (GuardKeys key : keys) {
            //TODO: Use a service locator for resolving this dependencies
            PGPKeysStorage pgpKeysStorage = Services.getService(PGPKeysStorage.class);
            pgpKeysStorage.deleteByKeyId(key.getKeyid());
        }
    }

    /**
     * Deletes all Guard PGP keys within a given context.
     * Keys are backed up in {@link DeletedKeysStorage} before deletion
     *
     * @param contextId the id of the context
     * @throws OXException due an error
     */
    public void deleteWithBackup(int contextId) throws OXException {

        KeyTableStorage keyTableStorage = Services.getService(KeyTableStorage.class);

        //Backup and remove all keys within the deleted context
        List<GuardKeys> keys = keyTableStorage.getKeysForContext(contextId);
        backupKeys(keys);

        removeKeyMappings(keys);
        removeKeys(contextId);
        //The order matters: removing the email mapping at the end, because it contains the shard id for the other operations
        removeEmailMappings(contextId);
    }

    /**
     * Deletes all Guard PGP keys related to a given user.
     * Keys are backed up in {@link DeletedKeysStorage} before deletion
     *
     * @param userId the user's id
     * @param contextId the context id
     * @throws OXException due an error
     */
    public void deleteWithBackup(int userId, int contextId) throws OXException {

        KeyTableStorage keyTableStorage = Services.getService(KeyTableStorage.class);
        if (keyTableStorage.exists(userId, contextId, 0 /* not relevant */)) {
            List<GuardKeys> keys = keyTableStorage.getKeysForUser(userId, contextId);
            backupKeys(keys);
            removeKeyMappings(keys);
            removeKeys(userId, contextId);
        }
        //The order matters: removing the email mapping at the end, because it contains the shard id for the other operations
        removeEmailMappings(userId, contextId);
    }

    /**
     * Deletes all Guard PGP keys related to a given user without creating a backup.
     * For backing up keys before deletion see {@link #deleteWithBackup(int,int)}
     *
     * @param userId
     * @param contextId
     * @throws OXException
     */
    public void delete(int userId, int contextId) throws OXException {

        KeyTableStorage keyTableStorage = Services.getService(KeyTableStorage.class);
        if (keyTableStorage.exists(userId, contextId, 0 /* not relevant */)) {
            List<GuardKeys> keys = keyTableStorage.getKeysForUser(userId, contextId);
            removeKeyMappings(keys);
            removeKeys(userId, contextId);
        }
        //The order matters: removing the email mapping at the end, because it contains the shard id for the other operations
        removeEmailMappings(userId, contextId);
    }
}
