/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.support;

import java.net.URI;
import com.openexchange.exception.OXException;
import com.openexchange.guard.email.storage.ogEmail.Email;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.guestupgrade.GuestUpgradeService;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.support.PasswordReseter.EmailTarget;

/**
 * Default implementation of the SupportService
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class SupportServiceImpl implements SupportService {

    @Override
    public URI exposeKey(int contextId, String email) throws OXException {
        return new DeletedKeyExposer().exposeKey(email, contextId);
    }

    @Override
    public EmailTarget resetPassword(String email, String language) throws OXException {
        //finding user name and context id for the given email
        EmailStorage ogEmailStorage = Services.getService(EmailStorage.class);
        Email mapping = ogEmailStorage.getByEmail(email);

        if (mapping != null) {
            return new PasswordReseter().resetPassword(mapping.getUserId(), mapping.getContextId(), language);
        } else {
            throw GuardCoreExceptionCodes.KEY_NOT_FOUND_FOR_MAIL_ERROR.create(email, 0);
        }
    }

    @Override
    public void deleteUser(int contextId, int userId) throws OXException {
        new UserDeleter().deleteWithBackup(userId, contextId);
    }

    @Override
    public void deleteUser(String email) throws OXException {
        if (email == null || email.isEmpty()) {
            throw new IllegalArgumentException("Missing parameter email");
        }
        //Getting user-ID and CID by the user's by email
        EmailStorage ogEmailStorage = Services.getService(EmailStorage.class);
        Email ogEmail = ogEmailStorage.getByEmail(email);
        if (ogEmail != null) {
            deleteUser(ogEmail.getContextId(), ogEmail.getUserId());
        } else {
            throw new IllegalArgumentException("no such user");
        }
    }

    @Override
    public void removePin(String email) throws OXException {
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        keyService.removePin(email);
    }

    @Override
    public void upgradeGuestAccount(String email) throws OXException {
        GuestUpgradeService guestUpgradeService = Services.getService(GuestUpgradeService.class);
        guestUpgradeService.upgrade(email);
    }
}
