/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.pgpmail;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import com.google.gson.JsonObject;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.handler.ResponseHandler;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.pgp.PGPUtils;
import com.openexchange.guard.pgp.decryption.GuardEMailDecryptionService;
import com.openexchange.guard.pgp.decryption.PGPDecryptionResult;
import com.openexchange.guard.pgp.decryption.impl.GuardEMailDecryptionServiceImpl;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.AuthenticationDataExtractor;
import com.openexchange.guard.servlets.authentication.GuardUserSession;

/**
 * Retrieves the decrypted PGP mail
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class GetPGPMailAction extends GuardServletAction {

    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception {
        JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
        OxCookie cookie = new OxCookie(request.getCookies());
        String session = request.getParameter("session");
        String userAgent = request.getHeader("User-Agent");
        String emailFolder = ServletUtils.getStringParameter(request, "folder", true);
        String emailId = ServletUtils.getStringParameter(request, "emailid", true);
        int userId = userSession.getUserId();
        int contextId = userSession.getContextId();
        String password = new AuthenticationDataExtractor().getPassword(request, json, cookie);
        if (password == null) {
            throw GuardCoreExceptionCodes.PARAMETER_MISSING.create("Password or auth missing.");
        }

        //Decrypt the MIME message
        GuardEMailDecryptionService decryptionService = new GuardEMailDecryptionServiceImpl();
        PGPDecryptionResult decryptMimeEmail = decryptionService.decryptMimeEmail(cookie, session, userAgent, emailFolder, emailId, userId, contextId, password, ServletUtils.getClientIP(request));

        // If has error in decode, send and return
        if (decryptMimeEmail.hasError()) {
            ServletUtils.sendAnswer(response, 406, decryptMimeEmail.getError());
            return;
        }

        //encrypting the user's password and send it back; because the client can reuse the encrypted password in further GET requests
        String jSessionId = cookie.getJSESSIONID();
        if(jSessionId != null && !jSessionId.isEmpty()) {
            String epassword = PGPUtils.encodeEPass(password, jSessionId, userId, contextId);
            //Create JSON response structure
            JsonObject jsonResponse = DecryptedMailResponseFactory.getMimeResponse(decryptMimeEmail, epassword);
            ServletUtils.sendJsonOK(response, jsonResponse);
        } else {
            throw GuardCoreExceptionCodes.PARAMETER_MISSING.create("JSESSIONID missing.");
        }
    }
}
