/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.guest;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.Enumeration;
import java.util.Properties;
import java.util.TimeZone;

import javax.mail.Address;
import javax.mail.Header;
import javax.mail.Message.RecipientType;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Part;
import javax.mail.Session;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.exception.OXException;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.common.java.Strings;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.common.util.IDNUtil;
import com.openexchange.guard.common.util.JsonUtil;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.database.GuardShardingService;
import com.openexchange.guard.email.storage.ogEmail.Email;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.exceptions.GuardAuthExceptionCodes;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.handler.ResponseHandler;
import com.openexchange.guard.inputvalidation.EmailValidator;
import com.openexchange.guard.inputvalidation.UUIDValidator;
import com.openexchange.guard.internal.UserData;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.GuardMasterKeyService;
import com.openexchange.guard.keymanagement.services.PasswordChangeService;
import com.openexchange.guard.keymanagement.storage.KeyTableStorage;
import com.openexchange.guard.mailcreator.MailCreatorService;
import com.openexchange.guard.notification.GuardNotificationService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.pgp.PGPResult;
import com.openexchange.guard.pgp.PGPUtils;
import com.openexchange.guard.pgp.decryption.PGPMimeAttachmentExtractor;
import com.openexchange.guard.pgp.exceptions.GuardPGPExceptionCodes;
import com.openexchange.guard.servlets.pgpmail.DecryptedMailResponseFactory;
import com.openexchange.guard.session.GuardSessionService;
import com.openexchange.guard.storage.Storage;

public class Guest {

    private static Logger LOG = LoggerFactory.getLogger(Guest.class);

    /**
     * Changing Guest password
     *
     * @param request
     * @param response
     * @param cookie
     */
    public void changePass(HttpServletRequest request, HttpServletResponse response, OxCookie cookie) throws OXException {
        String authToken = ServletUtils.getStringParameter(request, "auth", true);
        cookie.setJSESSIONID(cookie.getOxReaderID());  // For guest, use ReaderID
        UserData guestData = new UserData(authToken, cookie);
        JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
        PasswordChangeService passService = Services.getService(PasswordChangeService.class);
        String newpass = JsonUtil.getStringFromJson(json, "newpass", true).trim();
        String question = "";
        if (json.has("question")) {// Sent with new question and answer?
            question = json.get("question").getAsString();
        }
        String answer = "";
        if (json.has("answer")) {
            answer = json.get("answer").getAsString();
        }
        String email = "";
        if (json.has("email")) {
            email = new EmailValidator().assertInput(json.get("email").getAsString());
            answer = json.get("email").getAsString();
            if (!answer.equals("")) {
                question = "e";
            }
        }
        String oldpass = "";
        oldpass = JsonUtil.getStringFromJson(json, "oldpass", true);
        String result = passService.changePassword(guestData.getUserid(), guestData.getCid(), email, oldpass, newpass, question, answer, cookie.getOxReaderID());
        ServletUtils.sendOK(response, "application,json", result);
    }

    /**
     * For initial decoding of PGP Guest email
     *
     * @param request
     * @param response
     * @param cookie
     * @throws OXException
     */
    public void getEmail(HttpServletRequest request, HttpServletResponse response, OxCookie cookie) throws OXException {
        decodeGuestPgp(request, response, cookie, "mail");
    }

    /**
     * Decodes Guest PGP email. Opens attachment if action open, returns email json if action mail
     *
     * @param request
     * @param response
     * @param cookie
     * @param action
     * @throws OXException
     */
    private void decodeGuestPgp(HttpServletRequest request, HttpServletResponse response, OxCookie cookie, String action) throws OXException {

        JsonObject jdata = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
        String itemid = "";
        if (jdata != null) {
            itemid = JsonUtil.getStringFromJson(jdata, "item", true);
        } else {
            itemid = ServletUtils.getStringParameter(request, "email", true);
        }
        if (itemid.startsWith("pgp")) {
            itemid = new UUIDValidator("pgp-").assertInput(itemid, "itemid");
            String auth = ServletUtils.getStringParameter(request, "auth", true);
            cookie.setJSESSIONID(cookie.getOxReaderID()); // In case person uses both reader and OX main, we need to clear out the previous cookie
            UserData data = new UserData(auth, cookie);
            if (data.getEncr_password() == null) {
                response.setStatus(HttpServletResponse.SC_BAD_REQUEST);
                return;
            }
            try {
                Storage storage = Services.getService(Storage.class);
                InputStream in = storage.readObj(data.getUserid(), data.getCid(), itemid);
                if (in != null) {
                    PGPResult result = null;
                    try {
                        result = PGPUtils.decryptFile(in, data.getUserid(), data.getCid(), data.getEncr_password());

                    } catch (OXException ex) {
                        if (GuardAuthExceptionCodes.BAD_PASSWORD.equals(ex) || GuardPGPExceptionCodes.BAD_PASSWORD_ERROR.equals(ex)) {
                            in.close();
                            ServletUtils.sendNotAcceptable(response, "Bad password");
                            return;
                        }
                    }
                    in.close();
                    if (result.getError() != null) {
                        ServletUtils.sendNotAcceptable(response, result.getError());
                    } else {
                        switch (action) {
                            case "mail":
                                ServletUtils.sendHtmlOK(response, createReturnJson(result, data.getEncr_password(), cookie));
                                break;
                            case "open":
                                //Creating Mime message object from decoded content
                                MimeMessage decryptedMimeMessage = new MimeMessage(Session.getDefaultInstance(new Properties()), new ByteArrayInputStream(result.getDecoded()));

                                //Extracting the requested attachment from the MIME message
                                String filename = ServletUtils.getStringParameter(request, "attname", true);
                                Part decryptedAttachment = new PGPMimeAttachmentExtractor().getAttachmentFromMessage(decryptedMimeMessage, filename);
                                if (decryptedAttachment != null) {
                                    boolean download = (ServletUtils.getStringParameter(request, "download") == null ? false : (ServletUtils.getStringParameter(request, "download").toLowerCase().equals("true")));
                                    if (download) {
                                        response.setContentType(decryptedAttachment.getContentType());
                                        response.setHeader("Content-Disposition", "attachment; filename=\"" + filename + "\"");
                                        // response.addHeader("Access-Control-Allow-Origin", "*");
                                        response.setStatus(HttpServletResponse.SC_OK);
                                    } else {
                                        response.setContentType(decryptedAttachment.getContentType());
                                        response.setStatus(HttpServletResponse.SC_OK);
                                    }
                                    IOUtils.copy(decryptedAttachment.getInputStream(), response.getOutputStream());
                                    response.flushBuffer();

                                    LOG.info("Sending decoded attachment " + filename);
                                }
                                //PgpHandler.getAttach(response, result, filename, download);
                                break;
                            case "cid":
                                MimeMessage decryptedMimeMessage2 = new MimeMessage(Session.getDefaultInstance(new Properties()), new ByteArrayInputStream(result.getDecoded()));

                                //Extracting the requested cid attachment from the MIME message
                                String content_id = ServletUtils.getStringParameter(request, "content_id", true);
                                Part decryptedCidAttachment = new PGPMimeAttachmentExtractor().getAttachmentFromMessageID(decryptedMimeMessage2, content_id);
                                if (decryptedCidAttachment != null) {
                                    response.setContentType(decryptedCidAttachment.getContentType());
                                    response.setStatus(HttpServletResponse.SC_OK);
                                    IOUtils.copy(decryptedCidAttachment.getInputStream(), response.getOutputStream());
                                    response.flushBuffer();
                                    LOG.info("Sending decoded inline cid attachment " + content_id);
                                } else {
                                    ServletUtils.sendResponse(response, 404, "Not found");
                                }

                                break;
                        }
                    }
                } else {
                    ServletUtils.sendNotAcceptable(response, "not found");
                    LOG.info("Guest item not found: " + itemid);
                }
            } catch (Exception ex) {
                LOG.error("Problemd decoding Guest PGP ", ex);

            }
        } else {  // Proprietary Guard
            LOG.error("Call for proprietary format");
            try {
                ServletUtils.sendResponse(response, HttpServletResponse.SC_NOT_FOUND, "not found");
            } catch (IOException e) {
                LOG.error("Error sending response message", e);
            }
        }
    }

    /**
     * Create return JSON for display in Guest browser
     *
     * @param result
     * @param json
     * @param password
     * @param cookie
     * @return
     * @throws MessagingException
     * @throws IOException
     * @throws OXException
     */
    public String createReturnJson(PGPResult result, String password, OxCookie cookie) throws MessagingException, IOException, OXException {
        Session s = Session.getDefaultInstance(new Properties());
        JsonObject data = new JsonObject();
        InputStream in = new ByteArrayInputStream(result.getDecoded());
        MimeMessage message = new MimeMessage(s, in);
        Object content = message.getContent();
        JsonArray attachments = new JsonArray();
        if (content instanceof String) {
            DecryptedMailResponseFactory.ParseStringBody((String) content, attachments);  // Plaintext only message
        } else {
            DecryptedMailResponseFactory.ParseMultipart((Multipart) content, attachments); // Multipart message
        }
        JsonObject Results = new JsonObject();
        Results.add("signature", new JsonPrimitive(result.isSignature()));
        Results.add("verified", new JsonPrimitive(result.isVerified()));
        if (password != null) {
            GuardSessionService sessionService = Services.getService(GuardSessionService.class);
            String token = sessionService.getToken(cookie.getJSESSIONID());
            GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
            Results.addProperty("epassword", cipherService.encrypt(password, token));
        }
        if (result.getError() != null) {
            Results.add("error", new JsonPrimitive(result.getError()));
        }
        data.add("attachments", attachments);
        data.add("results", Results);
        Address[] fromarray = message.getFrom();
        JsonArray fromJson = new JsonArray();
        if (fromarray != null) {
            for (int i = 0; i < fromarray.length; i++) {
                JsonArray recip = new JsonArray();
                InternetAddress fromAddr = (InternetAddress) fromarray[i];
                recip.add(new JsonPrimitive(fromAddr.getPersonal() == null ? "" : fromAddr.getPersonal()));
                recip.add(new JsonPrimitive(IDNUtil.decodeEmail(fromAddr.getAddress() == null ? "" : fromAddr.getAddress())));
                fromJson.add(recip);
            }
        }
        data.add("from", fromJson);
        Address[] toarray = message.getRecipients(RecipientType.TO);
        JsonArray toJson = new JsonArray();
        if (toarray != null) {
            for (int i = 0; i < toarray.length; i++) {
                JsonArray recip = new JsonArray();
                InternetAddress to = (InternetAddress) toarray[i];
                recip.add(new JsonPrimitive(to.getPersonal() == null ? "" : to.getPersonal()));
                recip.add(new JsonPrimitive(IDNUtil.decodeEmail(to.getAddress() == null ? "" : to.getAddress())));
                toJson.add(recip);
            }
        }
        data.add("to", toJson);
        Address[] ccarray = message.getRecipients(RecipientType.CC);
        JsonArray ccJson = new JsonArray();
        if (ccarray != null) {
            for (int i = 0; i < ccarray.length; i++) {
                JsonArray recip = new JsonArray();
                InternetAddress cc = (InternetAddress) ccarray[i];
                recip.add(new JsonPrimitive(cc.getPersonal() == null ? "" : cc.getPersonal()));
                recip.add(new JsonPrimitive(IDNUtil.decodeEmail(cc.getAddress() == null ? "" : cc.getAddress())));
                ccJson.add(recip);
            }
        }
        data.add("cc", ccJson);
        Enumeration<Header> headers = message.getAllHeaders();
        JsonObject headersJson = new JsonObject();
        while (headers.hasMoreElements()) {
            Header h = headers.nextElement();
            headersJson.addProperty(h.getName(), h.getValue());
        }
        data.add("headers", headersJson);
        data.addProperty("subject", message.getSubject());
        final String ISO_FORMAT = "yyyy-MM-dd'T'HH:mm:ss.SSS'+00:00'";
        final SimpleDateFormat sdf = new SimpleDateFormat(ISO_FORMAT);
        final TimeZone utc = TimeZone.getTimeZone("UTC");
        sdf.setTimeZone(utc);
        data.addProperty("sent", sdf.format(message.getSentDate()));
        in.close();
        return (data.toString());
    }

    /**
     * Function to return encrypted attachment (non email)
     *
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void getEncrAttach(HttpServletRequest request, HttpServletResponse response, OxCookie cookie) throws Exception {
        String emailid = ServletUtils.getStringParameter(request, "email", true);
        if (emailid.startsWith("pgp")) {
            cookie.setJSESSIONID(cookie.getOxReaderID());
            decodeGuestPgp(request, response, cookie, "open");
            return;
        } else {
            LOG.error("Call for proprietary email format");
        }
    }

    /**
     * Function to return encrypted CID image for inline images
     *
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void getEncrCidAttach(HttpServletRequest request, HttpServletResponse response, OxCookie cookie) throws Exception {
        cookie.setJSESSIONID(cookie.getOxReaderID());
        decodeGuestPgp(request, response, cookie, "cid");
        return;
    }

    /**
     * Get password reset question
     *
     * @param request
     * @param response
     * @param cookie
     * @throws IOException
     */
    public void getQuestion(HttpServletRequest request, HttpServletResponse response, OxCookie cookie) throws OXException {
        String email = new EmailValidator().assertInput(ServletUtils.getStringParameter(request, "email", true), "email");
        EmailStorage ogEmailStorage = Services.getService(EmailStorage.class);

        //Getting the mail mapping
        String question = "nf";
        Email emailKeyMapping = ogEmailStorage.getByEmail(email);
        if (emailKeyMapping != null) {
            //Getting the guest's key
            KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
            GuardKeys key = ogKeyTableStorage.getKeyForEmailAndContext(email, emailKeyMapping.getContextId());
            if (key != null) {
                question = key.getQuestion();
                if (!key.isRecoveryAvail()) {
                    question = "nr";
                }
            }
        } else {
            LOG.error(String.format("No email mapping found for email %s", email));
        }
        ServletUtils.sendHtmlOK(response, question);
    }

    /**
     * Perform password reset after question answered
     *
     * @param request
     * @param response
     * @param cookie
     * @throws IOException
     * @throws OXException
     */
    public void resetPassword(HttpServletRequest request, HttpServletResponse response, OxCookie cookie) throws OXException {
        String email = new EmailValidator().assertInput(ServletUtils.getStringParameter(request, "email", true), "email");
        String answer = ServletUtils.getStringParameter(request, "answer", true);
        String lang = ServletUtils.getStringParameter(request, "language");
        Integer templId = ServletUtils.getIntParameter(request, "templid");
        if(templId == null) {
            templId = 0;
        }

        String question = "";

        int cid = 0;
        int userid = 0;
        try {
            GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
            GuardAntiAbuseService antiabuse = Services.getService(GuardAntiAbuseService.class);
            if (antiabuse.isBad("reset-" + email, configService.getIntProperty(GuardProperty.badPasswordCount))) {
                ServletUtils.sendHtmlOK(response, "lockout");
                return;
            }

            KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
            GuardShardingService shardingService = Services.getService(GuardShardingService.class);
            GuardKeys key = ogKeyTableStorage.getKeyForGuest(email, shardingService.getShard(email));

            String prevanswer = null;
            if (key != null) {
                prevanswer = key.getAnswer();
                userid = key.getUserid();
                cid = key.getContextid();
                question = key.getQuestion();
            }
            if (question != null) {  // PIN is not used for answer
                if (question.equals("PIN")) {
                    prevanswer = null;
                }
            }
            if (prevanswer != null) {
                GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
                if (Strings.isEmpty(question)) {
                    question = "default";
                }
                String rc = Services.getService(GuardMasterKeyService.class).getMasterKey().getRC();
                prevanswer = cipherService.decrypt(prevanswer, rc, question);
                if (prevanswer == null) {
                    ServletUtils.sendNotAcceptable(response, "Error decoding answer");
                    return;
                }
                prevanswer = prevanswer.toLowerCase().replace(" ", "");
                answer = answer.toLowerCase().replace(" ", "");
                if (!prevanswer.equals(answer)) {
                    ServletUtils.sendHtmlOK(response, "No Match");
                    antiabuse.addBad("reset-" + email);
                    return;
                }
            }
            String senderIp = (request == null ? null: request.getRemoteAddr());
            reset(email, lang, templId, cid, userid, request.getServerName(), senderIp);

            antiabuse.removeBad(email);
            ServletUtils.sendHtmlOK(response, "OK");
        } catch (Exception ex) {
            LOG.error("Error resetting password for email account " + email, ex);
            ServletUtils.sendNotAcceptable(response, "Failed to change");
        }
    }

    protected void reset(String email, String lang, int templId, int cid, int userid, String host, String senderIp) throws OXException {
        GuardKeyService keyService = Services.getService(GuardKeyService.class);

        boolean recoveryEnabled = keyService.isRecoveryEnabled(userid, cid);
        if (!recoveryEnabled) {  // No recovery available
            throw GuardCoreExceptionCodes.DISABLED_ERROR.create("reset password");
        }

        String newpass = keyService.generatePassword();

        MailCreatorService mailCreatorService = Services.getService(MailCreatorService.class);
        JsonObject mail = mailCreatorService.getResetEmail(email, mailCreatorService.getFromAddress(email, email, userid, cid), newpass, lang, templId, host, userid, cid);

        GuardNotificationService guardNotificationService = Services.getService(GuardNotificationService.class);
        guardNotificationService.send(mail, senderIp);

        keyService.resetPassword(email, newpass);
    }
}
