/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.authentication;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.session.GuardSessionService;
import static com.openexchange.guard.servlets.util.ServletLocaleUtil.*;
import static com.openexchange.guard.servlets.authentication.ConnectionInformationFactory.*;

/**
 * {@link GuestSessionValidator}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.2
 */
public class GuestSessionValidator implements SessionValidator {

    private static final String AUTH_PARAMETER_NAME = "auth";
    private static final String GUEST_READER_COOKIE_NAME = "OxReaderID";
    private static final Logger logger = LoggerFactory.getLogger(GuestSessionValidator.class);

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.servlets.authentication.SessionValidator#validateSession(javax.servlet.http.HttpServletRequest)
     */
    @Override
    public GuardUserSession validateSession(HttpServletRequest request) throws OXException {
        String authToken = request.getParameter(AUTH_PARAMETER_NAME);
        try {
            if (authToken != null && !authToken.isEmpty() && !authToken.trim().toLowerCase().equals("null")) {
                String token = null;
                String guestSessionId = null;
                Cookie[] cookies = request.getCookies();
                for (Cookie cookie : cookies) {
                    if (cookie.getName().equals(GUEST_READER_COOKIE_NAME)) {
                        guestSessionId = cookie.getValue();
                        GuardSessionService sessionService = Services.getService(GuardSessionService.class);
                        token = sessionService.getToken(guestSessionId);
                        if(token != null) {
                            break;
                        }
                    }
                }

                if (token != null) {
                    GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_GCM);
                    String userdata = cipherService.decrypt(authToken, token);
                    if (userdata != null && !userdata.isEmpty()) {
                        Gson json = new Gson();
                        JsonObject userDataJson = json.fromJson(userdata, JsonObject.class);
                        if (userDataJson.has("user_id") && userDataJson.has("cid")) {
                            int userId = userDataJson.get("user_id").getAsInt();
                            int contextId = userDataJson.get("cid").getAsInt();
                            //if (contextId < 0) {   // This check removed for bug 56873, was interfering after upgarde to full OX account
                                //Guest sessions do always have negative context-id
                                GuardUserSession guardUserSession = new GuardUserSession(guestSessionId, contextId, userId, getLocaleFor(request),createConnectionInformationFrom(request));
                                return guardUserSession;
                            //}
                        }
                    }
                }
            }
            return null;
        } catch (Exception e) {
            logger.error("Error decrypting guest auth-token: " + e.getMessage());
            return null;
        }
    }
}
