/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.authentication;

import javax.servlet.http.HttpServletRequest;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.internal.UserData;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.pgp.PGPUtils;

/**
 * {@link AuthenticationDataExtractor}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class AuthenticationDataExtractor {

    private static final String PARAM_E_PASSWORD = "epassword";
    private static final String PARAM_E_PASSWORD_ALTERNATIVE = "epass";
    private static final String PARAM_AUTH_TOKEN = "auth";
    private static final String PARAM_PLAIN_TEXT_PASSWORD = "password";

    private String getPassword(String authToken, String epassword, String password, OxCookie cookie) throws OXException {
        //Try to parse the RSA encrypted authentication structure
        if (authToken != null) {
            if (authToken.length() > 20) {
                UserData data = new UserData(authToken, cookie);
                if ((data.getUserid() != -1) && (data.getEncr_password() != null)) {
                    return data.getEncr_password();
                }
            }
        }

        //Try to parse the AES encrypted password
        if (epassword != null && !epassword.isEmpty()) {
            String jSessionId = cookie.getJSESSIONID();
            if(jSessionId != null && !jSessionId.isEmpty()) {
                return PGPUtils.decodeEPass(epassword, jSessionId);
            }
        }

        if (password != null) {
            return password;
        }

        return null;
    }

    private String extractEpasswordFromRequest(HttpServletRequest request) {
        String epassword = ServletUtils.getStringParameter(request, PARAM_E_PASSWORD);
        if (epassword != null && !epassword.isEmpty()) {
            return epassword;
        }
        return ServletUtils.getStringParameter(request, PARAM_E_PASSWORD_ALTERNATIVE);
    }

    public String getPassword(HttpServletRequest request, OxCookie cookie) throws OXException {
        return getPassword(request.getParameter(PARAM_AUTH_TOKEN), extractEpasswordFromRequest(request), ServletUtils.getStringParameter(request, PARAM_PLAIN_TEXT_PASSWORD), cookie);
    }

    public String requirePassword(HttpServletRequest request, OxCookie cookie) throws OXException {
        String password = getPassword(request,cookie);
        if(password == null) {
            throw GuardCoreExceptionCodes.PARAMETER_MISSING.create("Password or auth missing.");
        }
        return password;
    }

    public String getPassword(JsonObject json, OxCookie cookie) throws OXException {
        if (json != null) {
            String authToken = null;
            if (json.has(PARAM_AUTH_TOKEN) && !json.get(PARAM_AUTH_TOKEN).isJsonNull()) {
                authToken = json.get(PARAM_AUTH_TOKEN).getAsString();
            }
            String epassword = null;
            if (json.has(PARAM_E_PASSWORD) && !json.get(PARAM_E_PASSWORD).isJsonNull()) {
                epassword = json.get(PARAM_E_PASSWORD).getAsString();
            }

            String password = null;
            if (json.has(PARAM_PLAIN_TEXT_PASSWORD) && !json.get(PARAM_PLAIN_TEXT_PASSWORD).isJsonNull()) {
                password = json.get(PARAM_PLAIN_TEXT_PASSWORD).getAsString();
            }
            return getPassword(authToken, epassword, password, cookie);
        }
        return null;
    }

    public String requirePassword(JsonObject json, OxCookie cookie) throws OXException {
        String password = getPassword(json, cookie);
        if(password == null) {
            throw GuardCoreExceptionCodes.PARAMETER_MISSING.create("Password or auth missing.");
        }
        return password;
    }


    public String getPassword(HttpServletRequest request, JsonObject json, OxCookie cookie) throws OXException {
        String password = getPassword(request, cookie);
        if (password == null || password.isEmpty() /* UI seems to send empty password in some cases */) {
            password = getPassword(json, cookie);
        }
        return password;
    }

    public String requirePassword(HttpServletRequest request, JsonObject json, OxCookie cookie) throws OXException {
        String password = getPassword(request, json, cookie);
        if(password == null) {
            throw GuardCoreExceptionCodes.PARAMETER_MISSING.create("Password or auth missing.");
        }
        return password;
    }
}
