
package com.openexchange.guard.pgp;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.List;
import javax.mail.BodyPart;
import javax.mail.MessagingException;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMultipart;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.translation.GuardTranslationService;
import com.openexchange.pgp.mail.tools.PGPMimeMailCreator;


public class GuardPGPMimeMailCreator {

    private static final Logger LOG = LoggerFactory.getLogger(GuardPGPMimeMailCreator.class);

    public MimeMultipart createPGPMime(final byte[] data, final List<String> languages) throws Exception {
        List<BodyPart> additionalClearTextFiles = null;
        if(languages != null) {
            additionalClearTextFiles = getHelpFiles(languages);
        }
        return new PGPMimeMailCreator().createPGPMimePart(new ByteArrayInputStream(data), additionalClearTextFiles);
    }

    private List<BodyPart> getHelpFiles(List<String> languages) throws MessagingException, OXException {
        List<BodyPart> ret = null;
        if(languages != null) {
            ret = new ArrayList<BodyPart>();
            final List<HelpAttach> attachments = getHelp(languages);
            for (int i = 0; i < attachments.size(); i++) {
                final BodyPart help = new MimeBodyPart();
                GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
                help.setFileName(translationService.getTranslation(GuardPGPMessages.HELP, attachments.get(i).getLanguage()) + ".txt");
                help.setContent(attachments.get(i).getTranslated(), "text/plain; charset=utf-8");
                help.setDisposition("inline");
                ret.add(help);
            }
        }
        return ret;
    }

    private List<HelpAttach> getHelp(final List<String> languages) {
        final List<HelpAttach> helpfiles = new ArrayList<HelpAttach>();
        try {
            GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
            String externalReaderPath = guardConfigService.getProperty(GuardProperty.externalReaderPath);
            String oxURL = guardConfigService.getProperty(GuardProperty.externalEmailURL);

            for (int i = 0; i < languages.size(); i++) {
                GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
                final String lang = translationService.getAvailableCode(languages.get(i));
                String data = translationService.getHelpFile(lang);
                if (data != null) {
                    data = data.replace("%guest%", "https://" + externalReaderPath);
                    data = data.replace("%webmail%", "https://" + oxURL);
                    if (!contains(helpfiles, lang)) {
                        final HelpAttach att = new HelpAttach(lang, data);
                        helpfiles.add(att);
                    }
                }
            }
        } catch (final Exception e) {
            LOG.error("Problem loading help file ", e);
        }
        return (helpfiles);
    }

    private boolean contains(final List<HelpAttach> list, final String lang) {
        for (final HelpAttach att : list) {
            if (att.getLanguage().equals(lang)) {
                return (true);
            }
        }
        return (false);
    }
}
