/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.mail;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.mail.Message.RecipientType;
import javax.mail.internet.InternetAddress;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonNull;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonPrimitive;
import com.openexchange.exception.OXException;
import com.openexchange.guard.GuardInlineImageHandler;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.common.util.IDNUtil;
import com.openexchange.guard.common.util.JsonUtil;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.email.storage.ogEmail.Email;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.encryption.EncryptedItem;
import com.openexchange.guard.encryption.EncryptedItemsStorage;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.guest.GuestInlineImageHandler;
import com.openexchange.guard.internal.UserData;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.keymanagement.commons.export.KeyExportUtil;
import com.openexchange.guard.keymanagement.services.AccountCreationService;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.GuardMasterKeyService;
import com.openexchange.guard.keymanagement.services.RecipKeyService;
import com.openexchange.guard.mail.exceptions.OXMailExceptionCodes;
import com.openexchange.guard.mailcreator.MailCreatorService;
import com.openexchange.guard.mime.services.Attachment;
import com.openexchange.guard.mime.services.MimeParser;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.ApiResponse;
import com.openexchange.guard.oxapi.InlineImage;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.oxapi.Utilities;
import com.openexchange.guard.pgp.PGPMail;
import com.openexchange.guard.pgp.PGPResult;
import com.openexchange.guard.pgp.PGPSender;
import com.openexchange.guard.pgp.PGPUserIdChecker;
import com.openexchange.guard.pgp.PGPUtils;
import com.openexchange.guard.pgp.decryption.GuardEMailDecryptionService;
import com.openexchange.guard.pgp.decryption.PGPDecryptionResult;
import com.openexchange.guard.pgp.decryption.impl.GuardEMailDecryptionServiceImpl;
import com.openexchange.guard.pgpcore.services.TokenAuthenticationService;
import com.openexchange.guard.ratifier.GuardRatifierService;
import com.openexchange.guard.user.GuardCapabilities;
import com.openexchange.guard.user.GuardUserService;
import com.openexchange.guard.user.UserIdentity;
import com.openexchange.guard.user.GuardCapabilities.Permissions;
import com.openexchange.mail.usersetting.UserSettingMail;
import com.openexchange.mail.usersetting.UserSettingMailStorage;

public class Incoming {

    private static Logger LOG = LoggerFactory.getLogger(Incoming.class);

    private final OxCookie cookie;
    private String sessionId;
    private final InlineImageHandler[] inlineImageHandlers;

    public Incoming(OxCookie ck, HttpServletRequest rq) {
        cookie = ck;
        sessionId = rq.getParameter("session");
        inlineImageHandlers = new InlineImageHandler[] {
            new AppsuiteInlineImageHandler(),
            new GuardInlineImageHandler(),
            new GuestInlineImageHandler()
        };
    }

    private ServletFileUpload createServletFileUpload(long maxSize, long maxFileSize) {
        DiskFileItemFactory diskFileItemFactory = new DiskFileItemFactory();
        ServletFileUpload servletFilleUpload = new ServletFileUpload(diskFileItemFactory);
        servletFilleUpload.setSizeMax(maxSize);
        servletFilleUpload.setFileSizeMax(maxFileSize);
        return servletFilleUpload;
    }

    private ServletFileUpload getServletFileUploadForOxUser(int contextId, int userId) throws OXException {
        //Getting maxSize of a mail
        UserSettingMail usm = UserSettingMailStorage.getInstance().getUserSettingMail(userId, contextId);
        long maxSize = usm.getUploadQuota();
        if (maxSize <= 0) {
            if (maxSize == 0) {
                maxSize = -1L;
            }
            else {
                GuardConfigurationService configurationService = Services.getService(GuardConfigurationService.class);
                String maxSizeProperty = configurationService.getPropertyFromFile("server.properties", "MAX_UPLOAD_SIZE");
                if (maxSizeProperty != null) {
                    maxSize = Long.parseLong(maxSizeProperty);
                    if (maxSize ==0) {
                        maxSize = -1L;
                    }
                }
                else {
                    maxSize = -1L;
                }
            }
        }

        //Getting max file size of a mail attachment
        long maxFileSize = usm.getUploadQuotaPerFile();
        if (maxFileSize <= 0) {
            maxFileSize = -1L;
        }

        return createServletFileUpload(maxSize, maxFileSize);
    }

    private ServletFileUpload getServletFileUploadForGuest() throws OXException {
        GuardConfigurationService configurationService = Services.getService(GuardConfigurationService.class);
        String maxSizeProperty = configurationService.getPropertyFromFile("server.properties", "MAX_UPLOAD_SIZE");
        long maxSize = -1L;
        if (maxSizeProperty != null) {
            maxSize = Long.parseLong(maxSizeProperty);
            if(maxSize == 0) {
                maxSize = -1L;
            }
        }
        return createServletFileUpload(maxSize, maxSize);
    }

    private ServletFileUpload getServletFileUpload(int contextId, int userId) throws OXException {
        if (contextId > 0) {
            return getServletFileUploadForOxUser(contextId, userId);
        }
        else {
            return getServletFileUploadForGuest();
        }
    }

    private UserIdentity getIdentityForToken(String sessionIdentifier, String authToken) throws OXException {
        TokenAuthenticationService service = Services.getService(TokenAuthenticationService.class);
        return service.decryptUserIdentity(sessionIdentifier, authToken);
    }

    /**
     * Handle email sent from UI
     *
     * @param request
     * @param response
     * @throws Exception
     */

    public void incomingEmailForm(HttpServletRequest request, HttpServletResponse response, int userid, int cid) throws Exception {
        String sentfolder = ServletUtils.getStringParameter(request, "sentfolder");
        if (sentfolder == null) {
            sentfolder = "default0/Sent Items";
        }
        String guest = ServletUtils.getStringParameter(request, "guest");
        if (guest != null) {
            if (!guest.toLowerCase().trim().equals("true")) {
                guest = null;
            }
        }
        String language = ServletUtils.getStringParameter(request, "lang");
        String guestlanguage = ServletUtils.getStringParameter(request, "guestlanguage");
        String attach_password = ServletUtils.getStringParameter(request, "password");// encrypted attachment password
        String auth = ServletUtils.getStringParameter(request, "auth");
        String draftFolder = ServletUtils.getStringParameter(request, "draftfolder");
        String pin = null;
        boolean draft = ServletUtils.getBooleanParameter(request, "draft", false);
        Integer templId = ServletUtils.getIntParameter(request, "templid");
        if(templId == null) {
            templId = 0;
        }
        if (language == null) {
            language = "en-US";
        }
        EmailInfo info = null;
        String sr = ServletUtils.getStringParameter(request, "sr");
        // If not guest, verify logged in
        if (guest == null) {
            Api ap = new Api(cookie, request);
            if (!ap.verifyLogin()) {
                LOG.info("not logged into ui");
                ServletUtils.sendNotAcceptable(response, "not authorized");
                return;
            }
            // Load custom template Id for this user
        } else {
            if (sr == null) {  // Reply ID must be sent if not part of UI
                LOG.error("Reply ID not sent");
                ServletUtils.sendNotAcceptable(response, "not authorized");
                return;
            }
            cookie.setJSESSIONID(cookie.getOxReaderID());
            UserData userdata = new UserData(auth, cookie);
            cid = userdata.getCid();
            if (userdata.getUserid() < 0) {
                LOG.info("Bad authorization for guest login");
                ServletUtils.sendNotAcceptable(response, "not authorized");
                return;
            }
            userid = userdata.getUserid();

        }

        boolean sendPermission = false;
        // Check for permission for sending email
        if (guest == null) {

            GuardCapabilities guardUserCapabilities = Services.getService(GuardUserService.class).getGuardCapabilieties(cid, userid);
            if (!guardUserCapabilities.hasPermission(Permissions.MAIL)) {
                // If not permission, check if secure reply
                if (sr == null) {
                    if (guardUserCapabilities.hasMinimumPermission()) {  // Make sure they have minimal Guard permission
                        // Assuming secure reply from user without guard-mail settings
                        // ToDo in future...rate limit, recip limit, other
                        sendPermission = true;
                    } else {
                        LOG.info("not authorized");
                        ServletUtils.sendNotAcceptable(response, "not authorized");
                        return;
                    }
                }
            } else {
                sendPermission = true;
            }
        }
        if ((sr != null) && !sendPermission) {
            info = new EmailInfo();

            String emailId = sr;

            if (emailId.startsWith("pgp")) {
                emailId = emailId.substring(4);
            }

            EmailStorage ogEmailStorage = Services.getService(EmailStorage.class);
            EncryptedItemsStorage encryptedItemsStorage = Services.getService(EncryptedItemsStorage.class);

            //Getting the encrypted item
            EncryptedItem encryptedItem = encryptedItemsStorage.getById(emailId, userid, cid);
            if (encryptedItem != null) {
                info.setOwnerId(encryptedItem.getOwnerId());
                info.setOwnerCid(encryptedItem.getOwnerCid());
                info.populateRecips(encryptedItem.getXml());
            }

            //Getting sender and recipient
            if (info.getOwnerCid() != 0) {
                List<Email> ogOwnerEmails = ogEmailStorage.getById(info.getOwnerCid(), info.getOwnerId());
                if (ogOwnerEmails.size() > 0) {
                    info.setSenderEmail(ogOwnerEmails.get(0).getEmail());
                }

                List<Email> ogEmails = ogEmailStorage.getById(cid, userid);
                if (ogEmails.size() > 0) {
                    info.setFromEmail(ogEmails.get(0).getEmail());
                }
            }
        }
        GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
        if (cid < 0) {  // If negative context ID (non-ox user), use the orig senders templates for sending
            templId = guardConfigService.getIntProperty(GuardProperty.templateID, info.getOwnerId(), info.getOwnerCid());
        } else {
            templId = guardConfigService.getIntProperty(GuardProperty.templateID, userid, cid);
        }

        ArrayList<Attachment> attachments = new ArrayList<Attachment>();// Get list of all attachments
        String emaildata = "";
        String guestmessage = "";
        List<FileItem> files = null;
        try {
            ServletFileUpload sfu = getServletFileUpload(cid, userid);
            files = sfu.parseRequest(request);
        } catch (Exception uploadException) {
            throw OXMailExceptionCodes.SEND_EMAIL_ERROR.create(uploadException, uploadException.getMessage());
        }
        sessionId = ServletUtils.getStringParameter(request, "session", true);
        Iterator<FileItem> iterator = files.iterator(); // Iterate through attachments
        while (iterator.hasNext()) {
            FileItem fi = iterator.next();
            String name = fi.getFieldName();
            if (name.startsWith("guest_message")) {
                guestmessage = fi.getString("UTF-8");
                continue;
            }
            if (name.startsWith("pin")) {
                pin = fi.getString("UTF-8");
                continue;
            }
            if (name.startsWith("json")) { // Field name json is the email data (or should be). So get email data
                emaildata = fi.getString("UTF-8");
            } else {
                name = fi.getName(); // Otherwise, assume it is an attachment
                if (name != null) {// Make sure the name of the file exists, may just be blank field
                    if (!name.trim().equals("")) {
                        if (name.lastIndexOf("\\") > -1) {
                            name = name.substring(name.lastIndexOf("\\"));
                        }
                        if (name.lastIndexOf("/") > -1) {
                            name = name.substring(name.lastIndexOf("/"));
                        }
                        ByteArrayOutputStream out = new ByteArrayOutputStream();
                        InputStream input = fi.getInputStream();
                        IOUtils.copy(input, out); // Copy stream to byte array
                        Attachment attach = new Attachment();
                        attach.setContent(out.toByteArray()); // Populate attachment data
                        attach.setFilename(fi.getName());
                        attach.setType(fi.getContentType());
                        attachments.add(attach); // Add to our array of attachments
                        out.close();
                        input.close();
                    }
                }
            }
        }
        if (!emaildata.equals("")) { // We must have the basic email data
            JsonParser jsonParser = new JsonParser(); // Put it into json
            JsonObject json = (JsonObject) jsonParser.parse(emaildata);
            if (json.get("attachpassword") != null) {
                attach_password = json.get("attachpassword").getAsString();
            }
            if (json.get("epassword") != null) {
                String epass = json.get("epassword").getAsString();
                if (epass != null) {
                    attach_password = decodeRSApassword(epass);
                }
            }

            // If password wasn't sent, check if in auth
            if (attach_password == null && auth != null && !auth.isEmpty()) {
                UserIdentity id = getIdentityForToken(cookie.getJSESSIONID(), auth);
                if (id != null) {
                    attach_password = new String(id.getPassword());
                }
            }

            JsonObject data = json.getAsJsonObject("data");
            JsonArray from;
            try {
                from = data.get("from").getAsJsonArray().get(0).getAsJsonArray();
            } catch (java.lang.NullPointerException ex) {
                throw GuardCoreExceptionCodes.PARAMETER_MISSING.create("From Address Array");
            }
            if (!PGPUserIdChecker.checkHasUID(userid, cid, from.get(1).getAsString().trim())) {
                LOG.info("From email not contained in sender key");
                ServletUtils.sendNotAcceptable(response, "missingUID");
                return;
            }
            ArrayList<RecipKey> recipients = getUsers(data, userid, cid, guestlanguage, pin, info, draft, request.getServerName()); // get the users and populate the recipient keys
            if (recipients == null) {
                LOG.info("Unable to get keys for all users");
                ServletUtils.sendNotAcceptable(response, "keyproblem");
                return;
            } else {
                //Checking for new guest accounts
                Api apiAccess = new Api(new OxCookie(request.getCookies()), request);
                for (RecipKey recipient : recipients) {
                    if (guardConfigService.getBooleanProperty(GuardProperty.newGuestsRequirePassword)) {  // If Guests get new password email, send now
                        if (recipient.isGuest() && recipient.getNewGuestPass() != null && !recipient.getNewGuestPass().isEmpty()) {
                            //Sending a welcome/password email to new guests
                            try {

                                String fromName = from.get(0) != JsonNull.INSTANCE ? from.get(0).getAsString().trim() : null;
                                String fromEmail = from.get(1).getAsString().trim();
                                sendInitialGuestPasswordMail(apiAccess, fromName, fromEmail, recipient, templId, guestmessage, request.getServerName(), recipient.getUserid(), recipient.getCid());
                            } catch (OXException e) {
                                LOG.error("", e);
                                ServletUtils.sendNotAcceptable(response, e.getDisplayMessage(new Locale(language)));
                                return;
                            }
                        }
                    }
                }
            }
            if (!sendPermission) {
                if (!checkReply(from, recipients, info)) {
                    LOG.info("Attempt to reply to users not in original email list");
                    ServletUtils.sendNotAcceptable(response, "unauth recip");
                    return;
                }
            }

            // Encrypt email

            // Get infostore attachments
            // TODO This will need cleaning up, as this has changed between 7.62 on 7.8.  ?  Only infostoreID is known?  No longer in attachment data?
            try {
                if (data.has("infostore_ids")) { // If has infostore attachments, get the array of data.
                    JsonArray infostores = data.get("infostore_ids").getAsJsonArray();
                    if (infostores != null) {
                        for (int i = 0; i < infostores.size(); i++) {  // For each infostore
                            Attachment attach = null;
                            if (infostores.get(i).isJsonObject()) {  // 7.6 had json element for infostore
                                attach = getInfoAttach(infostores.get(i).getAsJsonObject(), request, cookie);
                            }
                            if (infostores.get(i).isJsonPrimitive()) { // Now sending just ID of element included in JSON Object
                                String id = infostores.get(i).getAsString();
                                boolean found = false;
                                // See if information in attachmentJson including version
                                JsonArray attachmentJson = data.get("attachments").getAsJsonArray();
                                for (int j = 0; j < attachmentJson.size(); j++) {
                                    if (attachmentJson.get(j).getAsJsonObject().has("id")) { // Search for identical ID
                                        if (attachmentJson.get(j).getAsJsonObject().get("id").getAsString().equals(id)) {
                                            found = true;  // We found it in the attachments, so pull the data now
                                            attach = getInfoAttach(attachmentJson.get(j).getAsJsonObject(), request, cookie);
                                            attachmentJson.remove(j);
                                        }
                                    }
                                }
                                if (!found) {  // Not found in attachment data, we only have the ID.  Pull
                                    attach = getInfoAttachIDOnly(id, request, cookie);
                                }
                            }
                            if (attach != null) {  // If found, add the attachment
                                attachments.add(attach);
                            }

                        }
                    }
                    data.remove("infostore_ids");  // We've added them to our attachments, now remove from the JSON
                }
            } catch (Exception ex2) {
                LOG.error("Problem extracting infostore items.", ex2);

            }
            // Search for inline images, convert to attachments
            data = getInlineImages(data, attachments, request, cookie, cid, userid);

            data = getNestedAttachments(data, attachments, request, cookie);

            data = getNestedMessages(data, attachments, request, cookie);

            // Search for attachments that reference a draft email, and aren't actually supplied.
            data = getDraftAttachments(data, attachments, request, cookie, userid, cid);

            data = getForwardedAttachments(data, attachments, request, cookie, userid, cid);

            addVcard(data, attachments, request, cookie, userid, cid);

            //Adding the sender's public key if at least one recipient is a non guest, remote user
            for (RecipKey key : recipients) {
                //The key belongs to an external if it does not belong to a context
                if (key.isRemoteKey() && !draft) {
                    //Adding the sender's public key as attachment to the mail
                    GuardKeyService keyService = Services.getService(GuardKeyService.class);
                    GuardKeys senderKeys = keyService.getKeys(userid, cid);

                    addPublicKey(attachments, KeyExportUtil.export(senderKeys.getPGPPublicKeyRing()));
                    break;
                }
            }

            if (!getAttachPGPFiles(data, attachments, request, cookie, userid, cid, attach_password)) {
                ServletUtils.sendNotAcceptable(response, "Unable to decode attachments");
                return;
            }

            boolean inline = ServletUtils.getBooleanParameter(request, "pgpinline");
            boolean signit = ServletUtils.getBooleanParameter(request, "pgpsign");
            String userAgent = request.getHeader("User-Agent");

            PGPMail mail = new PGPMail(draft, draftFolder, json, attachments, attach_password, recipients, guestmessage, pin, templId, cookie, userid, cid, sessionId, inline, signit, userAgent);
            try {
                String reply = new PGPSender().sendPGPMail(mail, request.getServerName(), request.getRemoteAddr());
                if (reply != null)
                {
                    if (JsonUtil.isJson(reply)) { // The reply will be either "OK" "Fail" or JSON Information
                        ServletUtils.sendOK(response, "application/json", reply);  // If JSON string returned, then return it
                    } else {
                        ServletUtils.sendOK(response, "text/html", reply);
                    }
                }
            } catch (OXException e) {
                LOG.error("Problem sending email ", e);
                ServletUtils.sendNotAcceptable(response, e.getDisplayMessage(new Locale(language)));
            }

        } else {// Our reply that the email was blank
            ServletUtils.sendHtmlOK(response, "No Email Data");
        }
    }

    private void sendInitialGuestPasswordMail(Api apiAccess, String fromName, String fromEmail, RecipKey recipientKey, int templateId, String guestMessage, String host, int userid, int cid) throws OXException, UnsupportedEncodingException {
        MailCreatorService mailCreatorService = Services.getService(MailCreatorService.class);
        JsonObject passwordEmail = mailCreatorService.getPasswordEmail(recipientKey.getEmail(), fromName, fromName, fromEmail, recipientKey.getNewGuestPass(), recipientKey.getLang(), templateId, guestMessage, host, userid, cid);
        boolean failed = Utilities.checkFail(apiAccess.sendMail(passwordEmail).toString());
        if (failed) {
            throw OXMailExceptionCodes.SEND_EMAIL_ERROR.create(recipientKey.getEmail());
        }
    }

    /**
     * Get the list of recipients from a json email, and get their keys
     *
     * @param json
     * @return
     * @throws Exception
     */
    public ArrayList<RecipKey> getUsers(JsonObject json, int senderId, int senderCID, String guestlanguage, String pin, EmailInfo info, boolean draft, String host) throws Exception {

        ArrayList<RecipKey> recipients = new ArrayList<RecipKey>();

        String from = json.get("from").getAsJsonArray().get(0).getAsJsonArray().get(1).getAsString();
        JsonElement fromNameElement = json.get("from").getAsJsonArray().get(0).getAsJsonArray().get(0);
        String fromName = fromNameElement != JsonNull.INSTANCE ? fromNameElement.getAsString() : null;
        JsonArray recips = new JsonArray();
        if (json.has("to")) {
            recips = json.get("to").getAsJsonArray();
            recipients = addUsers(recipients, recips, senderId, senderCID, from, fromName, RecipientType.TO, guestlanguage, pin, info, draft, host);
            if (recipients == null) {
                return (null);
            }
        }
        if (json.has("cc")) {
            recips = json.get("cc").getAsJsonArray();
            recipients = addUsers(recipients, recips, senderId, senderCID, from, fromName, RecipientType.CC, guestlanguage, pin, info, draft, host);
            if (recipients == null) {
                return (null);
            }
        }
        if (json.has("bcc")) {
            recips = json.get("bcc").getAsJsonArray();
            recipients = addUsers(recipients, recips, senderId, senderCID, from, fromName, RecipientType.BCC, guestlanguage, pin, info, draft, host);
        }
        return (recipients);

    }

    private ArrayList<RecipKey> addUsers(ArrayList<RecipKey> recipients, JsonArray recips, int senderId, int senderCID, String from, String fromName, RecipientType type, String guestlanguage, String pin, EmailInfo info, boolean draft, String host) throws Exception {
        for (int i = 0; i < recips.size(); i++) {
            JsonArray recip = recips.get(i).getAsJsonArray();
            String emailString = recip.get(1).getAsString();
            GuardRatifierService ratifierService = Services.getService(GuardRatifierService.class);
            ratifierService.validate(emailString); // Will throw error if invalid
            InternetAddress emailAddress = new InternetAddress(IDNUtil.aceEmail(emailString));
            String email = IDNUtil.decodeEmail(emailAddress.getAddress());

            //Lookup recipient key
            RecipKey key = Services.getService(RecipKeyService.class).getRecipKey(senderId, senderCID, email);

            //create key for recipient if this is not a draft
            if (key != null &&  key.isNewKey() && !draft) {
                key = Services.getService(AccountCreationService.class).createUserFor(key);
                if (key.isGuest()) {
                    key.setLang(guestlanguage);
                    if (pin != null && !pin.isEmpty()) {
                        Services.getService(GuardKeyService.class).updatePin(key.getUserid(), key.getCid(), pin);
                    }
                }
            }

            if (key != null) {
                if (key.getPubkey() == null) {
                    // Fail to create key
                    if (!draft) {
                        if (key.isPgp() == false) {
                            return (null);
                        }
                    }
                }
                key.setEmail(email);
                key.setType(type);
                try {
                    key.setName(recip.get(0).getAsString());
                } catch (Exception ex) {
                    key.setName("");
                    LOG.error("No key name set for " + email, ex);
                }
                recipients.add(key);
            } else {
                return (null);
                // No keys found or made
            }
        }
        return (recipients);
    }

    private boolean checkReply(JsonArray from, ArrayList<RecipKey> recips, EmailInfo info) {
        if (info == null) {
            return (false);
        }
        if (recips == null) {
            return (false);
        }
        if (!info.getFromEmail().toLowerCase().trim().equals(from.get(1).getAsString().toLowerCase().trim())) {
            return (false);
        }
        for (int i = 0; i < recips.size(); i++) {
            if (!info.getRecips().contains(recips.get(i).getEmail())) {
                if (!info.getSenderEmail().contains(recips.get(i).getEmail())) {
                    return (false);
                }
            }
        }
        return (true);
    }

    /**
     * Decode RSA encrypted passwords sent from UI
     *
     * @param epass
     * @return
     * @throws OXException
     */
    public String decodeRSApassword(String epass) throws OXException {
        GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.RSA);
        GuardMasterKeyService masterKeyService = Services.getService(GuardMasterKeyService.class);
        return cipherService.decrypt(epass, masterKeyService.getMasterKey().getPrivateKey());
    }

    /*
     * Get exp date of email
     */
    public static long getExpDate(JsonObject email) {
        try {
            JsonObject header = email.get("headers").getAsJsonObject();
            if (!header.has("X-OxGuard-Expiration")) {
                return (0);
            }
            int hours = header.get("X-OxGuard-Expiration").getAsInt();
            if (hours == 0) {
                return (0);
            }
            final long exphours = hours * 60L * 60L * 1000L;
            Date now = new Date();
            return (now.getTime() + exphours);
        } catch (Exception ex) {
            LOG.error("Problem getting email expiration date");
            return (0);
        }
    }

    /**
     * Pull infostore item for attachment if JSON from attachments is known
     *
     * @param jsonElement
     * @param request
     * @param ck
     * @return
     */
    public static Attachment getInfoAttach(JsonObject jsonElement, HttpServletRequest request, OxCookie ck) {
        try {
            String id = jsonElement.get("id").getAsString();
            String version = jsonElement.has("version") ? jsonElement.get("version").getAsString() : null;
            String folder = jsonElement.get("folder_id").getAsString();
            Api ap = new Api(ck, request);

            try (ApiResponse ent = ap.getInfoStoreAttach(id, folder, version)) {
                Attachment attach = new Attachment();
                String type = ent.getContentType().toString();
                attach.setType(type);
                int i = type.indexOf(":");
                if (i > 0) {
                    attach.setType(type.substring(i + 1).trim());
                    if (attach.getType().contains("application/pgp-signature")) {
                        // pgp attachment.  Type renamed by backend
                        JsonObject json = ap.getInfoItemJson(id, folder);
                        try {
                            JsonObject meta = json.get("data").getAsJsonObject().get("meta").getAsJsonObject();
                            if (meta.has("OrigMime")) {
                                attach.setType(meta.get("OrigMime").getAsString());
                            }
                        } catch (Exception ex) {
                            LOG.error("Error getting mime type of infostore pgp file", ex);
                        }
                    }
                }
                attach.setContent(ent.readContent());
                attach.setFilename(jsonElement.get("filename").getAsString());
                return (attach);
            }

        } catch (Exception ex) {
            LOG.error("Unable to get infostore attachment ", ex);
            return (null);
        }
    }

    /**
     * Get infostore attachment when infostoreID is the only thing known.
     * Retrieves the json data from the backend to get filename, etc. Latest version only
     *
     * @param id
     * @param request
     * @param ck
     * @return
     * @throws UnsupportedEncodingException
     */
    public Attachment getInfoAttachIDOnly(String id, HttpServletRequest request, OxCookie ck) throws UnsupportedEncodingException {
        try {
            Api ap = new Api(ck, request);
            String folder = getUnifiedFolder(id);
            JsonObject fileinfo = ap.getInfoItemJson(id, folder);
            if (fileinfo == null) {
                return (null);
            }
            JsonObject data = fileinfo.get("data").getAsJsonObject();

            try (ApiResponse ent = ap.getInfoStoreAttach(id, folder, null)) {
                Attachment attach = new Attachment();

                String type = ent.getContentType().toString();
                attach.setType(type);
                int i = type.indexOf(":");
                if (i > 0) {
                    attach.setType(type.substring(i + 1).trim());
                    if (attach.getType().contains("application/pgp-signature")) {
                        // pgp attachment.  Type renamed by backend
                        JsonObject json = ap.getInfoItemJson(id, folder);
                        try {
                            JsonObject meta = json.get("data").getAsJsonObject().get("meta").getAsJsonObject();
                            if (meta.has("OrigMime")) {
                                attach.setType(meta.get("OrigMime").getAsString());
                            }
                        } catch (Exception ex) {
                            LOG.error("Error getting mime type of infostore pgp file", ex);
                        }
                    }
                }
                attach.setContent(ent.readContent());
                attach.setFilename(data.get("filename").getAsString());
                return attach;
            }

        } catch (Exception ex) {
            LOG.error("Problem getting infostore attachment, ", ex);
            return (null);
        }
    }

    private String getUnifiedFolder(String emailid) {
        int i = emailid.lastIndexOf("/");
        return (emailid.substring(0, i));
    }

    /**
     * Get Attachments from msgref referenced object
     *
     * @param data
     * @param attachments
     * @param request
     * @param ck
     * @param userid
     * @param cid
     * @return
     */
    public JsonObject getForwardedAttachments(JsonObject data, ArrayList<Attachment> attachments, HttpServletRequest request, OxCookie ck, int userid, int cid) {
        try {
            String msgref = data.has("msgref") ? data.get("msgref").getAsString() : null;
            if (msgref == null) {
                return (data);
            }
            JsonArray attachmentsArray = data.get("attachments").getAsJsonArray();
            JsonArray newAttachments = new JsonArray();
            for (int i = 0; i < attachmentsArray.size(); i++) {
                boolean pulled = false;
                JsonObject attachment = attachmentsArray.get(i).getAsJsonObject();
                if (attachment.has("disp") && attachment.get("disp").getAsString().equals("attachment")) {
                    if (attachment.has("filename")) {
                        String filename = attachment.get("filename").getAsString();
                        boolean found = false;
                        // First, search through our already loaded attachments and make sure wasn't sent with post
                        for (Attachment att : attachments) {
                            if(filename.equals(att.getFilename())) {
                                found = true;
                            }
                        }
                        if (!found) {  // If not, pull from the msgref
                            String id = attachment.get("id").getAsString();
                            Api ap = new Api(ck, request);
                            int j = msgref.lastIndexOf("/");
                            String folder = msgref.substring(0, j);
                            String emailid = msgref.substring(j + 1).trim();
                            try (ApiResponse plainAttachment = ap.getPlainAttachment(emailid, id, folder)) {
                                if (plainAttachment != null) {
                                    Attachment attach = new Attachment();
                                    attach.setContent(plainAttachment.readContent());
                                    attach.setFilename(attachment.get("filename").getAsString());
                                    attach.setType(attachment.get("content_type").getAsString());
                                    attachments.add(attach);
                                    pulled = true;
                                } else {
                                    LOG.error("Problem pulling attachment for msgref " + msgref);
                                }
                            }
                        }
                    }
                }
                if (!pulled) {
                    newAttachments.add(attachment);
                }
            }
            // We need to remove the attachments from the JSON for the ones we pulled out, otherwise PGP Inline sends would double up, but we need it pulled for Guest/Mime
            data.remove("attachments");
            data.add("attachments", newAttachments);
        } catch (Exception e) {
            LOG.error("Problem getting msgref attachments ", e);
        }
        return (data);
    }

    /**
     * Get attachments from draft email if any. Looks for atmsgref for regular emails
     *
     * @param data
     * @param attachments
     * @param request
     * @return
     */
    public JsonObject getDraftAttachments(JsonObject data, ArrayList<Attachment> attachments, HttpServletRequest request, OxCookie ck, int userid, int cid) {
        try {
            JsonArray contents = data.get("attachments").getAsJsonArray();
            JsonArray newcontents = new JsonArray();
            for (int i = 0; i < contents.size(); i++) {
                boolean modified = false;
                JsonObject att = contents.get(i).getAsJsonObject();
                if (att.has("epass") && att.has("mail")) {
                    String jSessionId = cookie.getJSESSIONID();
                    String password = PGPUtils.decodeEPass(att.get("epass").getAsString(), jSessionId);
                    Attachment decattach = new Attachment();
                    String emailid = att.get("mail").getAsJsonObject().get("id").getAsString();
                    String folder = att.get("mail").getAsJsonObject().get("folder_id").getAsString();
                    String session = request.getParameter("session");
                    String userAgent = request.getHeader("User-Agent");
                    boolean pgpmime = true;
                    if (att.has("pgpFormat")) {
                        if (att.get("pgpFormat").getAsString().contains("inline")) {
                            pgpmime = false;
                        }
                    }
                    if (pgpmime) {
                        GuardEMailDecryptionService decryptionService = new GuardEMailDecryptionServiceImpl();
                        PGPDecryptionResult decryptionResult = decryptionService.decryptMimeAttachmentByName(ck, session, userAgent, folder, emailid, att.get("filename").getAsString(), userid, cid, password, ServletUtils.getClientIP(request));
                        decattach.setContent(IOUtils.toByteArray(decryptionResult.getInputStream()));
                    } else {
                        //Get a Stream to the attachment
                        Api api = new Api(cookie, session, userAgent);

                        try (ApiResponse plainAttachment = api.getPlainAttachment(emailid, att.get("id").getAsString(), folder)) {
                            Attachment attachment = new Attachment();
                            attachment.setContent(plainAttachment.readContent());
                            InputStream attachmentStream = new ByteArrayInputStream(attachment.getContent());

                            ByteArrayOutputStream output = new ByteArrayOutputStream();
                            PGPUtils.decryptFile(attachmentStream, output, userid, cid, password, false);
                            decattach.setContent(output.toByteArray());
                        }
                    }
                    decattach.setType(att.get("content_type").getAsString());
                    decattach.setFilename(att.get("filename").getAsString().replace(".pgp", ""));
                    attachments.add(decattach);
                    modified = true;
                } else {
                    if (att.has("atmsgref")) {
                        String attid = att.get("id").getAsString();
                        String item = att.get("atmsgref").getAsString();
                        int j = item.lastIndexOf("/");
                        String folder = item.substring(0, j);
                        String emailid = item.substring(j + 1).trim();
                        Api ap = new Api(ck, request);
                        try (ApiResponse plainAttachment = ap.getPlainAttachment(emailid, attid, folder)) {
                            Attachment attach = new Attachment();
                            attach.setContent(plainAttachment.readContent());
                            attach.setFilename(att.get("filename").getAsString());
                            attach.setType(att.get("content_type").getAsString());
                            attachments.add(attach);
                            modified = true;
                        }
                    }
                }
                if (!modified) {
                    newcontents.add(contents.get(i));
                }
            }
            data.remove("attachments");
            data.add("attachments", newcontents);
        } catch (Exception ex) {
            LOG.error("Problems getting attachments from draft email ", ex);
        }
        return (data);
    }

    /**
     * If the user forwards an item that was already encrypted (filestore item), then needs to be decoded before re-encrypting
     *
     * @param data
     * @param attachments
     * @param request
     * @param cookie
     * @param userid
     * @param cid
     * @param password
     * @return
     * @throws Exception
     */
    private boolean getAttachPGPFiles(JsonObject data, ArrayList<Attachment> attachments, HttpServletRequest request, OxCookie cookie, int userid, int cid, String password) throws Exception {
        boolean success = true;
        for (Attachment att : attachments) {
            if (att.getFilename() != null && att.getFilename().endsWith(".pgp")) {
                ByteArrayInputStream in = new ByteArrayInputStream(att.getContent());
                PGPResult result = PGPUtils.decryptFile(in, userid, cid, password);
                if (result.getError() == null) {
                    att.setContent(result.getDecoded());
                    String newFileName = att.getFilename().replace(".pgp", "");
                    att.setFilename(newFileName);
                } else {
                    success = false;
                }
            }
        }
        return (success);

    }

    /**
     * Check if vcard should be added. If so, add to attachments.
     * This is only supported 7.62 after July patch
     *
     * @param data
     * @param attachments
     * @param request
     * @param cookie
     * @param userid
     * @param cid
     */
    private void addVcard(JsonObject data, ArrayList<Attachment> attachments, HttpServletRequest request, OxCookie cookie, int userid, int cid) {
        if (data.has("vcard")) {
            if (data.get("vcard").getAsInt() == 1) {
                Api ap = new Api(cookie, request);
                byte[] vcard = ap.getVcard();
                if (vcard != null) {
                    String filename = "vcard";  // backup name is vcard
                    String vs = new String(vcard);
                    String[] lines = vs.split("\n");
                    for (String line : lines) {  // Parse the lines for the Full Name for the filename of the vcard
                        if (line.startsWith("FN:")) {
                            filename = line.replace("FN:", "").trim();
                        }
                    }
                    Attachment vcardAtt = new Attachment(); // Add to attachments
                    vcardAtt.setContent(vcard);
                    vcardAtt.setFilename(filename + ".vcf");
                    vcardAtt.setType("text/vcard");
                    attachments.add(vcardAtt);
                }
                data.remove("vcard");
                data.add("vcard", new JsonPrimitive(0));  // We need to remove the flag so the backend doesn't add it unencrypted.
            }
        }
    }

    private void addPublicKey(ArrayList<Attachment> attachments, String senderKey) {
        Attachment publicKeyAttachment = new Attachment();
        publicKeyAttachment.setContent(senderKey.getBytes());
        publicKeyAttachment.setFilename("public.asc");
        publicKeyAttachment.setType("application/pgp_keys");
        attachments.add(publicKeyAttachment);
    }

    public JsonObject getNestedMessages(JsonObject data, ArrayList<Attachment> attachments, HttpServletRequest request, OxCookie ck) {
        try {
            if (!data.has("nested_msgs")) {
                return (data);
            }
            JsonArray nested = data.get("nested_msgs").getAsJsonArray();
            for (int i = 0; i < nested.size(); i++) {
                JsonObject message = nested.get(i).getAsJsonObject();
                if (message.has("msgref")) {  // If message referenced, pull mime from backend
                    String message_ref = message.get("msgref").getAsString();
                    String subject = message.get("subject").getAsString();
                    if (subject.trim().length() < 1) {
                        subject = "eml" + i;
                    }
                    Api ap = new Api(ck, request);
                    String mime = ap.getMime(message_ref);
                    Attachment newatt = new Attachment();
                    newatt.setFilename(subject + ".eml");
                    newatt.setContent(mime.getBytes("UTF-8"));
                    newatt.setType("text/html");
                    attachments.add(newatt);
                    MimeParser mparse = new MimeParser(newatt.getContent());
                    mparse.ParseAttachments();
                    mparse.mergeAttachments(attachments);
                }
                // TODO convert netsted JSON message to mime
            }
            return (data);
        } catch (Exception ex) {
            LOG.error("Error getting nested messages", ex);
            return (data);
        }
    }

    public JsonObject getNestedAttachments(JsonObject data, ArrayList<Attachment> attachments, HttpServletRequest request, OxCookie ck) {
        try {
            JsonArray contents = data.get("attachments").getAsJsonArray();
            for (int i = 0; i < contents.size(); i++) {
                JsonObject content = contents.get(i).getAsJsonObject();
                if (content.has("disp")) {
                    if (content.get("disp").getAsString().equals("attachment")) {
                        if (content.has("content") && !content.get("content").isJsonNull()) {
                            if (content.get("content").getAsString().length() > 0) {
                                Attachment att = new Attachment();
                                att.setFilename(content.get("filename").getAsString());
                                att.setContent(content.get("content").getAsString().getBytes("UTF-8"));
                                att.setType(content.get("content_type").getAsString());
                                attachments.add(att);
                            }
                        } else {
                            if (content.has("msgref")) {
                                String msgref = content.get("msgref").getAsString();
                                Api ap = new Api(ck, request);
                                String mime = ap.getMime(msgref);
                                Attachment newatt = new Attachment();
                                newatt.setFilename(content.get("filename").getAsString() + ".eml");
                                newatt.setContent(mime.getBytes("UTF-8"));
                                newatt.setType("text/html");
                                attachments.add(newatt);
                                MimeParser mparse = new MimeParser(newatt.getContent());
                                mparse.ParseAttachments();
                                mparse.mergeAttachments(attachments);
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            LOG.error("Problem getting nested attachments", e);
        }
        return (data);
    }

    private JsonObject getInlineImages(JsonObject data, ArrayList<Attachment> attachments, HttpServletRequest request, OxCookie ck, int contextId, int userId) {
        try {
            JsonArray contents = data.get("attachments").getAsJsonArray();
            JsonArray newcontents = new JsonArray();
            String session = ServletUtils.getStringParameter(request, "session", true);
            String userAgent = request.getHeader("User-Agent");
            for (int i = 0; i < contents.size(); i++) {
                if (contents.get(i).getAsJsonObject().has("content") && contents.get(i).getAsJsonObject().get("content").isJsonNull()) {
                    newcontents.add(contents.get(i));
                } else {
                    JsonObject content = contents.get(i).getAsJsonObject();
                    if (content.has("content")) {
                        String cont = content.get("content").getAsString();
                        int start = cont.indexOf("<img");
                        boolean changed = false;
                        String newcont = new String(cont);
                        while ((start > 0) && (start < cont.length())) {
                            int end = cont.indexOf(">", start);
                            if (end < 0) {
                                break;
                            }
                            String image = cont.substring(start, end);
                            Pattern pattern = Pattern.compile("src=\"[^\"]*");
                            Matcher match = pattern.matcher(image);
                            if (match.find()) {
                                String url = image.substring(match.start() + 5, match.end());
                                if (url.length() > 1) {
                                    InlineImage inlineImage = null;
                                    for (InlineImageHandler inlineImageHandler : inlineImageHandlers) {
                                        //Getting an appropriated handler for receiving the inline image from the given URL
                                        if (inlineImageHandler.canHandle(url)) {
                                            inlineImage = inlineImageHandler.handle(url, ck, session, userAgent, userId, contextId, ServletUtils.getClientIP(request));
                                            if (inlineImage != null) {
                                                break;
                                            }
                                        }
                                    }

                                    if (inlineImage != null) {
                                        Attachment attach = new Attachment();
                                        attach.setFilename(inlineImage.getFileName());

                                        String type = inlineImage.getContentType().toString();
                                        attach.setType(type);
                                        if (type.contains(":")) {
                                            String newType = type.substring(type.indexOf(":") + 1).trim();
                                            attach.setType(newType);
                                        }

                                        attach.setContent(inlineImage.getContent());

                                        attach.setContent_id(CipherUtil.getUUID() + "@GUARD");
                                        attachments.add(attach);
                                        newcont = newcont.replace(url, "cid:" + attach.getContentId());
                                    }
                                    changed = true;
                                }
                            }
                            start = cont.indexOf("<img ", end);
                        }
                        if (changed) {
                            content.remove("content");
                            content.addProperty("content", newcont);
                        }
                    }
                    newcontents.add(content);
                }
            }
            data.remove("attachments");
            data.add("attachments", newcontents);
        } catch (Exception ex) {
            LOG.error("Problem retrieving inline images ", ex);
        }
        return (data);
    }

}
