/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.internal;

import java.util.List;
import java.util.Map;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.handler.ResponseHandler;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.storage.KeyTableStorage;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.user.GuardCapabilities;

public class GuardSettings {

    private static Logger LOG = LoggerFactory.getLogger(GuardCapabilities.class);

    public void setSettings(HttpServletRequest request, HttpServletResponse response, OxCookie cookie, int userId, int cid) throws Exception {
        JsonObject data = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
        Api ap = new Api(cookie, request);
        // Must either be logged in or sent with authorization code
        GuardAntiAbuseService antiabuse = Services.getService(GuardAntiAbuseService.class);
        if (!ap.verifyLogin()) {
            String auth = ServletUtils.getStringParameter(request, "auth", true);
            if (auth.length() > 20) {
                cookie.setJSESSIONID(cookie.getOxReaderID());	// This is used by guest reader
                UserData user = new UserData(auth, cookie);
                if ((user.getUserid() == -1) || (user.getEncr_password() == null)) {
                    ServletUtils.sendNotAcceptable(response, "Bad authorization");
                    antiabuse.addBad(cookie.getJSESSIONID());
                    ServletUtils.sendNotAcceptable(response, "Must be logged in");
                    return;
                }
            } else {
                ServletUtils.sendNotAcceptable(response, "Must be logged in");
                return;
            }
        }
        if (antiabuse.isBad(cookie.getJSESSIONID())) {
            ServletUtils.sendNotAcceptable(response, "Lockout");
            return;
        }
        JsonObject current = getCurrentSettings(userId, cid);
        JsonElement dataParam = data.get("data");
        if (dataParam == null) {
            throw GuardCoreExceptionCodes.PARAMETER_MISSING.create("data");
        }
        JsonObject settingsData = dataParam.getAsJsonObject();
        for (Map.Entry<String, JsonElement> entry : settingsData.entrySet()) {
            if (current.has(entry.getKey())) {
                current.remove(entry.getKey());
                current.add(entry.getKey(), entry.getValue());
            } else {
                current.add(entry.getKey(), entry.getValue());
            }
        }
        if (saveCurrentSettings(userId, cid, current.toString())) {
            JSONObject resp = new JSONObject();
            resp.put("data", "done");
            ServletUtils.sendHtmlOK(response, resp.toString());
        } else {
            ServletUtils.sendNotAcceptable(response, "Fail");
        }
    }

    private JsonObject getCurrentSettings(int id, int cid) throws Exception {
        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        List<GuardKeys> keysForUser = ogKeyTableStorage.getKeysForUser(id, cid);
        if (keysForUser.size() > 0) {
            GuardKeys key = keysForUser.get(0);
            return key.getSettings();
        }
        return new JsonObject();
    }

    private boolean saveCurrentSettings(int id, int cid, String settings) {
        try {
            KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
            ogKeyTableStorage.updateSettingsForUser(id, cid, settings);
        } catch (Exception e) {
            LOG.error("Error updating settings ", e);
            return false;
        }
        return true;
    }
}
