/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.guest;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.mail.Part;
import javax.mail.Session;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.ParameterList;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URLEncodedUtils;

import com.openexchange.exception.OXException;
import com.openexchange.guard.exceptions.GuardAuthExceptionCodes;
import com.openexchange.guard.internal.UserData;
import com.openexchange.guard.mail.InlineImageHandler;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.InlineImage;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.pgp.PGPResult;
import com.openexchange.guard.pgp.PGPUtils;
import com.openexchange.guard.pgp.decryption.PGPMimeAttachmentExtractor;
import com.openexchange.guard.pgp.exceptions.GuardPGPExceptionCodes;
import com.openexchange.guard.storage.Storage;

/**
 * {@link GuestInlineImageHandler} handles forwarded inline images within OX Guard
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.4.2
 */
public class GuestInlineImageHandler implements InlineImageHandler {

    /* (non-Javadoc)
     * @see com.openexchange.guard.mail.InlineImageHandler#canHandle(java.lang.String)
     */
    @Override
    public boolean canHandle(String url) throws Exception {
        return url.contains("guest?action=getattachcid");
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.mail.InlineImageHandler#handle(java.lang.String, com.openexchange.guard.oxapi.OxCookie, java.lang.String, java.lang.String)
     */
    @Override
    public InlineImage handle(String url, OxCookie cookie, String session, String userAgent, int userId, int contextId, String remoteIp) throws Exception {

    	cookie.setJSESSIONID(cookie.getOxReaderID());

        //Parsing parameters from the URL
        List<NameValuePair> parse = URLEncodedUtils.parse(StringEscapeUtils.unescapeHtml(url), StandardCharsets.UTF_8);
        Map<String,String> values = new HashMap<String,String>();
        for(NameValuePair pair : parse) {
            String name = pair.getName();
            String value = pair.getValue();
            if(!values.containsKey(name)) {
                values.put(name, value);
            }
        }

        String emailId = values.get("email");
        String content_id = values.get("content_id");
        String authToken = values.get("auth");
        String password = null;
        UserData guestData = new UserData(authToken, cookie);
        if (guestData.getUserid() > 0) {
        	password = guestData.getEncr_password();
        } else {
        	throw GuardPGPExceptionCodes.BAD_PASSWORD_ERROR.create();
        }

        Storage storage = Services.getService(Storage.class);
        InputStream in = storage.readObj(guestData.getUserid(), guestData.getCid(), emailId);
        if (in != null) {
            PGPResult result = null;
            try {
                result = PGPUtils.decryptFile(in, guestData.getUserid(), guestData.getCid(), guestData.getEncr_password());

            } catch (OXException ex) {
                if (GuardAuthExceptionCodes.BAD_PASSWORD.equals(ex)) {
                    in.close();
                    throw GuardPGPExceptionCodes.BAD_PASSWORD_ERROR.create();
                } else {
                	throw ex;
                }
            }

	        MimeMessage decryptedMimeMessage = new MimeMessage(Session.getDefaultInstance(new Properties()), new ByteArrayInputStream(result.getDecoded()));

	        //Extracting the requested cid attachment from the MIME message
	        Part decryptedCidAttachment = new PGPMimeAttachmentExtractor().getAttachmentFromMessageID(decryptedMimeMessage, content_id);


	        String contentType = decryptedCidAttachment.getContentType();
	        String fileName = null;
	        if(contentType.contains(";")) {
	            String contentTypeParameters = contentType.substring(contentType.indexOf(";"));
	            ParameterList parameterList = new ParameterList(contentTypeParameters);
	            fileName = parameterList.get("name");
	        }

	        byte[] imageData = IOUtils.toByteArray(decryptedCidAttachment.getInputStream());
	        return new InlineImage(fileName, contentType, imageData);
        }
        return null;
    }
}
