/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard;

import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.mail.internet.ParameterList;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URLEncodedUtils;
import com.openexchange.guard.mail.InlineImageHandler;
import com.openexchange.guard.oxapi.InlineImage;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.pgp.PGPUtils;
import com.openexchange.guard.pgp.decryption.GuardEMailDecryptionService;
import com.openexchange.guard.pgp.decryption.PGPDecryptionResult;
import com.openexchange.guard.pgp.decryption.impl.GuardEMailDecryptionServiceImpl;

/**
 * {@link GuardInlineImageHandler} handles forwarded inline images within OX Guard
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class GuardInlineImageHandler implements InlineImageHandler {

    /* (non-Javadoc)
     * @see com.openexchange.guard.mail.InlineImageHandler#canHandle(java.lang.String)
     */
    @Override
    public boolean canHandle(String url) throws Exception {
        return url.contains("inline?action=getattachcid");
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.mail.InlineImageHandler#handle(java.lang.String, com.openexchange.guard.oxapi.OxCookie, java.lang.String, java.lang.String)
     */
    @Override
    public InlineImage handle(String url, OxCookie cookie, String session, String userAgent, int userId, int contextId, String remoteIp) throws Exception {

        //Parsing parameters from the URL
        List<NameValuePair> parse = URLEncodedUtils.parse(StringEscapeUtils.unescapeHtml(url), StandardCharsets.UTF_8);
        Map<String,String> values = new HashMap<String,String>();
        for(NameValuePair pair : parse) {
            String name = pair.getName();
            String value = pair.getValue();
            if(!values.containsKey(name)) {
                values.put(name, value);
            }
        }
        String emailFolder = values.get("folder");
        String emailId = values.get("emailid");
        String attachmentId = values.get("content_id");
        String epassword = values.get("epassword");
        String jSessionID = cookie.getJSESSIONID();
        String password = null;
        if(jSessionID != null && !jSessionID.isEmpty()) {
            password = PGPUtils.decodeEPass(epassword,  jSessionID);
        }

        //Getting the inline image
        GuardEMailDecryptionService emailDecryptionService = new GuardEMailDecryptionServiceImpl();
        PGPDecryptionResult decryptionResult =
            emailDecryptionService.decryptMimeAttachmentById(cookie, session, userAgent, emailFolder, emailId, attachmentId, userId, contextId, password, remoteIp);
        //Parsing content-type
        String contentType = decryptionResult.getContentType();
        String fileName = null;
        if(contentType.contains(";")) {
            String contentTypeParameters = contentType.substring(contentType.indexOf(";"));
            ParameterList parameterList = new ParameterList(contentTypeParameters);
            fileName = parameterList.get("name");
        }

        byte[] imageData = IOUtils.toByteArray(decryptionResult.getInputStream());
        return new InlineImage(fileName, contentType, imageData);
    }
}
