/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.user;

import java.util.Objects;

/**
 * {@link UserIdentity} represents a generic PGP user.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.2
 */
public class UserIdentity {

    private final String    identity;
    private final char[]    password;
    private GuardUser       guardUser;

    /**
     * Initializes a new {@link UserIdentity}.
     *
     * @param identity The identity
     * @param password The password
     */
    public UserIdentity(String identity, char[] password) {
        this.identity = identity;
        this.password = password;
    }

    /**
     * Initializes a new {@link UserIdentity}.
     *
     * @param identity The identity
     * @param password The password
     */
    public UserIdentity(String identity, String password) {
        this.identity = identity;
        this.password = password != null ? password.toCharArray() : null;
    }

    /**
     * Initializes a new {@link UserIdentity}.
     *
     * @param guardUser A {@link GuardUser} to create the {@link UserIdentity} from.
     * @param password The password
     */
    public UserIdentity(GuardUser guardUser ,char[] password) {
        guardUser = Objects.requireNonNull(guardUser, "guardUser mus not be null");
        this.identity = guardUser.getEmail();
        this.password = password;
    }

    /**
     * Initializes a new {@link UserIdentity}.
     *
     * @param identity the identity
     */
    public UserIdentity(String identity) {
        this.identity = identity;
        this.password = new char[] {};
    }

    /**
     * Gets the identity
     *
     * @return The identity
     */
    public String getIdentity() {
        return identity;
    }

    /**
     * Gets the password
     *
     * @return The password
     */
    public char[] getPassword() {
        return password;
    }

    /**
     * Gets the related OX user for this PGP user entity, or null if this entity is not related to an OX account.
     *
     * @return The OX Guard user, or null, if this entity is not related to an OX account.
     */
    public GuardUser getGuardUser() {
        return this.guardUser;
    }

    /**
     * Sets the OX user related to this PGP entity.
     *
     * @param guardUser the related OX User
     */
    public void setGuardUser(GuardUser guardUser) {
        this.guardUser = guardUser;
    }

    /**
     * Returns if this PGP entity is related to an OX account or not.
     *
     * @return True if this PGP entity is related to an OX account, false otherwise.
     */
    public boolean isGuardUser() {
        return this.guardUser != null;
    }
}
