/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgpcore.services;

import java.util.List;
import javax.mail.Address;
import javax.mail.MessagingException;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.pgpcore.services.exceptions.PGPCoreServicesExceptionCodes;
import com.openexchange.pgp.core.PGPSignatureVerificationResult;

/**
 * {@link FromHeaderVerifier} - Helper class to verify FROM Headers of MimeMessages
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.10.2
 */
public class FromHeaderVerifier {

    private static Logger LOG = LoggerFactory.getLogger(FromHeaderVerifier.class);

    private static boolean issuerAllowed(Address[] fromAddresses, PGPSignatureVerificationResult result) throws AddressException {
        for(String pgpUserId : result.getIssuerUserIds()) {
            for(Address address : fromAddresses) {
                final InternetAddress fromAddress = (InternetAddress) address;
                // Remove username info from userId, may contain illegal internet address characters, etc
                final String id = (pgpUserId.contains("<") ? pgpUserId.substring(pgpUserId.indexOf("<")) : pgpUserId);
                final InternetAddress pgpUserIdAddress  = new InternetAddress(id);
                if (fromAddress.getAddress() != null && pgpUserIdAddress.getAddress() != null) {
                    if(fromAddress.getAddress().toLowerCase().trim().equals(pgpUserIdAddress.getAddress().toLowerCase().trim())) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /**
     * Verifies that the issuer of the PGPSignatureVerificationResults are in the actual FROM-Header for the given MimeMessage.
     * This sets the verification results to not verified if the issuer is not part of the message's FROM Header
     *
     * @param mimeMessage The MimeMessage
     * @param verificationResults The verification results
     * @throws OXException due Address parsing error
     */
    public static void verify(MimeMessage mimeMessage, List<PGPSignatureVerificationResult> verificationResults) throws OXException {
        try {
            Address[] fromAddresses = mimeMessage.getFrom();
            if (fromAddresses== null) {
                //We do not found any from headers
                for (PGPSignatureVerificationResult result : verificationResults) {
                    result.setVerified(false);
                    result.setError("Unable to verify signature due missing FROM header in MIME message.");
                    LOG.error("Unable to verify signature due missing FROM header in MIME message.");
                }
            } else {
                for (PGPSignatureVerificationResult result : verificationResults) {
                    if(result.isVerified()){
                        try {
                            boolean issuerInFromHeader =  issuerAllowed(fromAddresses, result);
                            if(!issuerInFromHeader) {
                                result.setVerified(false);
                                result.setError("The issuer of the signature was not found in the FROM header for the MIME message.");
                                LOG.debug("The issuer of the signature was not found in the FROM header for the MIME message.");
                            }
                        } catch (MessagingException e) { // Problem parsing/verifying email addresses.  Fail signature and add error message
                            result.setVerified(false);
                            result.setError(e.getMessage());
                        }
                    }
                }
            }
        } catch (MessagingException e) {
            throw PGPCoreServicesExceptionCodes.SIGNING_ERROR.create(e, e.getMessage());
        }
    }


}
