
package com.openexchange.guard.pgpcore.services.impl.osgi;

import com.openexchange.exception.OXException;
import com.openexchange.pgp.core.PGPKeyRetrievalStrategy;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.storage.OGPGPKeysStorage;
import com.openexchange.guard.pgpcore.services.GuardKeyRingRetrievalStrategy;
import com.openexchange.guard.pgpcore.services.PGPCryptoService;
import com.openexchange.guard.pgpcore.services.PGPSigningService;
import com.openexchange.guard.pgpcore.services.TokenAuthenticationService;
import com.openexchange.guard.pgpcore.services.impl.CompositeGuardKeyRingRetrievalStrategy;
import com.openexchange.guard.pgpcore.services.impl.DBKeyRetrievalStrategy;
import com.openexchange.guard.pgpcore.services.impl.EmailGuardKeyRingRetrievalStrategy;
import com.openexchange.guard.pgpcore.services.impl.PGPCryptoServiceImpl;
import com.openexchange.guard.pgpcore.services.impl.PGPSigningServiceImpl;
import com.openexchange.guard.pgpcore.services.impl.TokenAuthenticationServiceImpl;
import com.openexchange.guard.pgpcore.services.impl.UserIDGuardKeyRingRetrievalStrategy;
import com.openexchange.guard.session.GuardSessionService;
import com.openexchange.guard.user.GuardUserService;
import com.openexchange.osgi.HousekeepingActivator;

public class PGPCoreServicesImplActivator extends HousekeepingActivator {

    /**
     * Internal method to get an appropriated cipher service
     *
     * @return a cipher service
     * @throws OXException
     */
    private GuardCipherService getCipherService() throws OXException {
        GuardCipherFactoryService cipherFactory = getService(GuardCipherFactoryService.class);
        return cipherFactory.getCipherService(GuardCipherAlgorithm.AES_GCM);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.DeferredActivator#getNeededServices()
     */
    @Override
    protected Class<?>[] getNeededServices() {
        return new Class<?>[] { GuardCipherFactoryService.class, GuardKeyService.class, GuardSessionService.class, GuardUserService.class, OGPGPKeysStorage.class};
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void startBundle() throws Exception {
        org.slf4j.LoggerFactory.getLogger(PGPCoreServicesImplActivator.class).info("Starting bundle: {}", context.getBundle().getSymbolicName());

        //Key retrieval strategies
        GuardKeyRingRetrievalStrategy guardKeyRingRetrievalStrategy = new CompositeGuardKeyRingRetrievalStrategy(new EmailGuardKeyRingRetrievalStrategy(getService(GuardKeyService.class)), new UserIDGuardKeyRingRetrievalStrategy(getService(GuardKeyService.class)));
        PGPKeyRetrievalStrategy pgpKeyRetrievalStrategy = new DBKeyRetrievalStrategy(getService(GuardKeyService.class));
        //Register the services
        registerService(TokenAuthenticationService.class,
            new TokenAuthenticationServiceImpl(guardKeyRingRetrievalStrategy,
                                               getService(GuardSessionService.class),
                                               getCipherService(),
                                               getService(GuardUserService.class)));
        registerService(PGPCryptoService.class,
            new PGPCryptoServiceImpl(guardKeyRingRetrievalStrategy,
                getService(GuardKeyService.class),
                getService(OGPGPKeysStorage.class)));
        registerService(PGPSigningService.class, new PGPSigningServiceImpl(guardKeyRingRetrievalStrategy, pgpKeyRetrievalStrategy));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void stopBundle() throws Exception {
        org.slf4j.LoggerFactory.getLogger(PGPCoreServicesImplActivator.class).info("Stopping bundle: {}", this.context.getBundle().getSymbolicName());
        unregisterService(TokenAuthenticationService.class);
        unregisterService(PGPCryptoService.class);
        unregisterService(PGPSigningService.class);
        super.stopBundle();
    }
}
