/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.oxapi.pooling.internal;

import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;
import org.apache.http.HttpHost;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.conn.routing.HttpRoute;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.pool.ConnPoolControl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.oxapi.osgi.Services;

/**
 * Provides a pool of HTTP connections
 */
class ConnectionPool {

    private final Logger logger = LoggerFactory.getLogger(ConnectionPool.class);
    private CloseableHttpClient httpClient;
    private PoolingHttpClientConnectionManager cm;
    private boolean resetting = false;
    private IdleConnectionMonitor monitor;
    private final CookieHandlingMode cookieHandlingMode;
    private final String name;

    /**
     * The cookie handling mode for a connection pool
     */
    public enum CookieHandlingMode {
        /**
         * Default cookie handling,
         */
        CookieHandlingEnabled,

        /**
         * Disable cookie handling
         */
        CookieHandligDissabled
    };

    /**
     * Constructor
     *
     * @param name the name of the connection pool
     * @param cookieHandlingMode the cookie handling mode to use for the pool
     * @throws OXException
     */
    ConnectionPool(String name, CookieHandlingMode cookieHandlingMode) throws OXException {
        this.name = name;
        this.cookieHandlingMode = cookieHandlingMode;
        initialize();
    }

    /**
     * Internal method to initialize the connection pool
     *
     * @throws OXException
     */
    private void initialize() throws OXException {
        GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);

        cm = new PoolingHttpClientConnectionManager(10, TimeUnit.MINUTES);
        cm.setMaxTotal(guardConfigService.getIntProperty(GuardProperty.maxTotalConnections));
        cm.setDefaultMaxPerRoute(guardConfigService.getIntProperty(GuardProperty.maxHttpConnections));

        HttpHost ox = new HttpHost(guardConfigService.getProperty(GuardProperty.restApiHostname), guardConfigService.getIntProperty(GuardProperty.oxBackendPort));
        cm.setMaxPerRoute(new HttpRoute(ox), guardConfigService.getIntProperty(GuardProperty.maxHttpConnections));
        httpClient = buildHttpClient(cookieHandlingMode);
        monitor = new IdleConnectionMonitor(cm, cm, name);
        monitor.start();
        try {
            monitor.join(1000);
        } catch (InterruptedException e) {
            logger.info("HTTP Monitor thread shutdown");
        }
    }

    /**
     * Internal method to build the HttpClient-based connection pool
     *
     * @param cookieHandlingMode the cookie handling mode
     * @return A HTTClient representing the connection pool
     * @throws OXException
     */
    private CloseableHttpClient buildHttpClient(CookieHandlingMode cookieHandlingMode) throws OXException {
        GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);

        int connectionTimeout = guardConfigService.getIntProperty(GuardProperty.connectionTimeout);
        RequestConfig requestConfig =
            RequestConfig.custom().setStaleConnectionCheckEnabled(true).setConnectTimeout(connectionTimeout).setSocketTimeout(connectionTimeout).build();

        return cookieHandlingMode == CookieHandlingMode.CookieHandligDissabled ?
            HttpClients.custom().setDefaultRequestConfig(requestConfig).disableCookieManagement().disableAutomaticRetries().setConnectionManager(cm).build() :
            HttpClients.custom().setDefaultRequestConfig(requestConfig).disableAutomaticRetries().setConnectionManager(cm).build();
    }

    /**
     * Closes the connection pool
     */
    public void close() {
        monitor.shutdown();
        cm.shutdown();
    }

    /**
     * Resets the connection pool
     */
    public void reset() {
        try {
            if (resetting) {
                return;
            }
            logger.debug("Resetting connection pool");
            resetting = true;
            close();
            initialize();
            Timer resetDelay = new Timer();
            resetDelay.schedule(new TimerTask() {

                @Override
                public void run() {
                    resetting = false;
                    logger.debug("end reset delay");
                }
            }, 30000);
        } catch (Exception ex) {
            logger.error("Error resetting connection pool ", ex);
        }
    }

    /**
     * Gets a connection from the pool
     *
     * @return
     */
    public HttpClient getHttpClient() {
        return this.httpClient;
    }

    /**
     * Gets the runtime control of the connection pool
     *
     * @return The runtime control of the connection pool
     */
    public ConnPoolControl<HttpRoute> getPoolControl() {
        ConnPoolControl<HttpRoute> x;
        return cm;
    }
}
