/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.mime.services;

import java.io.UnsupportedEncodingException;
import javax.mail.Address;
import javax.mail.MessagingException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import javax.mail.Message.RecipientType;

/**
 * {@link GuardParsedMimeMessage}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class GuardParsedMimeMessage {

    private MimeMessage msg;
    private JSONArray recipients;
    private JSONObject settings;
    private final MimeParser mime;
    private final int senderUserId;
    private final int senderContextId;
    private final String fromName;
    private final boolean draft;
    private final String host;
    private String senderIp;


    public GuardParsedMimeMessage (String msg, String json, int userid, int cid, String fromName, String host) throws UnsupportedEncodingException, MessagingException, JSONException {
        this (msg, json, userid, cid, fromName, false, host);
    }

    public GuardParsedMimeMessage (String msg, JSONObject json, int userid, int cid, String fromName, boolean draft, String host) throws UnsupportedEncodingException, MessagingException, JSONException {
        this.mime = new MimeParser(msg.getBytes("UTF-8"));
        this.msg = this.mime.getMessage();
        if (json != null) {
            if (json.has("security")) {
                this.settings = json.getJSONObject("security");
            } else {
                this.settings = new JSONObject();
            }
            if (json.has("recipients")) {
                this.recipients = json.getJSONArray("recipients");
            } else {
                this.recipients = new JSONArray();
            }
            if (json.has("senderIp")) {
                this.senderIp = json.getString("senderIp");
            } else {
                this.senderIp = null;
            }
        }
        this.senderUserId = userid;
        this.senderContextId = cid;
        this.fromName = fromName;
        this.draft = draft;
        this.host = host;
    }

    public GuardParsedMimeMessage (String msg, String json, int userid, int cid, String fromName, boolean draft, String host) throws UnsupportedEncodingException, MessagingException, JSONException {
        this(msg, json != null ? new JSONObject(json) : null, userid, cid, fromName, draft, host);
    }

    public void setMessage(MimeMessage msg) {
        this.msg = msg;
    }

    public MimeMessage getMessage() {
        return msg;
    }

    public MimeParser getParser () {
        return mime;
    }

    public JSONArray getRecipients () {
        return recipients;
    }

    public boolean isSign() throws JSONException {
        if (!settings.has("sign")) {
            return false;
        }
        return settings.getBoolean("sign");
    }

    public boolean isInline() throws JSONException {
        if (!settings.has("inline")) {
            return false;
        }
        return settings.getBoolean("inline");
    }

    public boolean isEncrypt() throws JSONException {
        if (!settings.has("encrypt")) {
            return false;
        }
        return settings.getBoolean("encrypt");
    }

    /**
     * Pulls Guest greeting message from sent JSON
     * @return
     */
    public String getGuestMessage() {
        if (!settings.has("guest_message")) {
            return null;
        }
        try {
            return settings.getString("guest_message");
        } catch (JSONException e) {
            return null;
        }
    }

    /**
     * Pulls selected Guest language from JSON
     * @return
     * @throws JSONException
     */
    public String getGuestLanguage() throws JSONException {
        if (!settings.has("guest_language")) {
            return null;
        }
        return settings.getString("guest_language");
    }

    /**
     * Pull guest pin from JSON
     * @return
     * @throws JSONException
     */
    public String getPIN() throws JSONException {
        if (!settings.has("pin")) {
            return null;
        }
        return settings.getString("pin");
    }

    public int getSenderUserId() {
        return this.senderUserId;
    }

    public int getSenderContextId () {
        return this.senderContextId;
    }

    /**
     * Parse the from address from message header
     * @return
     */
    public InternetAddress getFromAddress () {
        try {
            Address[] from = this.msg.getFrom();
            if (from.length > 0) {
                return ((InternetAddress) from[0]);
            } else {
                return null;
            }
        } catch (MessagingException e) {
            return null;
        }
    }

    /**
     * Gets the senderName
     *
     * @return The senderName
     */
    public String getSenderName() {
        return fromName;
    }

    /**
     * Parse the subject from message header
     * @return
     */
    public String getSubject () {
        try {
            return this.msg.getHeader("Subject", null);
        } catch (MessagingException e) {
            return null;
        }
    }

    /**
     * Returns if message is a draft message
     * @return
     */
    public boolean isDraft () {
        return draft;
    }

    /**
     * Returns the domain that the client was using
     * @return
     */
    public String getHost () {
        return host;
    }

    /**
     * TO find the type of recipient, parse the mime header for the email address
     * If not in Mime, then returns bcc
     * @param email
     * @return
     */
    public RecipientType getRecipType (String email) {
        try {
            String [] toHeaders = this.msg.getHeader("To");
            for (String to : toHeaders) {
                if (to.toLowerCase().contains(email.toLowerCase())) {
                    return RecipientType.TO;
                }
            }
            String [] ccHeaders = this.msg.getHeader("Cc");
            for (String cc : ccHeaders) {
                if (cc.toLowerCase().contains(email.toLowerCase())) {
                    return RecipientType.CC;
                }
            }
            return RecipientType.BCC;
        } catch (MessagingException e) {
            return null;
        }

    }

    public String getSenderIP() {
        return senderIp;
    }
}
