/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.mime.encryptor.pgp.pgpmime;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import javax.mail.BodyPart;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Part;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import com.openexchange.exception.OXException;
import com.openexchange.guard.mime.encryptor.Decryptor;
import com.openexchange.guard.mime.encryptor.exceptions.MimeEncryptorExceptionCodes;
import com.openexchange.guard.mime.services.MimeParser;
import com.openexchange.guard.pgp.signing.impl.SignatureResults;
import com.openexchange.guard.pgpcore.services.FromHeaderVerifier;
import com.openexchange.guard.pgpcore.services.PGPCryptoService;
import com.openexchange.guard.user.UserIdentity;
import com.openexchange.pgp.core.PGPSignatureVerificationResult;

/**
 * {@link PGPMimeDecryptor}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.8.0
 */
public class PGPMimeDecryptor extends Decryptor {

    private final PGPCryptoService pgpCryptoService;
    private final UserIdentity userIdentity;

    /**
     * Initializes a new {@link PGPMimeDecryptor}.
     *
     * @param pgpCryptoService The {@link PGPCryptoService} to use for decrypting the content.
     * @param identity The identity of the user to decrypt the content for.
     */
    public PGPMimeDecryptor(PGPCryptoService pgpCryptoService, UserIdentity identity) {
        this.pgpCryptoService = pgpCryptoService;
        this.userIdentity = identity;
    }

    /**
     * Internal method to get an {@link InputStream} from the given message part.
     *
     * @param message The message part to get an {@link InputStream} from.
     * @return An {@link InputStream} for the given {@link Part}
     * @throws IOException
     * @throws MessagingException
     */
    private InputStream getInputStreamFrom(Part message) throws IOException, MessagingException {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        message.writeTo(outputStream);
        return new ByteArrayInputStream(outputStream.toByteArray());
    }

    /**
     * Internal method to return the encrypted body of a given message.
     *
     * @param multipartMessage The message to get the encrypted body from.
     * @return The body part containing the encrypted message.
     * @throws MessagingException
     */
    private BodyPart getPGPContentBody(Multipart multipartMessage) throws MessagingException {
        for (int i = 0; i < multipartMessage.getCount(); i++) {
            BodyPart part = multipartMessage.getBodyPart(i);
            if (part.getContentType().contains("application/octet-stream")) {
                return part;
            }
        }
        return null;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.mime.encryptor.Decryptor#doDecrypt(javax.mail.internet.MimeMessage)
     */
    @Override
    public MimeMessage doDecrypt(MimeMessage message) throws OXException {
        try {
            Object content = message.getContent();
            if (content instanceof MimeMultipart) {
                BodyPart pgpContentBody = getPGPContentBody((Multipart) content);
                //InputStream pgpContentStream = getInputStreamFrom(pgpContentBody);
                InputStream pgpContentStream = pgpContentBody.getInputStream();
                ByteArrayOutputStream decrypted = new ByteArrayOutputStream();
                List<PGPSignatureVerificationResult> decrypt = pgpCryptoService.decrypt(pgpContentStream, decrypted, userIdentity);
                FromHeaderVerifier.verify(message, decrypt);
                MimeMessage decodedMsg = new MimeParser(decrypted.toByteArray()).getMessage();
                checkMissingHeaders (message, decodedMsg);
                decodedMsg.removeHeader("X-Guard-Signature-Result");
                String[] signatures = SignatureResults.toHeaders(decrypt);
                for (String sig : signatures) {
                    decodedMsg.addHeader("X-Guard-Signature-Result", sig);
                }
                if (message.getHeader("X-Security") != null) {
                    decodedMsg.addHeader("X-Security", message.getHeader("X-Security")[0]);
                }
                return decodedMsg;
            } else {
                //TODO: NOT PGP/MIME EXCEPTION
                return null;
            }
        } catch (IOException | MessagingException e) {
            throw MimeEncryptorExceptionCodes.UNKNOWN_EXCEPTION.create(e.getMessage());
        }
    }

    /**
     * Check for missing required headers in the encyrpted body.  If missing, add from original plaintext
     * @param orig
     * @param decrypted
     */
    private void checkMissingHeaders (MimeMessage orig, MimeMessage decrypted) {
        checkMissingHeader (orig, decrypted, "Date");
        checkMissingHeader (orig, decrypted, "Subject");
        checkMissingHeader (orig, decrypted, "From");
        checkMissingHeader (orig, decrypted, "To");
        checkMissingHeader (orig, decrypted, "CC");
        checkMissingHeader (orig, decrypted, "Return-Path");
    }

    /**
     * Check for missing headers in decrypted email.  If not present, then add from original
     * @param orig
     * @param decrypted
     */
    private void checkMissingHeader (MimeMessage orig, MimeMessage decrypted, String header) {
            try {
                if (decrypted.getHeader(header) == null) {
                    if (orig.getHeader(header) != null && orig.getHeader(header).length > 0) {
                        String[] headers = orig.getHeader(header);
                        for (String data : headers) {
                            decrypted.addHeader(header, data);
                        }
                    }
                }
            } catch (MessagingException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
    }
}
