/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.mime.encryptor.pgp.pgpmime;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import javax.mail.BodyPart;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import com.openexchange.exception.OXException;
import com.openexchange.guard.file.FileTyper;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.mime.encryptor.exceptions.MimeEncryptorExceptionCodes;
import com.openexchange.guard.mime.encryptor.osgi.Services;
import com.openexchange.guard.mime.services.GuardParsedMimeMessage;
import com.openexchange.guard.pgpcore.services.PGPCryptoService;
import com.openexchange.guard.user.UserIdentity;

/**
 * {@link PGPEncryptedAttachmentManager}
 *
 * Handle PGP attachments in an email.  Need to be decoded with the senders key
 * before sending encrypted with the recipient keys
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class PGPEncryptedAttachmentManager {

    private GuardParsedMimeMessage msg;
    private MimeMessage message;
    private boolean encrypted;
    private String password;

    public PGPEncryptedAttachmentManager (GuardParsedMimeMessage msg, String password) throws OXException {
        this.msg = msg;
        this.message = msg.getMessage();
        this.password = password;
        encrypted = false;
        parse();
    }

    /**
     * Return true if has an encrypted attachment
     * @return
     */
    public boolean hasEncryptedAttachment () {
        return encrypted;
    }

    public MimeMessage getDecoded () {
        return message;
    }

    /**
     * Parse the message
     * @throws OXException
     */
    private void parse() throws OXException {
        Object content;
        try {
            content = message.getContent();
            if (content instanceof Multipart || content instanceof MimeMultipart) {
                final Multipart mp = (Multipart) content;
                parseMultipart(mp);
            }
            // If not multipart, will not have encrypted attachments
        } catch (IOException | MessagingException e) {
            throw MimeEncryptorExceptionCodes.UNKNOWN_EXCEPTION.create(e.getMessage());
        }
    }

    /**
     * Parse the multiparts and decrypt any attachments that are pgp
     * @param mp
     * @throws OXException
     */
    private void parseMultipart(final Multipart mp) throws OXException {
        try {
            final int count = mp.getCount();
            for (int i = 0; i < count; i++) {
                final BodyPart bp = mp.getBodyPart(i);
                final Object content = bp.getContent();
                if (content instanceof String) {  // Check if this string is a text attachment, or just the message body
                    // Nothing to do here, not an encrypted attachment
                } else if (content instanceof InputStream) {
                    boolean isEncrypted = false;
                    if (bp.getContentType() != null &&bp.getContentType().contains("application/pgp")) isEncrypted = true;
                    if (bp.getFileName() != null && bp.getFileName().contains(".pgp")) isEncrypted = true;
                    if (isEncrypted) {
                        encrypted = true;
                        if (password == null) {
                            throw MimeEncryptorExceptionCodes.PASSWORD_MISSING.create();
                        }
                        // decode attachment here
                        GuardKeyService keyService = Services.getService(GuardKeyService.class);
                        GuardKeys key = keyService.getKeys(msg.getSenderUserId(), msg.getSenderContextId());
                        PGPCryptoService cryptoService = Services.getService(PGPCryptoService.class);
                        ByteArrayOutputStream decoded = new ByteArrayOutputStream();
                        UserIdentity ident = new UserIdentity(key.getEmail(), password);
                        try {
                            cryptoService.decrypt((InputStream) bp.getContent(), decoded, ident);
                            if (decoded.size() > 0) {
                                String filename = bp.getFileName().replaceAll(".pgp", "");
                                bp.setFileName(filename);
                                bp.setContent(decoded.toByteArray(), FileTyper.getFileType(filename));
                            }
                        } catch (Exception ex) {
                            throw MimeEncryptorExceptionCodes.PROBLEM_DECODING.create(ex.getMessage());
                        }
                        decoded.close();
                    }
                } else if (content instanceof Multipart || content instanceof MimeMultipart) {
                    final Multipart mp2 = (Multipart) content;
                    parseMultipart(mp2);
                }
            }
            if (encrypted) {
                message.saveChanges();
            }

        } catch (MessagingException | IOException e) {

        }

    }

}
