/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.mime.encryptor.pgp.impl;

import java.io.IOException;
import java.util.ArrayList;
import javax.mail.MessagingException;
import javax.mail.internet.AddressException;
import javax.mail.internet.MimeMessage;
import org.apache.commons.io.output.ByteArrayOutputStream;
import org.json.JSONException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.mime.encryptor.pgp.inline.PGPInlineDecryptor;
import com.openexchange.guard.mime.encryptor.pgp.inline.PGPInlineEncryptor;
import com.openexchange.guard.mime.encryptor.pgp.guest.PGPGuestEncryptor;
import com.openexchange.guard.mime.encryptor.pgp.pgpmime.PGPMimeDecryptor;
import com.openexchange.guard.mime.encryptor.pgp.pgpmime.GuardPGPMimeEncryptor;
import com.openexchange.guard.mime.services.GuardParsedMimeMessage;
import com.openexchange.guard.mime.services.MimeParser;
import com.openexchange.guard.pgpcore.services.PGPCryptoService;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.mime.encryptor.Decryptor;
import com.openexchange.guard.mime.encryptor.Encryptor;
import com.openexchange.guard.mime.encryptor.ProcessMime;
import com.openexchange.guard.mime.encryptor.exceptions.MimeEncryptorExceptionCodes;
import com.openexchange.guard.mime.encryptor.osgi.Services;
import com.openexchange.guard.user.GuardCapabilities;
import com.openexchange.guard.user.GuardUserService;
import com.openexchange.guard.user.UserIdentity;
import com.openexchange.guard.user.GuardCapabilities.Permissions;

/**
 * {@link PGPProcessMime}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class PGPProcessMime implements ProcessMime {

    private static Logger LOG = LoggerFactory.getLogger(PGPProcessMime.class);

    private final PGPCryptoService pgpCryptoService;
    private final GuardConfigurationService guardConfigurationService;

    /**
     * Initializes a new {@link PGPProcessMime}.
     *
     * @param pgpCryptoService The {@link PGPCryptoService} to use for mime encryption
     */
    public PGPProcessMime(GuardConfigurationService guardConfigurationService, PGPCryptoService pgpCryptoService) {
        this.pgpCryptoService = pgpCryptoService;
        this.guardConfigurationService = guardConfigurationService;
    }

    @Override
    public String doEncrypt(GuardParsedMimeMessage msg, int userId, int contextId, String password) throws OXException {

        //Assures appropriated Guard permissions for user
        // If doesn't have guard-mail, possible secure reply.  Verify all users have keys
        // No new users allowed.
        // TODO enhance this
        if (!checkMailPermission(userId, contextId)) {
            try {
                // Check recipients as draft email (no new keys created)
                RecipientParser testRecipParse = new RecipientParser(msg, userId, contextId, true);
                ArrayList<RecipKey> checkRecipts = testRecipParse.getAllRecipients();
                // Verify found keys same length as number recipients
                if (checkRecipts.size() != msg.getRecipients().length()) {
                    LOG.error("Attempt to send email from guard only capable account, with new recipients");
                    throw MimeEncryptorExceptionCodes.SEND_NOT_AUTHORIZED.create();
                }
            } catch (AddressException | JSONException e) {
               LOG.error("Error checking for added recipients for guard capability only send", e);
               throw MimeEncryptorExceptionCodes.SEND_EMAIL_ERROR.create(e);
            }
        }

        try {
            String origMessageID = msg.getMessage().getMessageID();
            RecipientParser recipParse = new RecipientParser(msg, userId, contextId, msg.isDraft());
            ArrayList<RecipKey> recips = recipParse.getPgpRecipients();
            ArrayList<RecipKey> guests = recipParse.getGuestRecipients();
            ArrayList<RecipKey> allRecipts = recipParse.getAllRecipients();
            // Handle Guests
            if (guests.size() > 0 && !msg.isDraft()) {  // Won't create guests for draft messages
                Encryptor guestEncryptor = new PGPGuestEncryptor(pgpCryptoService, msg, guests, userId, contextId, password, allRecipts);
                if (!guestEncryptor.doEncryptAndSend()) {
                    throw MimeEncryptorExceptionCodes.SEND_EMAIL_ERROR.create("Unable to send Guest emails");
                }
            }

            if (recipParse.isInline() || msg.isInline()) {
                Encryptor inline = new PGPInlineEncryptor(pgpCryptoService, msg, recips, userId, contextId, password);
                return (removeRecips(inline.doEncrypt(), guests, origMessageID));

            } else {
                if (recips.size() > 0) {
                    Encryptor encr = new GuardPGPMimeEncryptor(pgpCryptoService, guardConfigurationService, msg, recips, userId, contextId, password);
                    return (removeRecips(encr.doEncrypt(), guests, origMessageID));
                } else {
                    if (guests.size() > 0) {  // No recips, but we had guests, we need to send this for the sent folder
                        Encryptor encr = new GuardPGPMimeEncryptor(pgpCryptoService, guardConfigurationService, msg, guests, userId, contextId, password);
                        return (removeRecips(encr.doEncrypt(), guests, origMessageID));
                    }
                }
            }
            throw MimeEncryptorExceptionCodes.NO_RECIPIENTS.create();
        } catch (JSONException | MessagingException | IOException e) {
            LOG.error("Error encrypting email ", e);
            throw MimeEncryptorExceptionCodes.SEND_EMAIL_ERROR.create(e.getMessage());
        }
    }

    /**
     * @param userId
     * @param contextId
     * @throws OXException
     */
    private void requirePermission(int userId, int contextId, boolean sending) throws OXException {
        // Check user permission
        GuardCapabilities guardUserCapabilities = Services.getService(GuardUserService.class).getGuardCapabilieties(contextId, userId);
        if (sending && !guardUserCapabilities.hasPermission(Permissions.MAIL)) {
            // TODO
            // Handle replies to encrypted emails when the user doesn't have Guard enabled
            // Guest replies when moved to core?
            throw MimeEncryptorExceptionCodes.SEND_NOT_AUTHORIZED.create();
        } else {
            if (!guardUserCapabilities.hasMinimumPermission()) {
                throw MimeEncryptorExceptionCodes.SEND_NOT_AUTHORIZED.create();
            }
        }
    }

    /**
     * Checks to see if has Guard-Mail capability.  If so, returns true.  Of Guard only, returns false
     * If no Guard permissions, throws Not_Authorized
     * @param userId
     * @param contextId
     * @return
     * @throws OXException
     */
    private boolean checkMailPermission (int userId, int contextId) throws OXException {
        GuardCapabilities guardUserCapabilities = Services.getService(GuardUserService.class).getGuardCapabilieties(contextId, userId);
        if (guardUserCapabilities.hasPermission(Permissions.MAIL)) return true;
        if (guardUserCapabilities.hasMinimumPermission()) return false;
        throw MimeEncryptorExceptionCodes.SEND_NOT_AUTHORIZED.create();
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.mime.encryptor.ProcessMime#doDecrypt(com.openexchange.guard.mime.services.GuardParsedMimeMessage, java.lang.String)
     */
    @Override
    public MimeMessage doDecrypt(GuardParsedMimeMessage msg, int userId, int contextId, UserIdentity userIdentity) throws OXException {

        //Assures appropriated Guard permissions for user
        requirePermission(userId, contextId, false);

        try {
            MimeMessage message = msg.getMessage();
            MimeParser mimeParser = new MimeParser(message);
            Decryptor decryptor = null;

            if (mimeParser.isPGPMime()) {
                decryptor = new PGPMimeDecryptor(pgpCryptoService, userIdentity);
            } else if (mimeParser.isPGPInline()) {
                decryptor = new PGPInlineDecryptor(pgpCryptoService, userIdentity);
            }

            if (decryptor != null) {
                return decryptor.doDecrypt(message);
            }
            return message;
        } catch (Exception e) {
            throw MimeEncryptorExceptionCodes.UNKNOWN_EXCEPTION.create(e, e.getMessage());
        }
    }

    /**
     * Add header to the email to remove the Guests from the list of recipients that the middleware will send to
     *
     * @param encrypted
     * @param guests
     * @return
     * @throws MessagingException
     * @throws IOException
     */
    private String removeRecips(MimeMessage encrypted, ArrayList<RecipKey> guests, String origMessageID) throws MessagingException, IOException {
        if (encrypted == null) {
            return null;
        }
        for (RecipKey key : guests) {
            encrypted.addHeader("Remove-recip", key.getEmail());
        }
        encrypted.removeHeader("X-OxGuard-PIN");  // Remove record of pin from non-encrypted email
        encrypted.saveChanges();
        if (origMessageID != null) {  // Save changes and enryption may change messageID.  Restore original here
            encrypted.setHeader("Message-ID", origMessageID);
        }
        ByteArrayOutputStream wrapped_out = new ByteArrayOutputStream();
        encrypted.writeTo(wrapped_out);
        byte[] wrapped_bytes = wrapped_out.toByteArray();
        return (new String(wrapped_bytes, "UTF-8"));
    }

}
