/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.mime.encryptor.pgp.impl;

import java.util.ArrayList;
import java.util.List;
import javax.mail.BodyPart;
import javax.mail.MessagingException;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMultipart;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.mime.encryptor.osgi.Services;
import com.openexchange.guard.pgp.GuardPGPMessages;
import com.openexchange.guard.pgp.HelpAttach;
import com.openexchange.guard.translation.GuardTranslationService;

/**
 * {@link HelpFileService}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class HelpFileService {

    private static final Logger LOG = LoggerFactory.getLogger(HelpFileService.class);

    public MimeMultipart addHelp (MimeMultipart mp, List<RecipKey> recips, int userid, int cid) throws MessagingException, OXException {
        final List<HelpAttach> attachments = getHelp(getLanguages(recips));
        for (int i = 0; i < attachments.size(); i++) {
            final BodyPart help = new MimeBodyPart();
            GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
            help.setFileName(translationService.getTranslation(GuardPGPMessages.HELP, attachments.get(i).getLanguage()) + ".txt");
            help.setContent(attachments.get(i).getTranslated(), "text/plain; charset=utf-8");
            help.setDisposition("inline");
            mp.addBodyPart(help);
        }
        return mp;
    }

    public List<BodyPart> getHelpFiles(List<RecipKey> recips) throws MessagingException, OXException {
        final List<BodyPart> ret = new ArrayList<BodyPart>();
        final List<HelpAttach> attachments = getHelp(getLanguages(recips));
        for (int i = 0; i < attachments.size(); i++) {
            final BodyPart help = new MimeBodyPart();
            GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
            help.setFileName(translationService.getTranslation(GuardPGPMessages.HELP, attachments.get(i).getLanguage()) + ".txt");
            help.setContent(attachments.get(i).getTranslated(), "text/plain; charset=utf-8");
            help.setDisposition("inline");
            ret.add(help);
        }
        return ret;
    }
    /**
     * Get an {@link List} of languages from all recipients
     *
     * @param recipients
     * @return
     */
    private List<String> getLanguages(List<RecipKey> recipients) {
        List<String> languages = new ArrayList<String>();
        for (RecipKey recip : recipients) {
            if (!languages.contains(recip.getLang())) {
                languages.add(recip.getLang());
            }
        }
        return languages;
    }

    private List<HelpAttach> getHelp(final List<String> languages) {
        final List<HelpAttach> helpfiles = new ArrayList<HelpAttach>();
        try {
            GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
            String externalReaderPath = guardConfigService.getProperty(GuardProperty.externalReaderPath);
            String oxURL = guardConfigService.getProperty(GuardProperty.externalEmailURL);

            for (int i = 0; i < languages.size(); i++) {
                GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
                final String lang = translationService.getAvailableCode(languages.get(i));
                String data = translationService.getHelpFile(lang);
                if (data != null) {
                    data = data.replace("%guest%", "https://" + externalReaderPath);
                    data = data.replace("%webmail%", "https://" + oxURL);
                    if (!contains(helpfiles, lang)) {
                        final HelpAttach att = new HelpAttach(lang, data);
                        helpfiles.add(att);
                    }
                }
            }
        } catch (final Exception e) {
            LOG.error("Problem loading help file ", e);
        }
        return (helpfiles);
    }

    private boolean contains(final List<HelpAttach> list, final String lang) {
        for (final HelpAttach att : list) {
            if (att.getLanguage().equals(lang)) {
                return (true);
            }
        }
        return (false);
    }

}
