/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.mime.encryptor.pgp.guest;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import javax.mail.BodyPart;
import javax.mail.MessagingException;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import org.json.JSONException;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.mailcreator.MailCreatorService;
import com.openexchange.guard.mime.encryptor.Encryptor;
import com.openexchange.guard.mime.encryptor.exceptions.MimeEncryptorExceptionCodes;
import com.openexchange.guard.mime.encryptor.osgi.Services;
import com.openexchange.guard.mime.services.GuardParsedMimeMessage;
import com.openexchange.guard.notification.GuardNotificationService;
import com.openexchange.guard.pgpcore.services.PGPCryptoService;
import com.openexchange.guard.storage.Storage;
import com.openexchange.guard.translation.GuardTranslationService;

/**
 * {@link PGPGuestEncryptor}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class PGPGuestEncryptor extends Encryptor {

    private ArrayList<RecipKey> allRecipients;
    /**
     * Initializes a new {@link PGPGuestEncryptor}.
     * @param pgpCryptoService The {@link PGPCryptoService} to use for encryption
     * @param msg
     * @param recips
     * @param userid
     * @param cid
     * @param password
     */
    public PGPGuestEncryptor(PGPCryptoService pgpCryptoService, GuardParsedMimeMessage msg, ArrayList<RecipKey> recips, int userid, int cid, String password, ArrayList<RecipKey> allRecipients) {
        super(pgpCryptoService, msg, recips, userid, cid, password);
        this.allRecipients = allRecipients;
    }

    @Override
    public MimeMessage doEncrypt () throws OXException {
        return null;
    }

    private byte[] encrypt () throws OXException  {
        try {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            msg.getMessage().writeTo(out);
            InputStream in = new ByteArrayInputStream(out.toByteArray());
            ByteArrayOutputStream encrypted = new ByteArrayOutputStream();

            if (msg.isSign()) {
                GuardKeyService gk = Services.getService(GuardKeyService.class);
                GuardKeys key = gk.getKeys(userid, cid);
                pgpCryptoService.encryptSigned(in, encrypted, true /*armored*/, key, password, recips.toArray(new RecipKey[recips.size()]));
            } else {
                pgpCryptoService.encrypt(in, encrypted, true /*armored*/, recips.toArray(new RecipKey[recips.size()]));
            }
            return encrypted.toByteArray();
        } catch (IOException | MessagingException | JSONException ex) {
            throw MimeEncryptorExceptionCodes.SEND_EMAIL_ERROR.create(ex.getMessage());
        }
    }

    public void sendInvitationByMail(RecipKey recipient, GuardParsedMimeMessage msg, int templateId) throws OXException {

        final String fromEmail = msg.getFromAddress().getAddress();
        final String fromName = msg.getSenderName();
        final String guestMessage = msg.getGuestMessage();
        final String host = null;

        //Create the Guest invitation email
        MailCreatorService mailCreatorService = Services.getService(MailCreatorService.class);
        JsonObject inviteEmail = mailCreatorService.getPasswordEmail(
            recipient.getEmail(),
            fromName,
            fromName,
            fromEmail,
            recipient.getNewGuestPass(),
            recipient.getLang(),
            templateId,
            guestMessage,
            host,
            userid,
            cid);

        //Send the invitation mail
        GuardNotificationService guardNotificationService = Services.getService(GuardNotificationService.class);
        guardNotificationService.send(inviteEmail, msg.getSenderIP());
    }

    /**
     * Initialises an OX Guard Guest account
     *
     * @throws OXException
     */
    private void inviteGuests(int templateId) throws OXException {
        GuardConfigurationService guardConfigurationService = Services.getService(GuardConfigurationService.class);
        if (guardConfigurationService.getBooleanProperty(GuardProperty.newGuestsRequirePassword)) {
            for (final RecipKey recipient : recips) {
                //Check for new guests
                if (recipient.isGuest() && recipient.getNewGuestPass() != null && !recipient.getNewGuestPass().isEmpty()) {
                    sendInvitationByMail(recipient, msg, templateId);
                }
            }
        }
    }

    @Override
    public boolean doEncryptAndSend () throws OXException {
        //Invite new guests before sending the original message
        GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
        int templId = guardConfigService.getIntProperty(GuardProperty.templateID, userid, cid);
        inviteGuests(templId);

        byte[] encrypted = encrypt();
        String id = createId();
        storeIds(id);
        id = "pgp-" + id;  // After storing, add the pgp identifier
        Storage storage = Services.getService(Storage.class);
        MailCreatorService mailCreatorService = Services.getService(MailCreatorService.class);
        String host = msg.getHost();
        String messageId = getMessageID();
        ArrayList<BodyPart> bpAttachments = new ArrayList <BodyPart>();
        final BodyPart att = new MimeBodyPart();

        try {
            att.setText(new String(encrypted, "UTF-8"));
            att.setHeader("Content-Type", "application/octet-stream");
            att.setFileName("encrypted.asc");
        } catch (MessagingException | UnsupportedEncodingException e1) {
            // TODO Auto-generated catch block
            e1.printStackTrace();
        }
        bpAttachments.add(att);

        for (RecipKey recip : recips) {
            String guestMessage = recip.isNewCreated() ? msg.getGuestMessage() : "";  // Only attach Guest message for new users
            storage.saveEncrObj(recip.getUserid(), recip.getCid(), id, encrypted);
            JsonObject email = mailCreatorService.createBlankGuestMail(recip.getLang(), templId, messageId, host, userid, cid, guestMessage);
            if (email == null) {
                // TODO THROW ERROR
                return false;
            }
            email = mailCreatorService.addTo(email, recip.getName(), recip.getEmail(), recip.getType());
            email = mailCreatorService.addFrom(email, msg.getFromAddress().getPersonal(), msg.getFromAddress().getAddress());
            email = mailCreatorService.addSubject(email, msg.getSubject());
            if (recip.isNewCreated()) {
                email = mailCreatorService.addPlainText(email, guestMessage);
            }
            email = mailCreatorService.noSave(email);

            // TODO, add headers?

            try {
                String url = guardConfigService.getProperty(GuardProperty.externalReaderURL, userid, cid);
                if (url == null || url.isEmpty()) {
                    url = "https://" + guardConfigService.getProperty(GuardProperty.externalReaderPath);
                }
                String newGuest = "";
                if (recip.getNewGuestPass() != null && !guardConfigService.getBooleanProperty(GuardProperty.newGuestsRequirePassword)) {
                    newGuest = "&newGuest=true";
                }
                GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
                email = mailCreatorService.addURL(email, url + "?email=" + id + "&user=" + URLEncoder.encode(recip.getEmail(), "UTF-8") + (translationService.customTemplateExists(templId, "guesttempl.html") ? ("&templid=" + templId) : "") + "&lang=" + recip.getLang() + newGuest);
            } catch (UnsupportedEncodingException e) {
                email = mailCreatorService.addURL(email, "https://" + guardConfigService.getProperty(GuardProperty.externalReaderPath) + "?email=" + id);
            }

            GuardNotificationService guardNotificationService = Services.getService(GuardNotificationService.class);
            guardNotificationService.send(email, bpAttachments, "", cid, recip.getLang(), messageId, msg.getSenderIP());
        }
        return true;
    }

    private String createId() {
        String uid = CipherUtil.getUUID();
        return uid;
    }

    private void storeIds (String id) throws OXException {
        for (RecipKey recip : recips) {
            GuestEmailSaver.saveEmailData(id, allRecipients, recip.getUserid(), recip.getCid(), msg.getFromAddress(), msg.getSenderUserId(), msg.getSenderContextId());
        }
    }

    /**
     * Get the message identifier
     *
     * @return the message identifier
     * @throws OXException if the GuardConfigurationService is absent
     */
    protected String getMessageID() throws OXException {
        final GuardConfigurationService guardConfigurationService = Services.getService(GuardConfigurationService.class);
        String mailIdDomain = guardConfigurationService.getProperty(GuardProperty.mailIdDomain);
        final String oxURL = guardConfigurationService.getProperty(GuardProperty.externalEmailURL);
        if (mailIdDomain.equals("")) {  // If maildomain not defined, try to extract domain from URL
            if (oxURL.contains("/")) {
                mailIdDomain = oxURL.substring(0, oxURL.indexOf("/"));
            }
            if (mailIdDomain.contains(":")) {
                mailIdDomain = mailIdDomain.substring(0, mailIdDomain.indexOf(":"));
            }
        }
        if (mailIdDomain.equals("")) {
            mailIdDomain = "GUARD";
        }  // Otherwise, just define it as Guard
        return ("<" + CipherUtil.getUUID() + "@" + mailIdDomain + ">");
    }

}
