/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.mime.encryptor;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Objects;
import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;
import javax.servlet.http.HttpServletResponse;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.exceptions.GuardAuthExceptionCodes;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.mime.encryptor.pgp.impl.PGPProcessMime;
import com.openexchange.guard.mime.encryptor.pgp.impl.SignMime;
import com.openexchange.guard.mime.services.MimeEncryptionService;
import com.openexchange.guard.pgpcore.services.PGPCryptoService;
import com.openexchange.guard.user.UserIdentity;
import com.openexchange.pgp.mail.PGPMimeService;
import com.openexchange.guard.mime.services.GuardParsedMimeMessage;
import com.openexchange.guard.mime.encryptor.exceptions.MimeEncryptorExceptionCodes;
import com.openexchange.guard.mime.encryptor.osgi.Services;

/**
 * {@link PGPMimeEncryptionService}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class PGPMimeEncryptionService implements MimeEncryptionService {

    private final PGPCryptoService pgpCryptoService;
    private final PGPMimeService pgpMimeService;
    private final GuardConfigurationService configurationService;

    /**
     * Initializes a new {@link PGPMimeEncryptionService}.
     *
     * @param pgpCryptoService The {@link PGPCryptoService} to use for encrypting mime messages.
     */
    public PGPMimeEncryptionService(PGPCryptoService pgpCryptoService,
                                    PGPMimeService pgpMimeService,
                                    GuardConfigurationService configurationService) {

        this.pgpCryptoService = Objects.requireNonNull(pgpCryptoService, "pgpCryptoService must not be null");
        this.pgpMimeService = Objects.requireNonNull(pgpMimeService, "pgpMimeService must not be null");
        this.configurationService = Objects.requireNonNull(configurationService, "configurationService must not be null");
    }

    @Override
    public void doEncryption (GuardParsedMimeMessage mimeMessage, int userId, int contextId, String password, OutputStream output) throws OXException {
        try {
            if (mimeMessage.isSign() && !mimeMessage.isEncrypt() && !mimeMessage.isDraft()) {  // not signing draft messages
                doSigning (output, userId, contextId, password, mimeMessage);
                return;
            }
            doEncrypt (output, userId, contextId, password, mimeMessage);
        } catch (OXException e) {
            throw e;
        } catch (Exception e) {
            throw MimeEncryptorExceptionCodes.SEND_EMAIL_ERROR.create(e, e.getMessage());
        }
    }

    @Override
    public void doDecryption(GuardParsedMimeMessage mimeMessage, int userId, int contextId, UserIdentity userIdentity, OutputStream output) throws OXException {
        try {
            doDecrypt(output, userId, contextId, mimeMessage, userIdentity);
        } catch (OXException e) {
            throw e;
        } catch (Exception e) {
            //TODO  Improve
            throw MimeEncryptorExceptionCodes.UNKNOWN_EXCEPTION.create(e, e.getMessage());
        }
    }

    /**
     * Send email for signing only.  Not encypted
     * @param response
     * @param userId
     * @param contextId
     * @param password
     * @param msg
     * @throws Exception
     */
    private void doSigning (OutputStream output, int userId, int contextId, String password, GuardParsedMimeMessage msg) throws Exception {
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        GuardKeys key = keyService.getKeys(userId, contextId);
        SignMime signer = new SignMime(pgpMimeService);
        output.write(signer.signMime(msg, key, password).getBytes(StandardCharsets.UTF_8));
    }

    /**
     * Do email encryption +/- signing and send response
     * @param response
     * @param userId
     * @param contextId
     * @param password
     * @param msg
     * @throws IOException
     * @throws OXException
     */
    private void doEncrypt (OutputStream output, int userId, int contextId, String password, GuardParsedMimeMessage msg) throws IOException, OXException {
        ProcessMime mimeProcessor = new PGPProcessMime(configurationService, pgpCryptoService);
        output.write(mimeProcessor.doEncrypt(msg, userId, contextId, password).getBytes(StandardCharsets.UTF_8));
    }

    private void doDecrypt(OutputStream output, int userId, int contextId, GuardParsedMimeMessage msg, UserIdentity userIdentity) throws IOException, OXException, MessagingException {
        ProcessMime mimeProcessor = new PGPProcessMime(configurationService, pgpCryptoService);
        MimeMessage resultMessage =  mimeProcessor.doDecrypt(msg, userId, contextId, userIdentity);
        resultMessage.writeTo(output);
    }

    /**
     * Create error JSON
     * @param ex
     * @param response
     * @throws IOException
     */
    private void handleOXException (OXException ex, HttpServletResponse response) throws IOException {
        JsonObject error = new JsonObject();
        JsonObject message = new JsonObject();
        message.addProperty("message", ex.getLocalizedMessage());
        error.add("error", message);
        if (ex.getCode() == GuardAuthExceptionCodes.BAD_PASSWORD.getNumber()) {
            ServletUtils.sendAnswer(response, 401, error.toString());
        } else {
            ServletUtils.sendNotAcceptable(response, error.toString());
        }
    }


}
