/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keys.dao;

import org.apache.commons.codec.binary.Base64;
import com.openexchange.guard.common.util.LongUtil;
import com.openexchange.guard.keymanagement.commons.util.PGPUtil;

/**
 * Results from checking signatures of Public Keys
 *
 * @author greg
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 */
public class KeySignatureVerificationResult {

    private final String id;
    private final int result;
    private final String certifier;
    private String imageData;
    private int imageType;
    private boolean hasImage = false;

    public KeySignatureVerificationResult(String id, int result, String certifier) {
        this.id = id;
        this.result = result;
        this.certifier = certifier;
    }

    public KeySignatureVerificationResult(String id, int result, long certifier) {
        this(id, result, certifier == 0 ? "0" : LongUtil.getHex(Long.toString(certifier)));
    }

    /**
     * Check if result was valid
     *
     * @return True if the signature is valid, false otherwise
     */
    public boolean isValid() {
        return (result == -1);
    }

    /**
     * Get level of signature
     *
     * @return
     */
    public String getLevel() {
        return PGPUtil.getSignatureType(result);
    }

    /**
     * Return who certified
     *
     * @return
     */
    public String getCertifier() {
        return (certifier);
    }

    /**
     * Return Id of the key verified
     *
     * @return
     */
    public String getIdVerified() {
        return (id);
    }

    /**
     * Set the image data from vector
     *
     * @param imagedata
     */
    public void setImage(byte[] imagedata) {
        imageData = Base64.encodeBase64String(imagedata);
        hasImage = true;
    }

    /**
     * Get base64 image data
     *
     * @return
     */
    public String getBase64Image() {
        return (imageData);
    }

    /**
     * Return base64 image data for jpg html image
     *
     * @return
     */
    public String getEmbeddedImage() {
        return ("data:image/jpg;base64," + imageData);
    }

    /**
     * Set the image type
     *
     * @param image
     */
    public void setImageType(int image) {
        imageType = image;
    }

    public int getImageType() {
        return (imageType);
    }

    /**
     * Return true if has image
     *
     * @return
     */
    public boolean hasImage() {
        return (hasImage);
    }
}
