/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.services.impl.lookup;

import com.openexchange.exception.OXException;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.impl.osgi.Services;
import com.openexchange.guard.keymanagement.services.lookup.RecipKeyLookupStrategy;
import com.openexchange.guard.oxapi.capabilities.Capabilities;
import com.openexchange.guard.user.GuardUser;
import com.openexchange.guard.user.GuardUserService;

/**
 * {@link MailResolverRecipKeyLookupStrategy} searches for OX Users using the configured Mail Resolver.
 * <p>
 * See: com.openexchange.guard.mailResolverUrl property
 * </p>
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class MailResolverRecipKeyLookupStrategy implements RecipKeyLookupStrategy {

    /**
     * Internal method to convert a {@link GuardUser} object into a {@link RecipKey} object
     *
     * @param user The user to conver
     * @return The {@link RecipKey} object for the given {@link GuardUser}
     */
    private RecipKey toRecipKey(GuardUser user) {
        RecipKey recipKey = new RecipKey();
        recipKey.setCid(user.getContextId());
        recipKey.setUserid(user.getId());
        recipKey.setLang(user.getLanguage());
        recipKey.setName(user.getName());
        recipKey.setEmail(user.getEmail());
        recipKey.setPgp(true);
        return recipKey;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keymanagement.services.impl.RecipKeyLookupStrategy#lookup(java.lang.String)
     */
    @Override
    public RecipKey lookup(String email) throws OXException {
        GuardUser user = Services.getService(GuardUserService.class).getUser(email);
        if (user != null) {
            //First, let's check if this is an alias for a known user
            GuardKeys aliasKeys = Services.getService(GuardKeyService.class).getKeys(user.getId(), user.getContextId());
            if (aliasKeys != null) {
                return new RecipKey(aliasKeys);
            } else {
                //Check if the resolved OX user has mail and guard capabilities enabled
                Capabilities capabilities = new Capabilities();
                if (capabilities.hasUserCapability(Capabilities.WEBMAIL_CAPABILITY, user.getId(), user.getContextId()) && capabilities.hasUserCapability(Capabilities.GUARD_CAPABILITY, user.getId(), user.getContextId())) {
                    return toRecipKey(user);
                }
            }
        }
        return null;
    }

}
