/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.services.impl;

import java.io.UnsupportedEncodingException;
import java.security.NoSuchAlgorithmException;
import java.util.Objects;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.RecipKeyService;
import com.openexchange.guard.keymanagement.services.impl.lookup.DefaultRecipKeyLookupStrategy;
import com.openexchange.guard.keymanagement.services.impl.lookup.InternalRecipKeyLookupStrategy;
import com.openexchange.guard.keymanagement.services.lookup.RecipKeyLookupStrategy;

/**
 * {@link RecipKeyServiceImpl} provides default functionality for resolving recipient keys.
 *
 * @author <a href="mailto:martin.schneider@open-xchange.com">Martin Schneider</a>
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class RecipKeyServiceImpl implements RecipKeyService {

    private final GuardKeyService guardKeyService;

    /**
     * Initializes a new {@link RecipKeyServiceImpl}.
     *
     * @param guardKeyService The {@link GuardKeyService} to use
     */
    public RecipKeyServiceImpl(GuardKeyService guardKeyService) {
        this.guardKeyService = guardKeyService;
    }

    /**
     * Internal method to create a token for HKP lookup
     *
     * @param userId The user's ID
     * @param contextId The user's context ID
     * @return A token created from the user's key email, or null if the user has no guard keys setup.
     * @throws OXException
     */
    private String createHKPClientToken(int userId, int contextId) throws OXException {
        try {
            GuardKeys keys = guardKeyService.getKeys(userId, contextId);
            if (keys != null) {
                return CipherUtil.getMD5(keys.getEmail());
            }
        } catch (NoSuchAlgorithmException | UnsupportedEncodingException e) {
            throw GuardCoreExceptionCodes.UNEXPECTED_ERROR.create(e, "Error while creating HKP Client token");
        }
        return null;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.keymanagement.services.RecipKeyService#getRecipKey(java.lang.String, com.openexchange.guard.keymanagement.services.lookup.RecipKeyLookupStrategy)
     */
    @Override
    public RecipKey getRecipKey(RecipKeyLookupStrategy lookupStrategy, String email) throws OXException {
        lookupStrategy = Objects.requireNonNull(lookupStrategy, "lookupStrategy must not be null");
        return lookupStrategy.lookup(email);
    }

    @Override
    public RecipKey getRecipKey(int senderUserId, int senderContextId, String email) throws OXException {
        return new DefaultRecipKeyLookupStrategy(senderUserId, senderContextId, createHKPClientToken(senderUserId, senderContextId)).lookup(email);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.keymanagement.services.RecipKeyService#getInternalRecipKey(int, int, java.lang.String)
     */
    @Override
    public RecipKey getInternalRecipKey(String email) throws OXException {
        return new InternalRecipKeyLookupStrategy().lookup(email);
    }
}