/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.keysources;

import java.io.IOException;
import java.io.StringReader;
import java.util.Date;
import java.util.Objects;
import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.util.PrivateKeyFactory;
import org.bouncycastle.crypto.util.PublicKeyFactory;
import org.bouncycastle.openpgp.PGPKeyPair;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.operator.bc.BcPGPKeyPair;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.common.java.Strings;
import com.openexchange.guard.keymanagement.keysources.exceptions.KeySourcesExceptionCodes;
import com.openexchange.guard.keymanagement.services.GuardMasterKeyService;
import com.openexchange.guard.keymanagement.services.KeyPairSource;
import com.openexchange.guard.keymanagement.storage.KeyCacheStorage;

/**
 * {@link DbCachingKeyPairSource} defines a {@link KeyPairSource} which gets pre generated key pairs from the {@link KeyCacheStorage}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class DbCachingKeyPairSource implements KeyPairSource {

    private static final Logger logger = LoggerFactory.getLogger(DbCachingKeyPairSource.class);
    private final GuardCipherService guardCipherService;
    private final GuardMasterKeyService guardMasterKeyService;
    private final KeyCacheStorage keyCacheStorage;

    /**
     *
     * Initializes a new {@link DbCachingKeyPairSource}.
     *
     * @param guardCipherService The service used for decrypting pre-generated keys
     * @param guardMasterKeyService The service used for getting the master key in order to decrypt pre-generated keys
     */
    public DbCachingKeyPairSource(GuardCipherService guardCipherService, GuardMasterKeyService guardMasterKeyService, KeyCacheStorage keyCacheStorage) {
        this.guardCipherService = Objects.requireNonNull(guardCipherService, "guardCipherService must not be null");
        this.guardMasterKeyService = Objects.requireNonNull(guardMasterKeyService, "guardMasterKeyService must not be null");
        this.keyCacheStorage = Objects.requireNonNull(keyCacheStorage, "keyCacheStorage must not be null");
    }

    /**
     * Internal method to fetch pre generated key data from the storage
     *
     * @return The pre generated key data, or null if no such data is available
     * @throws OXException
     */
    private synchronized String getKeyData() throws OXException {
        return keyCacheStorage.popNext();
    }

    /**
     * Internal method to decrypt the given keyData
     *
     * @param keyData The encrypted key data
     * @return The decrypted key data
     * @throws OXException
     */
    private String decryptKeyData(String keyData) throws OXException {
        if (!Strings.isEmpty(keyData)) {
            String decryptedKeyData = guardCipherService.decrypt(keyData, guardMasterKeyService.getMasterKey().getRC(), "key");
            return decryptedKeyData.isEmpty() ? null : decryptedKeyData;
        }
        else {
            logger.info("No pre-generated key pair available");
        }
        return null;
    }

    /**
     * Internal method for parsing key data
     *
     * @param keyData The key data to parse
     * @return The parsed key data as {@link PEMKeyPair} object
     * @throws IOException
     */
    private PEMKeyPair parseKeyData(String keyData) throws IOException {
        try (StringReader dataReader = new StringReader(keyData); PEMParser parser = new PEMParser(dataReader)) {
            return (PEMKeyPair) parser.readObject();
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keymanagement.services.KeyPairCreationStrategy#createKeyPair()
     */
    @Override
    public PGPKeyPair get() throws OXException {

        //Get pre generated key data from storage and decrypt it for further processing
        String keyData = decryptKeyData(getKeyData());
        if (keyData != null) {
            try {
                //Create a new key pair from the pre generated decrypted key data
                PEMKeyPair pemKeyPair = parseKeyData(keyData);
                AsymmetricCipherKeyPair keyPair =
                    new AsymmetricCipherKeyPair(PublicKeyFactory.createKey(pemKeyPair.getPublicKeyInfo()),
                                                PrivateKeyFactory.createKey(pemKeyPair.getPrivateKeyInfo()));
                return new BcPGPKeyPair(PGPPublicKey.RSA_GENERAL, keyPair, new Date());
            } catch (Exception e) {
                throw KeySourcesExceptionCodes.KEY_CREATION_ERROR.create(e, e.getMessage());
            }
        }
        return null;
    }
}
