package com.openexchange.guard.keymanagement.commons.util;

import java.util.Date;
import java.util.Iterator;

import org.bouncycastle.bcpg.CompressionAlgorithmTags;
import org.bouncycastle.bcpg.HashAlgorithmTags;
import org.bouncycastle.bcpg.SymmetricKeyAlgorithmTags;
import org.bouncycastle.bcpg.sig.Features;
import org.bouncycastle.bcpg.sig.KeyFlags;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureGenerator;
import org.bouncycastle.openpgp.PGPSignatureSubpacketGenerator;
import org.bouncycastle.openpgp.PGPSignatureSubpacketVector;
import org.bouncycastle.openpgp.operator.bc.BcPGPContentSignerBuilder;

public class UserIDCreator {

	/**
	 * Find either the primary UserId, or the most recent, and use those hashes for
	 * a new PGPSignatureSubpacketGenerator
	 * 
	 * @param publicKey
	 * @param keyId
	 * @return
	 */
	private static PGPSignatureSubpacketGenerator createSignatureGeneratorFromPrior(PGPPublicKey publicKey,
			Long keyId) {
		PGPSignatureSubpacketGenerator gen = new PGPSignatureSubpacketGenerator();
		Iterator<byte[]> userIds = publicKey.getRawUserIDs();
		PGPSignatureSubpacketVector last = null;
		Date mostRecentSignatureDate = null;
		while (userIds.hasNext()) {
			byte[] rawId = userIds.next();
			Iterator<PGPSignature> sigs = publicKey.getSignaturesForID(rawId);
			while (sigs.hasNext()) {
				PGPSignature sig = sigs.next();
				if (sig.isCertification() && sig.getKeyID() == keyId) {
					PGPSignatureSubpacketVector vectors = sig.getHashedSubPackets();
					if (vectors != null) {
						if (vectors.isPrimaryUserID()) {
							return setHashes(gen, vectors);
						}
						Date signatureCreationTime = vectors.getSignatureCreationTime();
						if (mostRecentSignatureDate == null || (signatureCreationTime != null
								&& signatureCreationTime.getTime() > mostRecentSignatureDate.getTime())) {
							mostRecentSignatureDate = signatureCreationTime;
							last = vectors;
						}
					}
				}
			}
		}
		if (last != null) {
			return setHashes(gen, last);
		}
		return defaultHashSettings(gen);
	}

	/**
	 * Return a PGPSignatureSubpacketGenerator populated with vectors
	 * 
	 * @param gen
	 * @param vectors
	 * @return
	 */
	private static PGPSignatureSubpacketGenerator setHashes(PGPSignatureSubpacketGenerator gen,
			PGPSignatureSubpacketVector vectors) {
		gen.setKeyFlags(false, vectors.getKeyFlags());
		if (vectors.getFeatures() != null) {
			byte[] features = vectors.getFeatures().getData();
			for (byte feature : features) {
				gen.setFeature(vectors.getFeatures().isCritical(), feature);
			}
		}
		if (vectors.getPreferredCompressionAlgorithms() != null) {
			gen.setPreferredCompressionAlgorithms(false, vectors.getPreferredCompressionAlgorithms());
		}
		if (vectors.getPreferredHashAlgorithms() != null) {
			gen.setPreferredHashAlgorithms(false, vectors.getPreferredHashAlgorithms());
		}
		if (vectors.getPreferredSymmetricAlgorithms() != null) {
			gen.setPreferredSymmetricAlgorithms(false, vectors.getPreferredSymmetricAlgorithms());
		}
		if (vectors.getPreferredCompressionAlgorithms() != null) {
			gen.setPreferredCompressionAlgorithms(false, vectors.getPreferredCompressionAlgorithms());
		}
		if (vectors.isPrimaryUserID()) {
			gen.setPrimaryUserID(false, true);
		}
		return gen;
	}

	/**
	 * Set PGPSignatureSubpacketGenerator to a set of defaults
	 * 
	 * @param gen
	 * @return
	 */
	private static PGPSignatureSubpacketGenerator defaultHashSettings(PGPSignatureSubpacketGenerator gen) {
		gen.setPreferredHashAlgorithms(false, new int[] { HashAlgorithmTags.SHA512 });
		gen.setPreferredSymmetricAlgorithms(false, new int[] { SymmetricKeyAlgorithmTags.AES_256 });
		gen.setPreferredCompressionAlgorithms(false, new int[] { CompressionAlgorithmTags.ZIP });
		gen.setFeature(true, Features.FEATURE_MODIFICATION_DETECTION);
		gen.setKeyFlags(false, KeyFlags.SIGN_DATA | KeyFlags.CERTIFY_OTHER);
		return gen;
	}

	/**
	 * Adds a new User ID to a {@link PGPPublicKeyRing}
	 *
	 * @param publicKeyRing
	 *            The public key ring to add the user ID to
	 * @param privateKey
	 *            The private key used for signing
	 * @param userId
	 *            The new user ID
	 * @return The public key ring containing the new user ID
	 * @throws PGPException
	 */
	public static PGPPublicKeyRing addUID(PGPPublicKeyRing publicKeyRing, PGPPrivateKey privateKey, String userId)
			throws PGPException {
		PGPPublicKey pub = publicKeyRing.getPublicKey();
		PGPSignatureGenerator generator = new PGPSignatureGenerator(
				new BcPGPContentSignerBuilder(PGPPublicKey.RSA_GENERAL, org.bouncycastle.openpgp.PGPUtil.SHA1));
		generator.init(PGPSignature.POSITIVE_CERTIFICATION, privateKey);
		PGPSignatureSubpacketGenerator signhashgen = createSignatureGeneratorFromPrior(pub, privateKey.getKeyID());
		generator.setHashedSubpackets(signhashgen.generate());
		PGPSignature certification = generator.generateCertification(userId, pub);
		PGPPublicKey newPubKey = PGPPublicKey.addCertification(pub, userId, certification);
		publicKeyRing = PGPPublicKeyRing.removePublicKey(publicKeyRing, pub);
		publicKeyRing = PGPPublicKeyRing.insertPublicKey(publicKeyRing, newPubKey);
		return publicKeyRing;
	}
}
