
package com.openexchange.guard.hkpclient.services;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xbill.DNS.SRVRecord;
import com.openexchange.exception.OXException;
import com.openexchange.guard.dns.DNSService;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.hkpclient.client.HKPClient;
import com.openexchange.guard.hkpclient.client.HKPClient.TLSMODE;
import com.openexchange.guard.hkpclient.osgi.Services;

/**
 *
 * {@link SRVHKPClientService} Service for querying DNS SRV configured HKP Servers
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class SRVHKPClientService implements HKPClientService {

    private static final Logger logger = LoggerFactory.getLogger(SRVHKPClientService.class);
    private static final String HKPS_SERVICE_NAME = "_hkps._tcp.";
    private static final String HKP_SERVICE_NAME  = "_hkp._tcp.";
    private final HKPClientService delegate;
    private final boolean forceTLS;

    /**
     *
     * Initializes a new {@link SRVHKPClientService}.
     */
    public SRVHKPClientService() {
        this(null, false);
    }

    /**
     *
     * Initializes a new {@link SRVHKPClientService}.
     *
     * @param delegate The service to delegate search queries to in a public key ring was not found
     * @param foceTLS True to only allow TLS connection to the HKP Server, false otherwise
     */
    public SRVHKPClientService(HKPClientService delegate, boolean forceTLS) {
        this.delegate = delegate;
        this.forceTLS = forceTLS;
    }

    /**
     * Internal method to query HKP servers which were configured in the email-domain's DNS SRV record
     *
     * @param clientToken an identification token put into the X-UI-INTERNAL-ACCOUNT-ID header, or null not not set the header
     * @param email the email to query
     * @return a list of found PGP key rings for the given email, an empty collection if no keys were found or the email's domain has no HKP SRV set
     * @throws Exception
     */
    private Collection<PGPPublicKeyRing> findForSRVRecords(String clientToken, String email) throws Exception {
        TLSMODE tlsMode = forceTLS ? TLSMODE.TLS : TLSMODE.BY_PORT;
        DNSService dnsService = Services.getService(DNSService.class);
        List<SRVRecord> records = dnsService.getSrvs(HKPS_SERVICE_NAME, email);
        if(records.isEmpty()) {
            records = dnsService.getSrvs(HKP_SERVICE_NAME, email);
        }
        else {
            //always force TLS for HKPS
            tlsMode = TLSMODE.TLS;
        }

        Collection<PGPPublicKeyRing> rings = new ArrayList<PGPPublicKeyRing>();
        int current = 0;
        SRVRecord nextcheck;
        while ((nextcheck = dnsService.findNext(current, records)) != null) {
            String target = nextcheck.getTarget().toString();
            if (target.endsWith(".")) {
                target = target.substring(0, target.length() - 1);
            }
            int port = nextcheck.getPort();
            rings = new HKPClient(clientToken, tlsMode).findKeys(target, port, email);
            if (rings != null) {
                return rings;
            }
            current = nextcheck.getPriority();
        }
        return rings;
    }

    /**
     * Internal method to query SRV records
     *
     * @param clientToken an identification token put into the X-UI-INTERNAL-ACCOUNT-ID header, or null not not set the header
     * @param email The email
     * @return A list of public keys for the email
     * @throws Exception
     */
    private Collection<RemoteKeyResult> findAllInternal(String clientToken, String email) throws Exception {

        //Searching DNS SRV records
        try {
            Collection<PGPPublicKeyRing> results = findForSRVRecords(clientToken, email);
            if (results != null && results.size() > 0) {
                logger.debug("Remote keys found through SRV");
                return RemoteKeyResult.createCollectionFrom(results, HKPKeySources.SRV_HKP_REMOTE_SERVER);
            }
        } catch (Exception e) {
            logger.error("Error querying remote HKP server form SRV record", e);
        }
        if (delegate != null) {
            return delegate.find(clientToken, email);
        }
        return null;
    }

    /**
     * Internal method to find key by id
     * Searches based on the from email domain and queries for the keyid
     *
     * @param clientToken
     * @param id
     * @param email
     * @return
     * @throws Exception
     */
    private RemoteKeyResult findInternal(String clientToken, Long id, String email) throws Exception {
        // To do SRV lookup, we have to search by email address to get domain
        if (email != null) {
            Collection<RemoteKeyResult> res = findAllInternal(clientToken, email);
            if (res != null && !res.isEmpty()) {  //If results, then search all to see if has the key by id
                for (PGPPublicKeyRing ring : RemoteKeyResult.getRingsFrom(res)) {
                    PGPPublicKey found = ring.getPublicKey(id);
                    if (found != null) {
                        return new RemoteKeyResult(ring, HKPKeySources.SRV_HKP_REMOTE_SERVER);
                    }
                }
            }
        }
        RemoteKeyResult result = null;
        if (delegate != null) {
            result = delegate.find(clientToken, email, id);
        }
        return result;

    }

    @Override
    public Collection<RemoteKeyResult> find(String clientToken, String email) throws OXException {
        try {
            return findAllInternal(clientToken, email);
        } catch (Exception e) {
            throw GuardCoreExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

    @Override
    public RemoteKeyResult find(String clientToken, String email, Long id) throws OXException {
        try {
            return findInternal(clientToken, id, email);
        } catch (Exception e) {
            throw GuardCoreExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }
}
