/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.encryption;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import com.openexchange.exception.OXException;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;

/**
 * {@link EncryptedItemAccess} - A helper class for verifying access to guest-email by using an instance of {@link EncryptedItem}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.8.0
 */
public class EncryptedItemAccess {

	private static final String TAG_CONTEXT_ID = "c";
	private static final String TAG_USER_ID = "i";
	private static final String TAG_SHARE = "share";

	/**
	 * Internal method to parse the items XML data
	 * 
	 * @param xmlData The xml data
	 * @return The parsed XML {@link Document}
	 * @throws SAXException
	 * @throws IOException
	 * @throws ParserConfigurationException
	 */
	private static Document parseXMLFrom(InputStream xmlData) throws SAXException, IOException, ParserConfigurationException {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder documentBuidler = factory.newDocumentBuilder();
		Document document = documentBuidler.parse(xmlData);
		document.getDocumentElement().normalize();
		return document;
	}
	
	/**
	 * Internal method to check if a user exists in the given user list
	 * 
	 * @param userList The user list
	 * @param id The ID of the user to check
	 * @param cid The ID of the user's context to check
	 * @return The {@link Element} describing the user in the list, or null if no such user was found in the list
	 */
	private static Element getUser(NodeList userList, int id, int cid) {
		for (int i = 0; i < userList.getLength(); i++) {
			Element user = (Element) userList.item(i);
			NodeList idNode = user.getElementsByTagName(TAG_USER_ID);
			NodeList cidNode = user.getElementsByTagName(TAG_CONTEXT_ID);
			if (idNode.getLength() > 0 && cidNode.getLength() > 0 &&
			    Integer.toString(id).equals(idNode.item(0).getTextContent()) && 
			    Integer.toString(cid).equals(cidNode.item(0).getTextContent())) {
					return user;
			}
		}
		return null;
	}

	/**
	 * Internal method to check if a given user is marked as sender or recipient  within the given document.
	 * 
	 * @param document The document
	 * @param id       The ID of the user
	 * @param cid      The ID of the user's context
	 * @return true, if the user is marked as sender or recipient of the given  document, false otherwise.
	 */
	private static boolean hasUser(Document document, int id, int cid) {		
		NodeList shares = document.getElementsByTagName(TAG_SHARE);
		if (shares.getLength() > 0) {
			Element share = (Element) shares.item(0);
			NodeList users = share.getChildNodes();
			return getUser(users, id, cid) != null;			
		}
		return false;
	}

	/**
	 * Verifies that the given user is a recipient or sender of the given {@link EncryptedItem}
	 * 
	 * @param encryptedItem The item
	 * @param userId        The ID of the user to check the access for
	 * @param contextId     The ID of the user's context
	 * @return True, if the user is The sender or a recipient of the given {@link EncryptedItem}
	 * @throws OXException
	 */
	public static boolean verifyAccess(EncryptedItem encryptedItem, int userId, int contextId) throws OXException {
		try {
			Document xml = parseXMLFrom(new ByteArrayInputStream(encryptedItem.getXml().getBytes(StandardCharsets.UTF_8)));
			return hasUser(xml, userId, contextId);
		} catch (ParserConfigurationException | SAXException | IOException e) {
			throw GuardCoreExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
		}
	}
}
