/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.dns.internal;

import java.io.IOException;
import org.xbill.DNS.Flags;
import org.xbill.DNS.Message;
import org.xbill.DNS.Rcode;
import org.xbill.DNS.Resolver;
import com.openexchange.exception.OXException;

/**
 * {@link ValidatingResolverDecorator} is a decorator which performs validation of DNS responses using a given {@link DNSValidator} after a query was sent.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class ValidatingResolverDecorator extends AbstractResolverDecorator {

    private static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(ValidatingResolverDecorator.class);
    private final DNSValidator validator;
    private final boolean validateFailures;

    /**
     * Initializes a new {@link ValidatingResolverDecorator}.
     *
     * @param decoratedResolver The resolver to decorate
     * @param validator The validator to use
     * @param validateFailures True, if the validator should retry the DNS lookup without validation after the first attempt was not validated in order to notice potential MITM-Attacks.
     */
    public ValidatingResolverDecorator(Resolver decoratedResolver, DNSValidator validator, boolean validateFailures) {
        super(decoratedResolver);
        this.validator = validator;
        this.validateFailures = validateFailures;
    }

    /**
     * Initializes a new {@link ValidatingResolverDecorator} which validates failures (i.e with validateFailures=true)
     *
     * @param decoratedResolver The resolver to decorate
     * @param validator The validator to use
     */
    public ValidatingResolverDecorator(Resolver decoratedResolver, DNSValidator validator) {
        this(decoratedResolver, validator, true);
    }

    /*
     * (non-Javadoc)
     *
     * @see org.xbill.DNS.Resolver#send(org.xbill.DNS.Message)
     */
    @Override
    public Message send(Message query) throws IOException {

        //Send the message
        Message answer = getDecoratedResolver().send(query);

        //Validate the response
        boolean validated = false;
        try {
            validated = this.validator.isValid(answer);
        } catch (OXException e) {
            logger.error("Error while validating DNS response. Response will be treated as not validated.", e);
        }

        //If validation fails we retry the DNS query without validation:
        //Retrying the DNS query with the CD (Check Disabled) Flag set. If this works we probably have a MITM-A and not a DNS/Network error
        if (!validated && validateFailures) {
            Message cdQuery = (Message) query.clone();
            cdQuery.getHeader().setFlag(Flags.CD);
            Message cdAnswer = getDecoratedResolver().send(cdQuery);
            if (cdAnswer.getRcode() != Rcode.SERVFAIL) {
                logger.warn("! WARNING ! DNSSEC validation failed but another DNS lookup was successful with validation turned off. This could be a potential Man-In-The-Middle-Attack. The DNS record will be discarded.");
            }
        }
        if (!validated) {
            answer.getHeader().setRcode(Rcode.SERVFAIL);
        }
        return answer;
    }
}
