/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.dns.internal;

import org.xbill.DNS.Flags;
import org.xbill.DNS.Message;
import org.xbill.DNS.Record;
import org.xbill.DNS.Section;

/**
 * {@link FlagTrustingDNSValidator} just checks the present of the <i>AD(=Authorized Data)</i> flag
 * <br>
 * <p><b> PLEASE NOTE: This implementation does not validate the DNS Records by itself. It just trusts the given <i>AD</i> flag from the DNS resolver.</b></p>
 * <p>Please ensure you either run a local DNS resolver which performs DNSSEC validation, or the network from this OX Guard node to the DNS validating resolver is fully trusted.</p>
 * <br>
 * 
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class FlagTrustingDNSValidator implements DNSValidator {

    /**
     * {@link ValidationMode} The validation mode to use
     *
     * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
     * @since v2.6.0
     */
    public enum ValidationMode {
        /**
         * Only log a warning
         */
        WARN,

        /**
         * Fail on invalid response
         */
        FAIL,
    };

    private static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(FlagTrustingDNSValidator.class);
    private final ValidationMode validationMode;

    /**
     * Initializes a new {@link FlagTrustingDNSValidator}.
     * 
     * @param validationMode The mode to use during validation
     */
    public FlagTrustingDNSValidator(ValidationMode validationMode) {
        this.validationMode = validationMode;
    }

    /**
     * Checks whether a message has a given flag
     * 
     * @param message The message
     * @param flag The flag to check
     * @return True if the message contains the given flag, false otherwise
     */
    private boolean hasFlag(Message message, int flag) {
        return message.getHeader().getFlag(flag);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.dns.internal.dnssec.DNSSECValidator#isValid(org.xbill.DNS.Message)
     */
    @Override
    public boolean isValid(Message answer) {
        boolean hasFlag = hasFlag(answer, Flags.AD);
        String name = "unknown domain";
        Record[] answerSection = answer.getSectionArray(Section.ANSWER);
        if (answerSection != null && answerSection.length > 0) {
            name = answerSection[0].getName().toString();
        }
        if (!hasFlag) {

            if (validationMode == ValidationMode.WARN) {
                logger.warn("The header of the obtained SRV DNS record for the domain \"{}\" does not contain the AD (Authenticated Data) Flag. Please consider to enable DNSSEC for the domain.", name);
            }
            else if (validationMode == ValidationMode.FAIL) {
                logger.error("The header of the obtained SRV DNS record for the domain \"{}\" does not contain the AD (Authenticated Data) Flag and is discarded.", name);
                return false;
            }
        }
        else {
            logger.debug("The header of the obtained SRV DNS record for the doman {} contains the AD (Authentication Data) Flag as expected.", name);
        }
        return true;
    }
}
