/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.dns.internal;

import java.util.ArrayList;
import java.util.List;
import org.xbill.DNS.Flags;
import org.xbill.DNS.Lookup;
import org.xbill.DNS.Record;
import org.xbill.DNS.SRVRecord;
import org.xbill.DNS.TXTRecord;
import org.xbill.DNS.TextParseException;
import org.xbill.DNS.Type;
import com.openexchange.exception.OXException;
import com.openexchange.guard.dns.DNSService;
import com.openexchange.guard.dns.exception.DNSExceptionCodes;

/**
 * {@link DNSServiceImpl}
 *
 * @author <a href="mailto:martin.schneider@open-xchange.com">Martin Schneider</a>
 * @since v2.4.0
 */
public class DNSServiceImpl implements DNSService {

    private final DNSValidator validator;

    /**
     * Initializes a new {@link DNSServiceImpl}.
     */
    public DNSServiceImpl() {
        this(null);
    }

    /**
     * Initializes a new {@link DNSServiceImpl}.
     *
     * @param validator A validator used for validating DNSSEC signatures of records, or null to not validate records
     */
    public DNSServiceImpl(DNSValidator validator) {
        this.validator = validator;
    }

    /**
     * Get a list of SRV Records for PGP Servers for given domain
     *
     * @param serviceName
     * @param email
     * @return
     * @throws TextParseException
     */
    @Override
    public List<SRVRecord> getSrvs(String serviceName, String email) throws OXException {
        List<SRVRecord> list = new ArrayList<SRVRecord>();
        if (!email.contains("@")) {
            return list;
        }
        String domain = email.substring(email.indexOf("@") + 1);
        try {
            Lookup lookup = new Lookup(serviceName + domain, Type.SRV);
            if (validator != null) {
                //Decorating the resolver with decorators which...
                lookup.setResolver(
                    new HeaderFlagResolverDecorator( //...adds the AD flag to each query
                        new ValidatingResolverDecorator(Lookup.getDefaultResolver(), validator), //..validates each DNS response with the given validator
                        Flags.AD));
            }
            Record[] records = lookup.run();
            if (records == null) {
                return list;
            }
            for (int i = 0; i < records.length; i++) {
                SRVRecord srv = (SRVRecord) records[i];
                list.add(srv);
            }
        } catch (TextParseException e) {
            throw DNSExceptionCodes.INVALID_NAME_ERROR.create("_hkp._tcp." + domain, e);
        }
        return list;
    }

    @Override
    public SRVRecord findNext(int current, List<SRVRecord> records) {
        int index = -1;
        int lowest = 0;
        for (int i = 0; i < records.size(); i++) {
            SRVRecord tocheck = records.get(i);
            if (tocheck.getPriority() > current) {
                if ((lowest == 0) || (tocheck.getPriority() < lowest)) {
                    lowest = tocheck.getPriority();
                    index = i;
                }
            }
        }
        if (index == -1) {
            return (null);
        }
        return (records.get(index));
    }

    @Override
    public String getUrl(String email) throws OXException {
        String url = null;
        try {
            Record[] records = new Lookup("_oxguard." + email, Type.TXT).run();
            if (records == null) {
                return url;
            }
            for (int i = 0; i < records.length; i++) {
                TXTRecord txt = (TXTRecord) records[i];
                if (txt.getName().toString().toLowerCase().contains("oxguard")) { // make sure this is oxguard record
                    url = txt.rdataToString().replace("\"", "").replace("\\", "");
                    if (!url.endsWith("/")) {
                        url = url + "/";
                    }
                    // make sure ssl
                    if (url.startsWith("http:")) {
                        url = url.replace("http:", "https:");
                    }
                    return url;
                }
            }
        } catch (TextParseException e) {
            throw DNSExceptionCodes.INVALID_NAME_ERROR.create("_oxguard." + email, e);
        }
        return url;
    }
}
