/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.database.osgi;

import java.sql.Connection;
import java.sql.SQLException;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.ServiceReference;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import com.openexchange.database.AssignmentFactory;
import com.openexchange.database.migration.DBMigration;
import com.openexchange.database.migration.DBMigrationConnectionProvider;
import com.openexchange.database.migration.DBMigrationExecutorService;
import com.openexchange.database.migration.resource.accessor.BundleResourceAccessor;
import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.database.DatabaseMaintenanceService;
import com.openexchange.guard.database.GuardDatabaseService;
import com.openexchange.guard.database.GuardShardingService;
import com.openexchange.guard.database.dao.SchemaCandidate;
import com.openexchange.guard.database.internal.DatabaseMaintenanceServiceImpl;
import com.openexchange.guard.database.internal.GuardDatabaseServiceImpl;
import com.openexchange.guard.database.internal.GuardShardingServiceImpl;
import com.openexchange.guard.database.utils.DBUtils;

/**
 *
 * {@link DBMigrationServiceTracker}
 *
 * @author <a href="mailto:martin.schneider@open-xchange.com">Martin Schneider</a>
 * @since 2.4.0
 */
public class DBMigrationServiceTracker implements ServiceTrackerCustomizer<DBMigrationExecutorService, DBMigrationExecutorService> {

    private static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(DBMigrationServiceTracker.class);

    public static final String GUARDDB_CHANGE_LOG = "/liquibase/guarddbChangeLog.xml";
    public static final String SHARDDB_CHANGE_LOG = "/liquibase/sharddbChangeLog.xml";

    private final BundleContext context;

    private final GuardDatabaseService guardDatabaseService;
    private final GuardConfigurationService guardConfigurationService;
    private DatabaseMaintenanceService databaseMaintenanceService;
    private GuardShardingService guardShardingService;
    private final AssignmentFactory assignmentFactory;

    public DBMigrationServiceTracker(BundleContext context, GuardDatabaseService guardDatabaseService, GuardConfigurationService guardConfigurationService, AssignmentFactory assignmentFactory) {
        this.context = context;
        this.guardDatabaseService = guardDatabaseService;
        this.guardConfigurationService = guardConfigurationService;
        this.assignmentFactory = assignmentFactory;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public DBMigrationExecutorService addingService(ServiceReference<DBMigrationExecutorService> serviceRef) {
        DBMigrationExecutorService dbMigrationExecutorService = context.getService(serviceRef);

        databaseMaintenanceService = new DatabaseMaintenanceServiceImpl(guardDatabaseService, guardConfigurationService, assignmentFactory);
        context.registerService(DatabaseMaintenanceService.class, databaseMaintenanceService, null);

        guardShardingService = new GuardShardingServiceImpl(guardDatabaseService, dbMigrationExecutorService, databaseMaintenanceService);
        context.registerService(GuardShardingService.class, guardShardingService, null);

        try {
            String guardDbSchemaName = guardConfigurationService.getProperty(GuardProperty.dbSchemaBaseName);
            handleGuardSchema(dbMigrationExecutorService, guardDbSchemaName);
            handleInitialShardSchema(1);
            handleAllShardSchemas();

            //Only if above is successful: register the GuardDatabaseService
            context.registerService(GuardDatabaseService.class, guardDatabaseService, null);
        } catch (OXException oxException) {
            LOG.error("Unable to init (create or update) guard schema", oxException);
        }
        return dbMigrationExecutorService;
    }

    private void handleAllShardSchemas() {
        try {
            if (guardShardingService != null) {
                guardShardingService.updateAllShardSchemas();
            } else {
                LOG.error("Unable to update all existing shard schemas as required GuardShardingService is null.");
            }
        } catch (OXException oxException) {
            LOG.error("Unable to init shard schema", oxException);
        }
    }

    private void handleInitialShardSchema(int shardId) {
        try {
            createShardSchema(shardId);
        } catch (OXException oxException) {
            LOG.error("Unable to init (create or update) shard schema", oxException);
        }
    }

    private void createShardSchema(int shardId) throws OXException {
        String shardDbSchema = guardDatabaseService.getShardName(shardId);

        Connection shardMasterInitConnection = guardDatabaseService.getWritableForShardInit();
        String shardSlaveHost = guardConfigurationService.getProperty(GuardProperty.oxGuardShardRead);

        SchemaCandidate shardSchemaCandidates = new SchemaCandidate(guardDatabaseService, shardMasterInitConnection, shardDbSchema, shardSlaveHost);
        try {
            shardSchemaCandidates.start();
            databaseMaintenanceService.createAndRegisterSchemaIfNotExistent(shardSchemaCandidates);
            shardSchemaCandidates.commit();
        } finally {
            shardSchemaCandidates.finish();
        }
    }

    private void handleGuardSchema(DBMigrationExecutorService service, String guardDbSchema) throws OXException {
        createGuardSchema(guardDbSchema);
        updateGuardSchema(service, guardDbSchema);
    }

    /**
     * @param guardDbSchema
     * @throws OXException
     * @throws SQLException
     */
    private void createGuardSchema(String guardDbSchema) throws OXException {
        Connection guardMasterInitConnection = guardDatabaseService.getWritableForGuardInit();
        String guardSlaveHost = guardConfigurationService.getProperty(GuardProperty.oxGuardDatabaseRead);

        SchemaCandidate masterSchemaCandidates = new SchemaCandidate(guardDatabaseService, guardMasterInitConnection, guardDbSchema, guardSlaveHost);
        try {
            masterSchemaCandidates.start();
            databaseMaintenanceService.createAndRegisterSchemaIfNotExistent(masterSchemaCandidates);
            masterSchemaCandidates.commit();
        } finally {
            masterSchemaCandidates.finish();
        }
    }

    /**
     * @param service
     * @throws OXException
     */
    private void updateGuardSchema(DBMigrationExecutorService service, String guardDbSchema) throws OXException {
        final Connection guardConnection = guardDatabaseService.getWritableForGuardNoTimeout();

        BundleResourceAccessor bundleResourceAccessor = new BundleResourceAccessor(FrameworkUtil.getBundle(GuardDatabaseServiceImpl.class));
        DBMigration dbMigration = new DBMigration(new DBMigrationConnectionProvider() {

            @Override
            public Connection get() {
                return guardConnection;
            }

            @Override
            public void back(Connection connection) {
                DBUtils.autocommit(connection);
                guardDatabaseService.backWritableForGuardNoTimeout(connection);
            }

            @Override
            public void backAfterReading(Connection connection) {
                back(connection);
            }
        }, GUARDDB_CHANGE_LOG, bundleResourceAccessor, guardDbSchema);
        service.scheduleDBMigration(dbMigration);
    }

    @Override
    public void modifiedService(ServiceReference<DBMigrationExecutorService> serviceRef, DBMigrationExecutorService migrationService) {
        // nothing to do
    }

    @Override
    public void removedService(ServiceReference<DBMigrationExecutorService> serviceRef, DBMigrationExecutorService migrationService) {
        context.ungetService(serviceRef);
    }
}
