/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.configuration;

import java.io.File;
import com.openexchange.config.Reloadable;
import com.openexchange.exception.OXException;

/**
 * {@link GuardConfigurationService}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public interface GuardConfigurationService extends Reloadable {

    /**
     * Fetches the value of specified for the given String
     *
     * @param property The key the configuration should be fetched
     * @return The string value of the property
     */
    String getProperty(String property);
    
    /**
     * Fetches a property from a specified property file
     * @param fileName the name of the property file
     * @param property the property to fetch
     * @return the value of the property
     */
    String getPropertyFromFile(String fileName, String property);

    /**
     * Fetches the value of specified {@link GuardProperty}
     *
     * @param property The {@link GuardProperty} name to fetch
     * @return The string value of the property
     */
    String getProperty(GuardProperty property);

    /**
     * Fetches the value of specified {@link GuardProperty}
     *
     * @param property The {@link GuardProperty} name to fetch
     * @return The integer value of the property
     */
    int getIntProperty(GuardProperty property);

    /**
     * Fetches the value of specified {@link GuardProperty}
     *
     * @param property The {@link GuardProperty} name to fetch
     * @return The the boolean value of the property
     */
    boolean getBooleanProperty(GuardProperty property);

    /**
     * Fetches the value of specified {@link GuardProperty}
     *
     * @param property The {@link GuardProperty} name to fetch
     * @param userId The user identifier
     * @param contextId The context identifier
     * @return The string value of the property
     */
    String getProperty(GuardProperty property, int userId, int contextId);

    /**
     * Fetches the value of specified {@link GuardProperty}
     *
     * @param property The {@link GuardProperty} name to fetch
     * @param userId The user identifier
     * @param contextId The context identifier
     * @return The integer value of the property
     */
    int getIntProperty(GuardProperty property, int userId, int contextId);

    /**
     * Fetches the value of specified {@link GuardProperty}
     *
     * @param property The {@link GuardProperty} name to fetch
     * @param userId The user identifier
     * @param contextId The context identifier
     * @return The the boolean value of the property
     */
    boolean getBooleanProperty(GuardProperty property, int userId, int contextId);

    /**
     * Fetches the string value of the specified OX property for the specified user in the specified context
     *
     * @param property The name of the OX property
     * @param userId The user identifier
     * @param contextId The context identifier
     * @param defaultValue The default value to return if the property does not exist
     * @return The string value if the property exists, or <code>null</code>
     */
    String getProperty(String property, int userId, int contextId, String defaultValue);

    /**
     * Fetches the integer value of the specified OX property for the specified user in the specified context
     *
     * @param property The name of the OX property
     * @param userId The user identifier
     * @param contextId The context identifier
     * @param defaultValue The default value to return if the property does not exist
     * @return The integer value if the property exists, or <code>-1</code>
     */
    int getIntProperty(String property, int userId, int contextId, int defaultValue);

    /**
     * Fetches the boolean value of the specified OX property for the specified user in the specified context
     *
     * @param property The name of the OX property
     * @param userId The user identifier
     * @param contextId The context identifier
     * @param defaultValue The default value to return if the property does not exist
     * @return The boolean value if the property exists
     */
    boolean getBooleanProperty(String property, int userId, int contextId, boolean defaultValue);

    /**
     * Gets the file denoted by given file name.
     *
     * @param fileName The file name
     * @return The file or <code>null</code>
     */
    File getPropertyFile(String propertyFileName);

    /**
     * Gets the directory denoted by given directory name.
     *
     * @param directoryName The directory name
     * @return The directory or <code>null</code>
     */
    File getPropertyDirectory(String propertyDirectory);
    
    /**
     * Gets a common server configuration
     * @param host the host to get the configuration for
     * @param configKey the key of the configuration value to get 
     * @return the configuration value for the parameters
     * @throws OXException
     */
    <T> T getServerConfiguration(String host,String configKey) throws OXException;
    
    /**
     * Gets a user specific server configuration
     * @param host the host to get the configuration for
     * @param userid the id of the user to get the configuration for
     * @param cid the context of the user to get the configuration for
     * @param configKey the key of the configuration value to get
     * @return the configuration value for the parameters
     * @throws OXException
     */
    <T> T getServerConfiguration(String host, int userid, int cid, String configKey) throws OXException;

	/**
	 * Gets a string property only(!) from the "contextSet" scope
	 * 
	 * @param property     The property to get
	 * @param userId       The ID of the user to get the property for
	 * @param contextId    The CID of the user to tget the property for
	 * @param defaultValue The default value to return if the property does not
	 *                     exist
	 * @return The String property
	 * @throws OXException
	 */
	String getContextSetStringProperty(GuardProperty property, int userId, int contextId, String defaultValue) throws OXException;
}
