/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */
define('oxguard/register', [
    'io.ox/core/extensions',
    'gettext!oxguard',
    'settings!oxguard',
    'io.ox/core/capabilities',
    'less!oxguard/style'
], function (ext, gt, settings, capabilities) {
    'use strict';

    var productName = ox.serverConfig['guard.productName'] === undefined ? settings.get('productName') : ox.serverConfig['guard.productName'];
    if (productName === undefined || productName === 'undefined' || productName === '') {
        productName = 'Guard';
    }

    var oxguarddata = {
        passcode: null,
        settings:  {},
        productName: sanitize(productName)
    };

    window.oxguarddata = oxguarddata;

    // Load data generated at compile (build version, etc)
    require(['oxguard/oxguard_data'], function (generatedData) {
        _.extend(window.oxguarddata, generatedData);
    });

    // Login section
    sendauth(''); // initial login, no password yet

    // Default settings if needed
    if (settings.get('advanced') === undefined) {
        settings.set('advanced', settings.get('defaultAdvanced') ? true : false);
        settings.save();
    }

    function sendauth(pass) {
        require (['oxguard/oxguard_core'], function (oxguard) {
            oxguard.auth(ox.user_id, pass)
            .done(function (data) {
                try {
                    if (data.error) {
                        console.error(data.error);
                        console.debug('Unable to connect to Encryption Server');
                        return;
                    }
                    oxguarddata = _.extend(window.oxguarddata, data);
                    oxguarddata.loaded = true;
                    if (data.auth.length > 20) {// If good code comes back, then store userid
                        oxguarddata.userid = ox.user_id;
                        oxguarddata.passcode = data.auth;
                    } else {// If bad code, then log and mark oxguard.user_id as -1
                        if ((data.auth.indexOf('Bad') > -1) || (data.auth.indexOf('Lockout') > -1)) {
                            oxguarddata.passcode = null;
                        } else {
                            oxguarddata.passcode = data.auth;
                        }
                        if (data.auth !== 'No Key') {
                            if (settings.get('cryptoProvider') === undefined) {
                                settings.set('cryptoProvider', 'guard').save();
                            }
                        }
                        if (data.error) {
                            require(['io.ox/core/notifications'], function (notify) {
                                notify.yell('error', productName + '\r\n' + data.error);
                            });
                        }
                        oxguarddata.userid = -1;
                    }
                    if (ox.context_id === 0) ox.context_id = data.cid; // If the cid wasn't loaded, update from backend
                    oxguarddata.baton = window.oxguarddata.baton;
                    oxguarddata.recovery = data.recoveryAvail === undefined ? !data.noRecovery : data.recoveryAvail;
                    window.oxguarddata = oxguarddata;
                    // Check for any pending onload actions
                    if (oxguarddata.onload) {
                        try {
                            oxguarddata.onload.forEach(function (load) {
                                load.call();
                            });
                            oxguarddata.onload = null;
                        } catch (e) {
                            console.error('Problem executing onload functions for Guard ' + e);
                        }
                    }
                    if (sessionStorage !== null) {// If we were passed login info from login screen
                        try {
                            if (sessionStorage.oxguard && (sessionStorage.oxguard !== 'null')) {
                                var p = sessionStorage.oxguard;
                                sessionStorage.oxguard = null;
                                sendauth(p);
                            }
                        } catch (e) {
                            console.log('Private mode');
                            return;
                        }
                    }
                } catch (e) {
                    console.log(e);
                    console.debug('Unable to connect to the encryption server');
                }
            })
            .fail(function () {
                console.debug('Unable to connect to the encryption server');
            });
        });

    }

    ext.point('io.ox/core/logout').extend({
        id: 'logoutOG',
        index: 1,
        logout: function () {
            return doLogout();
        }
    });

    ox.on('relogin:required', function () {
        doLogout();
    });

    function doLogout () {
        var def = $.Deferred();
        var params = '&userid=' + ox.user_id + '&cid=' + ox.context_id + '&session=' + ox.session;
        window.oxguarddata.passcode = null;
        require (['oxguard/oxguard_core'], function (oxguard) {
            oxguard.saveAuthToSession('', -1)  // destroy auth token
            .always(function () { // Destroy Guard mapping and auth-token
                $.get(ox.apiRoot + '/oxguard/login?action=logout' + params, function () {
                    require(['io.ox/core/notifications'], function (notify) {
                        notify.yell('success', gt('Logged out of %s', productName));
                    });
                })
                .fail(function (e) {
                    require(['io.ox/core/notifications'], function (notify) {
                        notify.yell('error', gt('Failed to log out of %s', productName));
                    });
                    console.log(e);
                })
                .always(function () {
                    def.resolve();
                });
            });
        });

        return def;
    }

    ext.point('io.ox/core/topbar/right/dropdown').extend({
        id: 'guard-guided-tour',
        index: 260,
        draw: function () {
            if (_.device('small')) return;

            if (capabilities.has('guard-mail') || capabilities.has('guard-drive')) {
                this.append(
                        $('<li class="og_tour">').append(
                            //#. %s product name
                            $('<a href="#" data-action="guard-tour" role="menuitem">').text(gt('Guided tour for %s', window.oxguarddata.productName))
                        )
                        .on('click', function (e) {
                            e.preventDefault();
                            require(['io.ox/core/tk/wizard', 'oxguard/tour/main'], function (Tour) {
                                Tour.registry.run('default/oxguard');
                            });
                        })
                    );
            }
        }
    });

    ext.point('io.ox/core/topbar/right/dropdown').extend({
        id: 'logoutOG',
        index: 990,
        draw: function () {
            var dropdown = this.parent();
            dropdown.on('show.bs.dropdown', function () {
                var pass = window.oxguarddata.passcode;
                if (pass && pass.length > 10) {
                    $('.og_signout').show();
                } else {
                    $('.og_signout').hide();
                }
            });
            this.append(
                $('<li class="divider og_signout" aria-hidden="true" role="presentation"></li>'),
                $('<li class="og_signout">').append(
                    //#. %s product name
                    $('<a href="#" data-action="logout" role="menuitem">').text(gt('Sign out %s', window.oxguarddata.productName))
                )
                .on('click', function (e) {
                    e.preventDefault();
                    window.oxguarddata.passcode = null;
                    doLogout();
                })
            );
            var auth = window.oxguarddata.passcode;
            if (auth === null || auth === undefined) {
                return;
            }
        }
    });

    function sanitize (data) {
        var regex = new RegExp('<(?:!--(?:(?:-*[^->])*--+|-?)|script\\b(?:[^"\'>]|"[^"]*"|\'[^\']*\')*>[\\s\\S]*?</script\\s*|style\\b(?:[^"\'>]|"[^"]*"|\'[^\']*\')*>[\\s\\S]*?</style\\s*|/?[a-z](?:[^"\'>]|"[^"]*"|\'[^\']*\')*)>', 'gi');
        return (data.replace(regex, ''));
    }

});
