/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */
define('oxguard/pgp/publickeys', [
    'gettext!oxguard',
    'oxguard/core/og_http',
    'oxguard/oxguard_core',
    'less!oxguard/style'
], function (gt, http, core) {

    var deleting = false;

    // Change the status of a key to sharing or not
    function changeShare (e) {
        var params = '&userid=' + ox.user_id + '&cid=' + ox.context_id +
        '&ids=' + encodeURI(e.target.value) + '&share=' + ($(e.target).is(':checked') ? 'true' : 'false') +
        '&session=' + ox.session;
        wait();
        $.ajax({
            url: ox.apiRoot + '/oxguard/pgp?action=share' + params,
            type: 'PUT',
            success: function () {
                hideWait();
            }
        })
        .fail (function (data) {
            hideWait();
            console.log(data);
            fail(data.responseText);
        });
    }

    // Change the status of a key to sharing or not
    function changeInline (e) {
        var params = '&userid=' + ox.user_id + '&cid=' + ox.context_id +
        '&ids=' + encodeURI(e.target.value) + '&inline=' + ($(e.target).is(':checked') ? 'true' : 'false') +
        '&session=' + ox.session;
        wait();
        $.ajax({
            url: ox.apiRoot + '/oxguard/pgp?action=inline' + params,
            type: 'PUT',
            success: function () {
                hideWait();
            }
        })
        .fail (function (data) {
            console.log(data);
            hideWait();
            alert ('fail');
        });
    }

    // General fail call to display error message
    function fail (message) {
        require(['io.ox/core/notifications'], function (notify) {
            notify.yell('error', message);
        });
    }

    function wait () {
        $('#og_wait').show();
    }

    function hideWait () {
        $('#og_wait').hide();
    }

    function deleteConfirm (e) {
        require (['io.ox/core/tk/dialogs'], function (dialogs) {
            var dialog = new dialogs.ModalDialog({ width: 300, center: true, enter: 'ok' });
            dialog.header($('<h4>').text(gt('Delete Public Key')));
            var confirm = $('<span style="padding-left:5px; font-weight:bold;">').append(gt('Click OK to confirm'));
            dialog.getBody().append(confirm);
            dialog.addPrimaryButton('ok', gt('OK'), 'ok')
            .addButton('cancel', gt('Cancel'), 'cancel')
            .on('ok', function () {
                deleteShare (e);
            }).show();
        });
    }
    function deleteShare (e) {
        if (deleting) return;
        deleting = true;
        var trash = $(e.target);
        var tableid = trash.parent().attr('tableid');
        var list = trash.parent().attr('emaillist');
        var params = '&userid=' + ox.user_id + '&cid=' + ox.context_id +
        '&ids=' + encodeURI(trash.parent().attr('value')) +
        '&session=' + ox.session;
        trash.removeClass('fa-trash').addClass('fa-spinner fa-spin');
        core.metrics('settings', 'delete-public-key');
        $.ajax({
            url: ox.apiRoot + '/oxguard/pgp?action=share' + params,
            type: 'DELETE',
            success: function () {
                trash.removeClass('fa-spinner fa-spin').addClass('fa-trash');
                deleting = false;
                trash.parent().parent().parent().remove();
                update(tableid, list);
            }
        })
        .fail (function (data) {
            console.log(data);
            alert ('fail');
            trash.removeClass('fa-spinner fa-spin').addClass('fa-trash');
            deleting = false;
        });
        e.preventDefault();
    }

    // Guard sends seperate keys for each ID, consolidate by keyid
    function consolidate (data) {
        var newData = {};
        for (var i = 0; i < data.length; i++) {
            var id = data[i].ids.replace(/\s/g,'');
            if (newData[id] === undefined) {
                newData[id] = data[i];
            } else {
                if (newData[id].shared && !data[i].shared) {  // If this is owned and not shared, then should be used
                    newData[id] = data[i];
                }
            }
        }
        return newData;
    }
    function updateKeyTable(id, list) {
        var deferred = $.Deferred();
        var params = '&userid=' + ox.user_id + '&cid=' + ox.context_id;
        if (list !== undefined && list !== 'undefined')  {
            params += '&emails=' + encodeURI(list);
        }
        // Send the request to the oxguard server for the encrypted content key to decrypt
        http.get(ox.apiRoot + '/oxguard/pgp?action=getpublist', params)
        .done(function (data) {
            data = consolidate(JSON.parse(data));
            var showDelete = false;
            var newtable = $('<table id="keylisttable' + id + '">');
            //#. Table headers for list of keys, option for Details and to delete
            var headers = $('<th scope="col">' + gt('Email') + '</th><th scope="col">' + gt('Key Ids') + '</th><th scope="col">' + gt('Details') + '</th><th class="KeyDelete" scope="col">' + gt('Delete') + '</th>');
            if (data !== {}) {
                newtable.append(headers);
                for (var i in data) {
                    var tr = $('<tr>');
                    if (data[i].guard === true || data[i].shared === true) tr.addClass('oxguard_key_share');
                    if (data[i].expired === true) tr.css('color', 'red');
                    var email = data[i].email.replace(/>/gi,'&gt;').replace(/</gi,'&lt;');
                    var td1 = $('<th scope="row" class="oxguard_pubkeylist" title="' + email + '">');
                    td1.append(email);
                    tr.append(td1);
                    var td2 = $('<td>');
                    td2.append(data[i].ids);
                    tr.append(td2);
                    var td4 = $('<td>');
                    var search = $('<a class="pgpdetail" aria-label="' + gt('Details') + '" ids="' + data[i].ids + '" notowned="' + (((data[i].guard === true) || (data[i].shared === true)) ? 'true' : 'false') + '" href="#"><icon class="fa fa-search"/></a>');
                    td4.append(search);
                    var td5 = $('<td class="KeyDelete">');
                    if ((data[i].guard !== true) && (data[i].shared !== true)) {
                        var del = $('<a href="#" aria-label="' + gt('Delete') + '" value="' + data[i].ids + '" tableid="' + id + '" emaillist="' + list + '"><icon class="fa fa-trash"/></a>')
                        .click (deleteConfirm);
                        td5.append(del);
                        showDelete = true;
                    }
                    tr.append(td4).append(td5);
                    newtable.append(tr);
                }
                $('#keylisttable' + id).replaceWith(newtable);
                if (!showDelete) {  // If only system keys, then hide the delete column
                    $('.KeyDelete').hide();
                }
                $('.pgpdetail').unbind('click');
                $('.pgpdetail').on('click', function (e) {
                    var id = $(this).attr('ids');
                    var guard = $(this).attr('notowned');
                    showDetail(id, guard);
                    e.preventDefault();
                });
            } else {
                $('#keylisttable' + id).replaceWith('<table id="keylisttable' + id + '"><tr><td>' + gt('No Keys Found') + '</td></tr></table>');
            }
            deferred.resolve('OK');
            hideWait();
        });
        return (deferred);
    }

    function showDetail (id, guard) {
        var params = '&userid=' + ox.user_id + '&cid=' + ox.context_id + '&ids= ' + id + '&guard=' + guard;
        // Send the request to the oxguard server for the encrypted content key to decrypt
        var url = '/oxguard/pgp?action=getpubdetail';
        http.get(ox.apiRoot + url, params)
        .done(function (resp) {
            var json = JSON.parse(resp);
            var data = json.data;
            require (['io.ox/core/tk/dialogs'], function (dialogs) {
                var dialog = new dialogs.CreateDialog({ width: 450, height: 450, center: true,  enter: 'ok' });
                dialog.header($('<h4>').text(gt('Key Details')));
                var keydetail = keyDetail(data);
                dialog.getBody().append(keydetail.div);

                if ((data.length > 0) && (guard !== 'true')) { // we don't display share options for Guard created keys
                    var optionsdiv = $('<div class="keyoptions">');
                    var sharingtext = $('<label for="pgpShare">' + gt('Share Keys') + '</label>');
                    var sharing = $('<input id="pgpShare" type="checkbox" value="' + keydetail.keyids + '" style="margin-right:15px;">');
                    sharing.prop('checked', json.share > 0)
                    .on('click', changeShare);
                    sharingtext.click(function () {
                        sharing.click();
                    });
                    var inlinetext = $('<label for="pgpInline">' + gt('Use PGP Inline') + '</label>');
                    var inline = $('<input id="pgpInline" type="checkbox" value="' + keydetail.keyids + '" style="margin-right:15px; margin-left:40px;">')
                    .on('click', changeInline);
                    inline.prop('checked', json.inline);
                    inlinetext.click(function () {
                        inline.click();
                    });
                    dialog.getBody().append(optionsdiv.append(sharing).append(sharingtext).append(inline).append(inlinetext));
                }
                dialog
                .addPrimaryButton('ok', gt('OK'), 'ok')
                .addButton('sigs', gt('Signatures'), 'sigs')
                .on('sigs', function () {
                    showSignatures(keydetail.keylong);
                })
                .show(function () {
                    $('#imageID').click(function () {
                        if (keydetail.images !== null) {
                            var params = '&userid=' + ox.user_id + '&cid=' + ox.context_id + '&session=' + ox.session;
                            // Send the request to the oxguard server for the encrypted content key to decrypt
                            var url = '/oxguard/pgp?action=getkeyimage';
                            for (var i = 0; i < keydetail.images.length; i++) {
                                var imageParams = params + '&image=' + i + '&keyid=' + keydetail.images[i].keyid;
                                window.open(ox.apiRoot + url + imageParams);
                            }
                        }
                    });
                });
            });
        });
    }

    function keyDetail (data) {
        var keyids = '';
        var keylong = ''; // This will be the key that has the ids (usually master)
        var images = null;
        var resultdiv = $('<div>');
        for (var i = 0; i < data.length; i++) {
            keyids = keyids + data[i].Key + ' ';  // List of key ids for changing share, inline (linked)
            if (i > 0) resultdiv.append('<hr>');
            var detaildiv = $('<div style="padding-left:20px;">');
            //#. if key is "expired"
            if (data[i].Expired) {
                detaildiv.append('<b>' + gt('Expired') +'</b><br>');
                detaildiv.css('color', 'red');
            }
            if (data[i].Revoked) detaildiv.append('<b style="color:red">' + gt('Revoked') +'</b><br>');
            //#. Fingeprint of the key
            detaildiv.append(gt('Fingerprint:') + ' ' + formatFingerprint(data[i].Fingerprint) + '<br>');
            detaildiv.append(gt('Created:') + ' ' + data[i].Created + '<br>');
            //#. Key is a Master key
            detaildiv.append(gt('Master:') + ' ' + (data[i].Master ? gt('True') : gt('False')) + ',  ');
            //#. Key is an encryption key
            detaildiv.append(gt('Encryption:') + ' ' + (data[i].Encryption ? gt('True') : gt('False')) + '<br>');
            var ids = '';
            for (var j = 0; j < data[i].ids.length; j++) {
                if (j > 0) ids = ids + ', ';
                ids = ids + data[i].ids[j].replace(/>/gi,'&gt;').replace(/</gi,'&lt;');
            }
            if (ids !== '') {
                detaildiv.append(gt('IDs: ') + ids + '<br>');
                keylong = data[i].KeyLong;  // The ID of the key with IDs for signature verification
            }
            if (data[i].images !== undefined) {
                if (data[i].images.length > 0) {
                    images = data[i].images;
                    var image = $('<img alt="Embedded Image" id="imageID" src="' +  data[i].images[0].data + '" style="float:right;max-height:50px;max-width:50px;"/>');
                    detaildiv.append(image);
                }
            }
            if (data[i].Expires !== 'never') {
                if (data[i].Expires !== undefined)
                    detaildiv.append(gt('Expires in %1$s days', data[i].Expires) + '<br>');
            }
            resultdiv.append(detaildiv);
        }
        return {
            div: resultdiv,
            keyids: keyids,
            keylong: keylong,
            images: images
        };
    }

    function formatFingerprint (fp) {
        if (fp.length < 28) return (fp);
        var line1 = fp.substring (0,24);
        var line2 = fp.substring (25, fp.length-10) + '<b>' + fp.substring(fp.length-10) + '</b>';
        return ('<span style="float:right;">' + line1 + '</span><br/><span style="float:right">' + line2 + '</span>');
    }

    function update(id, list) {
        $('.refreshkeys').addClass('fa-spin');
        updateKeyTable(id, list).done(function () {
            $('.refreshkeys').removeClass('fa-spin');
        });
    }

    function showSignatures (id) {
        var params = '&userid=' + ox.user_id + '&cid=' + ox.context_id + '&keyid= ' + id;
        // Send the request to the oxguard server for the encrypted content key to decrypt
        var url = '/oxguard/pgp?action=getsignatures';
        http.get(ox.apiRoot + url, params)
        .done (function (data) {
            data = JSON.parse(data);
            showSigs(data);
        })
        .fail (function () {
            require(['io.ox/core/notifications'], function (notify) {
                notify.yell('error', gt('Problem getting signatures'));
            });
        });
    }

    function translate (result) {
        switch (result) {
        case 'Positive':
            //#. In table, the key was "Positively" identified by the signer (verification level)
            return (gt('Positive'));
        case 'Fail':
            //#. In table, the signature failed to be verified
            return (gt('Failed check'));
        case 'Missing Public key':
            //#. In table, there was no public key available to verify the signature
            return (gt('Missing public'));
        case 'Casual':
            //#. In table, the key was "Casually" identified by the signer (verification level)
            return (gt('Casual'));
        case 'Revoke Cert':
            //#. In table, the key was revoked with this signature
            return (gt('Revocation'));
        case 'Direct':
            //#. In table, the key was "Directly" certified by the signer (verification level)
            return (gt('Direct cert'));
        case 'Revoke':
            //#. In table, the key was revoked with this signature
            return (gt('Revocation'));
        case 'Default Cert':
            //#. In table, the key had "Default certification" by the signer (verification level)
            return (gt('Default Certification'));
        case 'Sub revoke':
            //#. In table, the key was revoked with this signature
            return (gt('Revocation'));
        default:
            return (result);
        }
    }

    function showSigs (data) {
        var table = $('<table>');
        if (data.length === 0) {
            table = $('<b>' + gt('No signatures') + '</b>');
        } else {
            //#. Table headers for the ID "signed", the level of the "certification", and who it was "signed by"
            table.append($('<th>' + gt('ID Signed') + '</th><th>' + gt('Certification') + '</th><th>' + gt('Signed by') + '</th>'));
            for (var i = 0; i < data.length; i++) {
                var tr = $('<tr>');
                var td1 = $('<td style="text-align:left;max-width:150px;overflow:hidden;text-overflow:ellipsis;">');

                if (data[i].ID === 'image') {
                    if (data[i].image !== undefined) {
                        var image = $('<img alt="Embedded Image" src="' +  data[i].image + '" style="max-height:50px;max-width:50px;"/>');
                        td1.append(image);
                    } else {
                        //#. Type of object signed was an image
                        data[i].ID = gt('Image');  // Translate image response
                        td1.append(data[i].ID);
                    }
                } else {
                    td1.append ($('<span>' + data[i].ID.replace(/>/gi,'&gt;').replace(/</gi,'&lt;') + '</span>'));
                }
                tr.append(td1);
                var td2 = $('<td>');
                td2.append (translate(data[i].result));
                tr.append(td2);
                var td3 = $('<td>');
                td3.append (data[i].certifier.replace(/>/gi,'&gt;').replace(/</gi,'&lt;'));
                tr.append(td3);
                table.append(tr);
            }
        }
        require (['io.ox/core/tk/dialogs'], function (dialogs) {
            var dialog = new dialogs.CreateDialog({ width: 450, center: true,  enter: 'ok' });
            dialog.header($('<h4>').text(gt('Key Signatures')));
            dialog.getBody().append($('<div class="oxguard_settings oxguard_signatures">').append(table));
            dialog
                .addPrimaryButton('ok', gt('OK'), 'ok')
                .show();
        });
    }

    function listPublic (id, list) {
        if (list === ',,' || list === '') return;
        var pubdiv = $('<div class="publicKeyDiv"/>');
        var tableHeader = $('<legend>');
        if (id == 'settings') tableHeader = $('<b>');
        tableHeader.append(gt('PGP Public Key List'));
        var refresh = $('<a href="#" style="margin-left:10px;" aria-label="' + gt('Refresh') + '"><icon class="fa fa-refresh fa-spin refreshkeys"></a>')
            .click(function (e) {
                $(this).addClass('fa-spin');
                updateKeyTable(id, list).done(function () {
                    refresh.removeClass('fa-spin');
                });
                e.preventDefault();
            });
        var fileinput = $('<input type="file" name="publicKey" style="display:none;" multiple>');
        var addKey = $('<a href="#" style="margin-left:10px;" aria-label="' +
            gt('Upload Public Key Only') + '"><icon class="fa fa-plus" id="addpgppublickeys"></a>')
        .click(function (e) {
            fileinput.unbind('change');
            fileinput.on('change', function () {
                var files = this.files;
                if (files.length > 0) {
                    core.metrics('settings', 'upload-public-key');
                    require(['oxguard/pgp/uploadkeys'], function (uploader) {
                        uploader.upload(files, false, '', '', list)
                        .done(function () {
                            updateKeyTable(id, list);
                            fileinput.val('');
                        });
                    });
                }
            });
            fileinput.click();
            e.preventDefault();
        });
        tableHeader.append(refresh).append(fileinput).append(addKey);
        var tablediv = $('<div class="keytable">');
        var table = $('<table id="keylisttable' + id + '">');
        pubdiv.append(tableHeader).append(tablediv.append(table));
        update(id, list);
        return (pubdiv);
    }

    return {
        listPublic: listPublic,
        showSignatures: showSignatures,
        keyDetail: keyDetail
    };
});
