/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */
define('oxguard/oxguard_core', [
    'settings!io.ox/mail',
    'settings!oxguard',
    'oxguard/core/og_http',
    'gettext!oxguard'
], function (settings, guardSettings, og_http, gt) {
    'use strict';

    var badCount = 0;

    //// Utils

    function validateEmail(email) {
        var re = /^(([^<>()[\]\\.,;:\s@\"]+(\.[^<>()[\]\\.,;:\s@\"]+)*)|(\".+\"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
        return re.test(email);
    }

    // Unified inbox combines folder and email ID, returns email id
    function getId(baton) {
        var id = decodeURIComponent(baton.data.id);
        try {
            id = id.substring(id.lastIndexOf('/') + 1) + (baton.view.cid ? baton.view.cid : '');
        } catch (e) {}
        return (id);
    }

    function metrics (app, action) {
        require(['io.ox/metrics/main'], function (metrics) {
            if (!metrics.isEnabled()) return;
            metrics.trackEvent({
                app: app,
                target: 'guard',
                type: 'click',
                action: action
            });
        });
    }

    function sanitize (data) {
        var regex = new RegExp('<(?:!--(?:(?:-*[^->])*--+|-?)|script\\b(?:[^"\'>]|"[^"]*"|\'[^\']*\')*>[\\s\\S]*?</script\\s*|style\\b(?:[^"\'>]|"[^"]*"|\'[^\']*\')*>[\\s\\S]*?</style\\s*|/?[a-z](?:[^"\'>]|"[^"]*"|\'[^\']*\')*)>', 'gi');
        return (data.replace(regex, ''));
    }

    //// Failure Handling

    function handleFailEmail (data, baton, deferred, goFunction) {
        baton.view.loaded = true;
        window.oxguarddata.priv = undefined;
        var eid = getId(baton);
        var errorResponse = {
            retry: false,
            errorMessage: 'Fail'
        };
        if (data.status == 503) {
            showError (baton, gt('Unable to connect to the encryption server.'), eid);
        } else {
            if (data.code !== undefined) {
                handleJsonError (data, errorResponse, baton);
            } else {
                var resp = data.responseText.split(':');
                switch (resp[0].trim()) {
                case 'noitem':
                    errorResponse.errorMessage = gt('Unable to find the item on the server');
                    break;
                case 'nocc':
                    errorResponse.errorMessage = gt('Unable to get the key for that item');
                    break;
                case 'nokey':
                    errorResponse.errorMessage = gt('Unable to retrieve your user keys to decode');
                    break;
                case 'Bad extra':
                    errorResponse.errorMessage = gt('Bad extra password');
                    errorResponse.retry = true;
                    break;
                case 'No Private Key associated with the Public Key':
                    errorResponse.errorMessage = gt('No private key to decode this message');
                    errorResponse.retry = false;
                    break;
                case 'Bad Password':
                    errorResponse.errorMessage = gt('Unable to decrypt Email, incorrect password.');
                    require (['io.ox/core/notifications'], function (notifications) {
                        notifications.yell('error', gt('Bad password'));
                    });
                    badCount++;
                    if (badCount > 2 && window.oxguarddata.recoveryAvail) {
                        errorResponse.errorMessage += ('<br/>' + gt('To reset or recover your password, click %s', '<a id="guardsettings" href="#">' + gt('Settings') + '</a>'));
                    }
                    errorResponse.retry = true;
                    break;
                case 'No secret key':
                    errorResponse.errorMessage = gt('No Private Key available for this encrypted item.') + (resp[1] !== undefined ? ('<br>' + gt('Key Ids:') + resp[1]) : '');
                    errorResponse.retry = false;
                    break;
                case 'Exceeded':
                    errorResponse.errorMessage = gt('Your account is locked out for %s minutes', resp[1]);
                    errorResponse.retry = false;
                    break;
                case 'Error':
                    errorResponse.errorMessage = gt('Error');
                    errorResponse.retry = false;
                    break;
                default:
                    errorResponse.errorMessage = gt('Unable to decrypt Email, incorrect password.');
                    errorResponse.retry = true;
                    break;
                }
            }

            if (data.status == 500) errorResponse.errorMessage = gt('Server Error: ') + data.responseText.trim();

        }
        if (errorResponse.retry) {
            require (['oxguard/mail/oxguard_mail_password'], function (og_mail) {
                baton.view.$el.find('#content' + eid).replaceWith(og_mail.passwordPrompt(baton, true, undefined, goFunction));
                showErrorOld (baton, errorResponse.errorMessage, eid);
                if (badCount > 2) {
                    baton.view.$el.find('#guardsettings').click(function (e) {
                        ox.launch('io.ox/settings/main', { id: 'oxguard' }).done(function () {
                            this.setSettingsPane({ id: 'oxguard' });
                        });
                        e.preventDefault();
                    });
                }
            });
            baton.view.$el.find('.body').show();
        } else {
            showErrorOld (baton, errorResponse.errorMessage, eid);
            $('.content').hide();
        }
        try {// blank out auth if present
            if (window.oxguarddata.passcode.length > 20) window.oxguarddata.passcode = null;
        } catch (ex) {}
        deferred.reject('error');
    }

    function showErrorOld (baton, error, eid) {
        baton.view.$el.find('#error' + eid).replaceWith('<div class="oxguard_error" id="error' + eid + '"><br>' + error + '</hr></div>');
    }

    function handleJsonError (error, errorResponse, baton) {
        errorResponse.errorMessage = gt('Error');
        errorResponse.retry = false;
        if (error.code !== undefined) {
            switch (error.code) {
                case 'GRD-PGP-0005':
                    if (window.oxguarddata.passcode && (window.oxguarddata.passcode.length > 20)) {  // If bad password based on auth, then clear auth and redo
                        window.oxguarddata.passcode = null;
                        if (baton) baton.view.redraw();
                        return;
                    }
                    errorResponse.errorMessage = gt('Unable to decrypt Email, incorrect password.');
                    require (['io.ox/core/notifications'], function (notifications) {
                        notifications.yell('error', gt('Bad password'));
                    });
                    badCount++;
                    if (badCount > 2 && window.oxguarddata.recoveryAvail) {
                        errorResponse.errorMessage += ('<br/>' + gt('To reset or recover your password, click %s', '<a id="guardsettings" href="#">' + gt('Settings') + '</a>'));
                    }
                    errorResponse.retry = true;
                    break;
                default:
                    if (error.error !== undefined) {
                        errorResponse.errorMessage = error.error;
                    }
            }

        }
    }

    function showError (errorJson) {
        var errorResp = {};
        try {
            handleJsonError (errorJson, errorResp);
        } catch (e) {
            errorResp.errorMessage = gt('Unspecified error');
        }
        require (['io.ox/core/notifications'], function (notifications) {
            notifications.yell('error', errorResp.errorMessage);
        });

    }

    // Parse the error JSON from an HTML returned error message
    function handleErrorHTML (error) {
        var i = error.indexOf('{');
        if (i >= 0) {
            var j = error.indexOf('}', i);
            if (j > 0) {
                var jsonString = error.substring(i, j + 1);
                var json = JSON.parse(jsonString);
                showError (json);
            }
        }
    }

    /// Authentication Section

    // Password prompt
    function getpassword (message, timeOption) {
        var def = $.Deferred();
        if ($('.passwordPrompt').length > 0) {
            def.reject('duplicate');
            return def;  // Max one password prompt box open
        }
        require (['io.ox/core/tk/dialogs', 'oxguard/core/passwords'], function (dialogs, passwords) {
            var dialog = new dialogs.CreateDialog({ width: 490, center: true,  enter: 'ok' });
            dialog.header($('<h4>').text(gt('Password needed')).addClass('passwordPrompt'));
            var noSaveWorkAround = $('<input style="display:none" type="text" name="dontremember"/><input style="display:none" type="password" name="dontrememberpass"/>');
            //#. %s product name
            var help =  $('<p>' + gt('Please enter your %s password', window.oxguarddata.productName) + '</p>');
            if (message !== undefined) {
                help = $('<p>' + message + '</p>');
            }
            var pass = passwords.createPasswordInput('password');
            dialog.getBody().append(help).append(noSaveWorkAround).append(pass);
            if (timeOption) {
                var remembertext = $('<span class="selectable">&nbsp ' + gt('Remember my password') + '</span>');
                var checkbox = $('<input type="checkbox" name="rememberpass" value="true" style="margin-top:20px;vertical-align:text-bottom;margin-bottom:1px;"/>');
                var rememberpass = $('<div>').append(checkbox);
                var duration = $('<select id="duration" class="og_duration"><option value="10">' + gt('10 minutes') + '</option>' +
                        '<option value="20">' + gt('20 minutes') + '</option>' +
                        '<option value="30">' + gt('30 minutes') + '</option>' +
                        '<option value="60">' + gt('1 hour') + '</option>' +
                        '<option value="120">' + gt('2 hours') + '</option>' +
                        '<option value="99999">' + gt('Session') + '</option' +
                        '</select>')
                        .on('change', function () {
                            $('input[name="rememberpass"]').prop('checked', true);
                        });
                rememberpass.append(remembertext).append(duration);
                remembertext.click(function () {
                    $('[name="rememberpass"]').click();
                });
                if (guardSettings.get('defRemember')) {
                    duration.val(guardSettings.get('defRemember'));
                    if (guardSettings.get('defRemember') !== '0') checkbox.prop('checked', true);
                }
                dialog.getBody().append(rememberpass);
            }
            dialog.addPrimaryButton('ok', gt('OK'), 'ok')
            .on('ok', function () {
            })
            .addButton('cancel', gt('Cancel'), 'cancel')
            .on('cancel', function () {
                def.reject('cancel');
            })
            .on('ok', function () {
                var duration = -1;
                if ($('input[name="rememberpass"]').is(':checked')) {
                    duration = $('#duration').val();
                }
                var data = {
                    password: $('#password').val(),
                    duration: duration
                };
                def.resolve(data);
            })
            .show(function () {
                window.setTimeout(function () {
                    pass.removeAttr('readonly');
                    pass.focus();
                }, 500);
            });
        });
        return (def);
    }

    // Save existing authentication token to OX Session
    function saveAuthToSession (auth, time) {
        var def = $.Deferred();
        var url = ox.apiRoot + '/guard-json?action=auth-token';
        var params = '';
        var authToken = {
            auth: auth,
            minutesValid: (time === undefined ? -1 : time)
        };
        if (auth === null) authToken = {};
        if (time !== undefined) {
            authToken.minutesValid = time;
        }
        og_http.put (url, params, authToken)
        .done (function (data) {
            def.resolve(data.auth);
        })
        .fail(function () {
            def.reject();
        });
        return def;
    }

    // Remove authentication from session
    function removeAuth() {
        window.oxguarddata.passcode = null;
        return saveAuthToSession('');
    }

    // Save a password in both auth code and token
    function savePassword (password, duration) {
        var def = $.Deferred();
        if (duration > 9999) duration = 0;  // 0 for session
        authAndSave (password, duration)
        .done(function (data) {
            def.resolve(data.data);
            storeAuth (data.data.auth, duration);
        })
        .fail(function (e) {
            def.reject(e);
            require (['io.ox/core/notifications'], function (notify) {
                switch (e.code) {
                case 'GRD-MW-0003':
                    notify.yell('error', gt('Bad password'));
                    break;
                case 'GRD-MW-0005':
                    notify.yell('error', gt('Account locked out due to bad attempts.  Please try again later.'));
                    break;
                default:
                    notify.yell('error', gt('Error checking password'));
                    break;
                }
                console.error(e);
            });
        });
        return def;
    }

    // Store the auth locally for a time
    function storeAuth (auth, duration) {
        if (duration >= 0) {
            window.oxguarddata.passcode = auth;
            var time = duration * 1000 * 60;
            if (time > 0) {  // 0 is indefinite
                window.setTimeout(function () {
                    window.oxguarddata.passcode = null;
                }, time);
            }
        }
    }

    // Check if authentication token in session and valid
    function checkAuth () {
        var def = $.Deferred();
        var url = ox.apiRoot + '/guard-json?action=auth-token';
        og_http.get (url, '')
        .done (function (data) {
            if (data.data === undefined) {
                if (window.oxguarddata.passcode && window.oxguarddata.passcode.length > 20) {
                    window.oxguarddata.passcode = null;  // wipe if contained auth, no longer valid
                }
                def.reject('none');
                return;
            }
            data = data.data;
            if (data.minutesValid !== undefined) {
                if (data.minutesValid >= -1) { // -1 is single use
                    if (data.minutesValid >= 0) { // If more than single use, store locally
                        storeAuth (data.auth, data.minutesValid);
                    }
                    def.resolve(data); // OK only if exists and still time left.
                    return;
                } else {
                    window.oxguarddata.passcode = null;
                }
            }
            def.reject('exp');
        })
        .fail (function (e) {
            console.log(e);
            def.reject('fail');
        });
        return def;
    }

    // Send password for authentication and save it to session
    function authAndSave (password, duration) {
        var def = $.Deferred();
        var userdata = {
            password: password,
            minutesValid: duration
        };
        var url = ox.apiRoot + '/guard-json?action=auth-token';
        var params = '&time=' + new Date().getTime();
        og_http.post(url, params, userdata)
        .done (function (data) {
            if (data.error) {
                def.reject(data);
                return;
            }
            def.resolve(data);
        })
        .fail (function (e) {
            def.reject(e);
        });
        return def;

    }

    // Authenticate against the Guard server, checking for specific key ID, return settings info
    function authorize(userid, password, extra, keyid)
    {
        var def = $.Deferred();
        var defaultSendAddress = $.trim(settings.get('defaultSendAddress', ''));
        var userdata = {
            user_id: userid,
            encr_password: password,
            sessionID: ox.session,
            lang: ox.language,
            cid: ox.context_id,
            email: defaultSendAddress,
            extrapass: extra ? extra : ''
        };
        var url = ox.apiRoot + '/oxguard/login?action=login';
        var params = '&time=' + new Date().getTime();
        if (keyid !== undefined) params += '&keyid=' + keyid;
        og_http.post (url, params, userdata)
        .done (function (data) {
            def.resolve(data);
        })
        .fail(function () {
            def.reject();
        });
        return def;
    }

    return {
        auth: authorize,
        validateEmail: validateEmail,
        handleFailEmail: handleFailEmail,
        getPassword: getpassword,
        showError: showError,
        sanitize: sanitize,
        handleErrorHTML: handleErrorHTML,
        saveAuthToSession: saveAuthToSession,
        checkAuth: checkAuth,
        savePassword: savePassword,
        removeAuth: removeAuth,
        metrics: metrics,
        storeAuth: storeAuth
    };
});

