define('oxguard/mail/register_compose', [
    'io.ox/core/extensions',
    'gettext!oxguard',
    'io.ox/backbone/mini-views/dropdown',
    'oxguard/mail/oxguard_mail_compose_core',
    'oxguard/util',
    'pgp_mail/toggle-encryption',
    'io.ox/backbone/mini-views/help',
    'oxguard/mail/keymanager',
    'settings!oxguard',
    'less!pgp_mail/style',
    'less!oxguard/style'
], function (ext, gt, Dropdown, core, util, ToggleEncryption, HelpView, keyman, settings) {
    'use strict';

    // Setup of compose

    ext.point('io.ox/mail/compose/fields').extend({
        id: 'lock',
        index: 'last',
        draw: function (baton) {
            if (!util.hasCryptoCapability()) {
                // TODO check for encr reply
                return;
            }
            var node = this;
            var view = new ToggleEncryption.View({
                model: baton.model
            });
            view.noLinkMail(baton.view);
            baton.view.toggleEncryption = view;
            //HACK: always insert first, since we do not control
            //other content
            node.prepend(view.render().$el);
            setupCompose (baton);
        }
    });

    function setupCompose (baton) {

        if (util.isMailvelope()) return;

        // Check the options for the page.  Lock icon settings, etc
        require (['oxguard/mail/options_new'], function (options) {
            options.createOptions(baton);
        });
        //register crypto provider choice wizard
        require(['io.ox/core/capabilities'], function (capabilities) {
            //skip if no choice is possible

            if (settings.get('cryptoProvider') !== undefined || !capabilities.has('mailvelope && guard')) return;

            baton.model.listenTo(baton.model, 'change:encrypt', function (model, val) {
                if (val !== true || settings.get('cryptoProvider') !== undefined || !capabilities.has('mailvelope && guard')) return;

                //no setup, yet. do not encrypt
                model.set('encrypt', false);
                require(['oxguard/tour/main'], function (registry) {
                    registry.get('default/oxguard/choice').get('run')().then(function () {
                        //TODO: remove this "change:encrypt" handler
                        //setup complete, encrypt!
                        model.set('encrypt', true);
                    }, function () {
                        if (ox.debug) console.log(arguments);
                    });
                });
            });
        });
    }

    // Check if attachment is PGP file, and set email to encrypt if found
    function checkEncrAttachmentAdded (val, baton) {
        if ((val.get('file_mimetype') === 'application/pgp-enrypted') ||
                (val.get('filename') && val.get('filename').indexOf('.pgp') > 0)) {
            require (['io.ox/core/capabilities'], function (capabilities) {
                if (capabilities.has('guard-mail')) {
                    baton.model.set('encrypt', true);
                }
            });
        }
    }

    // Add to attachment extensions monitor to see if attaching PGP file
    ext.point('io.ox/mail/compose/attachments').extend({
        id: 'checkPGPAttachment',
        index: 200,
        draw: function (baton) {
            baton.model.get('attachments').on('add', function (val) {
                checkEncrAttachmentAdded (val, baton);
            });
        }
    });

    ext.point('io.ox/mail/compose/menu').replace({
        id: 'security',
        index: 100,
        draw: function (baton) {
            var node = this;
            if (util.isGuardEnabled()) {
                drawDropDown (baton, $(node));
            } else {
                settings.on('change:cryptoProvider', function () {
                    if (util.isGuardEnabled()) {
                        drawDropDown (baton, $(node));
                        require (['oxguard/mail/options_new'], function (options) {
                            options.createOptions(baton);
                        });
                    }
                });
            }
        }
    });

    // Token and key handling

    ext.point('io.ox/mail/compose/createtoken').extend({
        id: 'guardToken',
        action: function (baton) {
            if (util.isGuardEnabled()) {
                if (baton.model.get('encrypt') === true) {
                    var email = baton.event.attrs.model.get('token').value;
                    var target = $(baton.event.relatedTarget);
                    var result = keyman.checkRecips (email, baton);
                    if (result !== null) {
                        core.drawKeyIcons(result, target);
                    }
                }
            }
        }
    });

    // If token removed, then remove it from our list of gkeys
    ext.point('io.ox/mail/compose/removetoken').extend({
        id: 'guardDelToken',
        action: function (baton) {
            if (baton.model.get('gkeys')) {
                var email = baton.event.attrs.model.get('token').value;
                require(['oxguard/mail/keymanager'], function (keyman) {
                    keyman.deleteRecip (email, baton);
                });
            }
        }
    });

    // Send extension points

    ext.point('io.ox/mail/compose/actions/send').extend({
        id: 'signCheck',
        index: 601,
        perform: function (baton) {
            var def = $.Deferred();
            var security = baton.model.get('security');
            if (security.sign) {
                require(['oxguard/auth'], function (auth) {
                    auth.authorize(gt('Please enter your %s password to sign this email.', window.oxguarddata.productName), true)
                    .done(function (auth) {
                        var security = baton.model.get('security');
                        security.authentication = auth;
                        baton.model.set('security', security);
                        if (baton.model.get('security').encrypt) {
                            checkKeys(baton, def);
                        } else {
                            def.resolve();
                        }
                    })
                    .fail(function () {
                        baton.stopPropagation();
                        def.reject();
                    });
                });
            } else {
                if (security.encrypt) {
                    checkKeys(baton, def);
                } else {
                    def.resolve();
                }
            }
            return (def);
        }
    });

    // Checks to see if any encrypted attachments and if auth is preset
    ext.point('io.ox/mail/compose/actions/send').extend({
        id: 'encryptedFiles',
        index: 620,
        perform: function (baton) {
            if (!baton.model.get('encrypt')) return true;
            var security = baton.model.get('security');
            var def = $.Deferred();
            var attachments = baton.model.get('attachments');
            var found = false;
            attachments.models.forEach(function (attach) {
                if (found) return;  // Only prompt once
                var mimeType = attach.get('file_mimetype');
                var fileName = attach.get('filename');
                if ((mimeType && (mimeType.indexOf('pgp-encrypted') > -1)) || (fileName && (fileName.indexOf('.pgp') > -1))) {
                    found = true;
                    require(['oxguard/auth'], function (auth) {
                        auth.authorize(gt('There appears to be an encrypted attachment.  Please enter your %s password to decode the attachment for sending.', window.oxguarddata.productName), true)
                        .done(function (auth) {
                            security.authentication = auth;
                            baton.model.set('security', security);
                            def.resolve();
                        })
                        .fail(function () {
                            def.reject();
                        });
                    });
                }
            });
            if (!found) def.resolve();
            return (def);
        }
    });

    // Check that all keys have been found prior to sending
    ext.point('io.ox/mail/compose/actions/send').extend({
        id: 'keysDoneCheck',
        index: 750,
        perform: function (baton) {
            if (!baton.model.get('encrypt')) return true;
            var def = $.Deferred();
            if (isReady(baton)) {
                def.resolve();
            } else {
                waitUntilReady(baton, def);
            }
            return (def);
        }
    });

    ext.point('io.ox/mail/compose/actions/send').extend({
        id: 'checkAuthErrors',
        index: 1100,
        perform: function (baton) {
            if (baton.result && containsError(baton.result.code, baton.result.error)) {
                handleFail(baton);
                handleReauth (gt('Please re-enter your %s password before sending.', window.oxguarddata.productName), baton)
                .done(function () {
                    baton.view.send();
                });
                return new $.Deferred().reject();
            }
        }
    });

    // Save extension points

    ext.point('io.ox/mail/compose/actions/save').extend({
        id: 'checkAuthErrors',
        index: 1100,
        perform: function (baton) {
            if (baton.result && containsError(baton.result.code, baton.result.error)) {
                handleFail(baton);
                handleReauth (gt('Please re-enter your %s password before saving.', window.oxguarddata.productName), baton)
                .done(function () {
                    baton.view.saveDraft();
                });
                return new $.Deferred().reject();
            }
        }
    });

    var reauthorizing = false;

    ext.point('io.ox/mail/compose/autosave/error').extend({
        index: 1,
        id: 'guardAuthError',
        handler: function (baton) {
            if (reauthorizing) return;
            if (containsError(baton.code, baton.error)) {
                baton.preventDefault();
                if (reauthorizing) return;
                var reauthorizing = true,
                security = baton.model.get('security');
                require(['oxguard/auth'], function (auth) {
                    auth.authorize(gt('Please re-enter your %s password.', window.oxguarddata.productName), true, true)
                    .done(function (auth) {
                        if (security) {  // Update security authorization
                            security.authentication = auth;
                            baton.view.model.set('security', security);
                        }
                        baton.view.autoSaveDraft();  // Try again
                    })
                    .always(function () {
                        reauthorizing = false;
                        baton.returnValue.reject(); // Reject the action as needed re-auth
                    });
                });

            }
        }
    });

    // Functions

    // Drawing dropdown options for security
    function drawDropDown (baton, node) {
        if (window.oxguarddata && window.oxguarddata.settings) {
            if (window.oxguarddata.settings.oxguard) {
                var dropdown = new Dropdown({ model: baton.model, label: gt('Security'), caret: true });
                if (settings.get('advanced')) {
                    dropdown
                        .header(gt('PGP'));
                } else {
                    dropdown
                    .header(gt('Security'));
                }
                dropdown
                    .option('encrypt', true, (settings.get('advanced') ? gt('Encrypt') : gt('Secure')))
                    .option('PGPSignature', true, gt('Sign'));
                if (settings.get('advanced')) {
                    dropdown
                  //#. Format of the email, HTML or Plaintext
                    .header(gt('Format'))
                    .option('PGPFormat', 'mime', gt('PGP Mime'))
                    .option('PGPFormat', 'inline', gt('PGP Inline'));
                    dropdown
                    .header(gt('Keys'))
                    .option('PubKey', true, gt('Attach my key'));
                }
                var dropdownDiv = dropdown.render().$el.addClass('text-left');
                if (_.device('small')) {
                    node.addClass('security-mobile');
                } else {
                    //  Help icon
                    dropdownDiv.find('ul').prepend(new HelpView({
                        base: 'help-guard',
                        iconClass: 'guardhelp',
                        href: 'ox.guard.user.sect.usage.email.send.html',
                        tabindex: '-1'
                    }).render().$el.css('float', 'right').css('margin-right', '10px'));
                    dropdownDiv.css('padding-left', '20px').addClass('security-options');
                }
                node.append(dropdownDiv);
            }
        }
    }

    // Check if any of the recipients are Guest and prompt for greeting/password
    function checkKeys (baton, def) {
        var keys = baton.model.get('gkeys');
        var isGuest = false;
        var fail = false;
        keys.forEach(function (k) {
            if (k.result === 'guest') isGuest = true;
            if (k.result === 'fail') fail = true;
        });
        if (fail) {
            require (['io.ox/core/notifications'], function (notifications) {
                notifications.yell('error', gt('Unable to find keys for all recipients.'));
            });
            baton.stopPropagation();
            def.reject();
            return;
        }
        if (!isGuest) {
            def.resolve();
            return;
        }
        require (['oxguard/mail/guestPrompt'], function (prompt) {
            prompt.guestOptions(baton)
            .done(function (data) {
                var security = _.extend(baton.model.get('security'), data);
                baton.model.set('security', security);
                def.resolve();
            })
            .fail(function () {
                baton.stopPropagation();
                def.reject();
            });
        });
    }

    // Check all keys ready
    function isReady (baton) {
        function checkKey (key) {
            return (!key.pending);
        }
        var keys = baton.model.get('gkeys');
        if (keys === undefined) return true; // No keys
        return (keys.every(checkKey));
    }

    // Check if response code and error string contain auth error
    function containsError (code, string) {
        if (code === 'GRD-MW-0001') return true;  // Midleware Auth error
        if (code !== 'GUARD-0003') return false;
        return (string && string !== null &&
            (string.indexOf('GRD-AUTH-0002') > -1 || string.indexOf('GRD-PGP-0005') > -1));
    }

    function handleFail(baton) {
        if (baton.error && !baton.warning) {
            baton.stopPropagation();
            var win = baton.app.getWindow();
            if (win) {
                // reenable the close button in toolbar
                if (baton.closelink) {
                    baton.closelink.show();
                }
                win.idle().show();
            }
            baton.app.launch();
            return;
        }
        return;
    }

    function handleReauth(prompt, baton) {
        window.oxguarddata.passcode = null;  // wipe stored auth token
        var def = $.Deferred();
        require(['oxguard/auth'], function (auth) {
            var security = baton.model.get('security');
            auth.authorize(prompt, true, true)  // authorize and force relogin
            .done(function (auth) {
                security.authentication = auth;
                baton.model.set('security', security);
                baton.stopPropagation();
                def.resolve();
            })
            .fail(function () {
                baton.stopPropagation();
                def.reject(true);
            });
        });
        return def;
    }

    // If not all keys returned, wait until completed.
    // Timeout if no backend response
    function waitUntilReady (baton, def) {
        baton.waiting = true;
        var win = baton.view.app.getWindow();
        win.busy();
        baton.model.on('change:gkeys', function () {
            if (isReady(baton)) {
                win.idle();
                def.resolve();
                baton.waiting = false;
            }
        });
        window.setTimeout(function () {
            if (baton !== undefined) {
                if (baton.waiting === true) {
                    win.idle();
                    baton.waiting = false;
                    def.resolve();
                }
            }
        }, 7000);
    }

});
