/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */

define('oxguard/mail/keymanager', ['oxguard/core/og_http', 'gettext!oxguard', 'settings!io.ox/calendar'],
    function (og_http, gt, calSettings) {
    'use strict';

    var blackList;   // We only want to pull the blacklist once

    // Check the list of previously fetched key results to see if the email is there.  If so, return, otherwise query backend
    function checkRecips (email, baton) {
        var gkeys = baton.model.get('gkeys');
        if (gkeys === undefined) { // If undefined, create gkeys now
            gkeys = [];
            baton.model.set('gkeys', gkeys);
            getKeyResult (email, baton);
            return;
        }
        var result;
        gkeys.forEach(function (key) {
            if (key.email === email) {  // If email matches, and resulted, then return result
                if (key.pending === true) {
                    result = null;
                } else {
                    result = key.result;
                }
                return (false);
            }
        });
        if (result !== undefined) return (result);
        getKeyResult (email, baton);
        return (null);
    }

    // Query guard server for key result
    function getKeyResult (email, baton) {
        var result = {
            email: email,
            result: null,
            pending: true
        };
        baton.model.get('gkeys').push(result);
        if (isBlackListed(email)) {
            require (['io.ox/core/notifications'], function (notifications) {
                notifications.yell('error', gt('Mailing list / blocked email found.  Encrypted emails to this email will fail.  %s', email));
            });
            updateKeys (email, baton, 'fail');
            return;
        }
        var params = '&email=' + email + '&cid=' + ox.context_id + '&from=' +
        encodeURIComponent(baton.model.get('from')[1]) + '&userid=' + ox.user_id;
        og_http.get(ox.apiRoot + '/oxguard/pgpmail?action=getkey', params)
        .done(function (data) {//Get the key
            try {
                if ($.isArray(data)) {  // error responses sent as array
                    data = data[0];
                }
                if (data.inline) { // If needs inline, then set
                    if (baton.model.set('PGPFormat', 'inline'));
                }
                if (data.error) {
                    updateKeys (email, baton, 'fail');
                    require (['io.ox/core/notifications'], function (notifications) {
                        if (data.error)
                            notifications.yell('error', data.error);
                    });
                    return;
                }
                if (data.expired === true) {
                    updateKeys (email, baton, 'fail');
                    require (['io.ox/core/notifications'], function (notifications) {
                        notifications.yell('error', gt('The key for %s has expired.  You will not be able to send an encrypted email to this recipient.', data.email));
                    });
                    return;
                }
                if (data.pgp === true) {
                    updateKeys (email, baton, 'pgp');
                    return;
                }
                if (data.guest === true) {
                    updateKeys (email, baton, 'guest');
                }
            } catch (e) {
                console.log(e);
            }
        })
        .fail (function () {
            updateKeys (email, baton, 'fail');
        });
    }

    // Check if email is in blacklist
    function isBlackListed (email) {
        if (!blackList) {  // Initialize blacklist
            blackList = calSettings.get('participantBlacklist');
            if (blackList) {
                blackList = blackList.toLowerCase();
            } else {
                blackList = '';
            }
        }
        if (blackList && blackList !== '') {
            if (blackList.indexOf(email.toLowerCase()) > -1) {
                return true;
            }
        }

        return false;
    }

    // Update the list of gkeys with the result
    function updateKeys (email, baton, result) {
        var gkeys = baton.model.get('gkeys');
        gkeys.forEach(function (key) {
            if (key.email === email) {
                key.result = result;
                key.pending = false;
            }
        });
        baton.model.set('gkeys', gkeys);
        baton.model.trigger('change:gkeys');
        updateTokens(baton);
    }

    function updateTokens(baton) {
        baton.model.trigger('change:to', baton.model, baton.model.get('to'));
        baton.model.trigger('change:cc', baton.model, baton.model.get('cc'));
        baton.model.trigger('change:bcc', baton.model, baton.model.get('bcc'));
    }

    function checkAll(baton) {
        updateTokens(baton);
    }

    // Delete specific email from gkeys array
    function deleteRecip (email, baton) {
        var gkeys = baton.model.get('gkeys');
        var found = -1;
        gkeys.forEach(function (key, i) {
            if (key.email === email) {
                found = i;
            }
        });
        if (found > -1) {
            gkeys.splice(found, 1);
            baton.model.set('gkeys', gkeys);
        }
    }

    // Check if an email is located in gkeys
    function isPresent (email, recips) {
        for (var i = 0; i < recips.length; i++) {
            if (recips[i][1] === email) return true;
        }
        return false;
    }

    // Clean the list of gkeys, remove tokens that were deleted
    function clean(baton) {
        var to = baton.model.get('to') || [];
        var cc = baton.model.get('cc') || [];
        var bcc = baton.model.get('bcc') || [];
        var total = to.concat(cc).concat(bcc);
        var gkeys = baton.model.get('gkeys') || [];
        var newkeys = [];
        for (var i = 0; i < gkeys.length; i++) {
            if (isPresent(gkeys[i].email, total)) newkeys.push(gkeys[i]);
        }
        baton.model.set('gkeys', newkeys);
    }

    // Check if Guest present in list
    function isGuest(baton) {
        try {
            clean(baton);
            var keys = baton.model.get('gkeys');
            if (keys === undefined) keys = [];
            for (var i = 0; i < keys.length; i++) {
                if (keys[i].result === 'guest') return true;
            }
        } catch (e) {
            console.log(e);
        }

        return false;
    }

    function getGuests(baton) {
        var guests = [];
        try {
            var keys = baton.model.get('gkeys');
            if (keys === undefined) keys = [];
            for (var i = 0; i < keys.length; i++) {
                if (keys[i].result === 'guest') guests.push(keys[i].email);
            }
        } catch (e) {
            console.log(e);
        }
        return (guests);
    }

    return {
        checkRecips: checkRecips,
        checkAll: checkAll,
        clean: clean,
        isGuest: isGuest,
        getGuests: getGuests,
        deleteRecip: deleteRecip
    };

});
