/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */
define('oxguard/core/createKeys', [
    'oxguard/core/og_http',
    'gettext!oxguard'
], function (og_http, gt) {
    'use strict';

    var CreateKeysViewModel = Backbone.Model.extend({
        initialize: function () {
            //FIXME: reduce callback hell a little
            if (window.oxguarddata.username === undefined) {
                require(['io.ox/core/api/user']).done(function (user) {
                    window.oxguarddata.username = user.getName().done(function (data) {
                        window.oxguarddata.username = data;
                    });
                });
            }
        },
        defaults: {
            email: '',
            initialSetup: false,
            sent: false
        },
        toJSON: function () {
            var userdata = {
                user_id: ox.user_id,
                sessionID: ox.session,
                language: ox.language,
                cid: ox.context_id
            };

            if (window.oxguarddata.username !== undefined) userdata.name = window.oxguarddata.username;

            return _.extend(userdata, this.pick('email', 'password'));
        },
        send: function () {
            var self = this;
            this.trigger('before:send');
            if (!this.isValid()) {
                this.trigger('send send:error', this.validationError);
                return;
            }
            og_http.post(ox.apiRoot + '/oxguard/login?action=create', (this.get('initialSetup') ? '' : '&add=true'), this.toJSON())
            .done(function (data) {
                if (data && data.result === 'ok') {
                    window.oxguarddata.passcode = null;
                    window.oxguarddata.recoveryAvail = window.oxguarddata.recovery = !window.oxguarddata.settings.noRecovery;
                    self.trigger('send send:ok');
                    self.set('sent', true);
                    if (window.oxguarddata.recovery) window.oxguarddata.recoveryAvail = true;
                } else {
                    self.trigger('send send:error', gt('Problems creating keys, please try again later.'));
                }
            })
            .fail(function (data) {
                var error;
                if (data.responseText.trim() === 'Existing') {
                    error = gt('Problems creating keys.  Keys already exist for this email address in another account.');
                } else {
                    error = gt('Problems creating keys, please try again later.');
                }
                self.trigger('send send:error', error);
            });
        },
        validate: function (attrs) {
            var errors = [];
            var minlen = window.oxguarddata.settings.min_password_length;
            if (minlen !== undefined && attrs.password !== undefined && attrs.password.length < minlen) {
                errors.push({
                    field: '.password',
                    message: gt('Passwords must be at least %s characters in length', minlen)
                });
            }
            if (attrs.password !== attrs.passwordValidation) {
                errors.push({
                    field: '.password-validation',
                    message: gt('Passwords not equal')
                });
            }
            if ((attrs.email.length > 1) && (!validateEmail(attrs.email))) {
                errors.push({
                    field: '.email',
                    message: gt('Enter new secondary Email address')
                });
            }
            return errors.length === 0 ? undefined : errors;
        },
        //handle password strength warnings (those not catched by validate) - adopted from oxquard/core/passwords.passwordCheck
        checkStrength: function () {
            var min = window.oxguarddata.settings.min_password_length;
            var len = window.oxguarddata.settings.password_length;
            if (min === undefined) min = 6;
            if (len === undefined) len = 10;

            if (len <= min) len = min + 1;
            if (!this.get('password')) return;

            if (this.get('password').length < min) {
                this.set('passwordStrength', 'bad');
                return;
            }

            var regex = /(?=^.{8,}$)(?=.*\d)(?=.*[!@#$%^&*]+)(?![.\n])(?=.*[A-Z])(?=.*[a-z]).*$/;
            if (this.get('password').match(regex)) {
                this.set('passwordStrength', 'good');
                return;
            }

            this.set('passwordStrength', 'weak');
        }
    });

    var CreateKeysView = Backbone.View.extend({
        className: 'guard-create-key form-horizontal',
        initialize: function () {
            if (!this.model) this.model = new CreateKeysViewModel();
            this.listenTo(this.model, 'change', this.handleChange);
            this.listenTo(this.model, 'before:send', this.busy);
            this.listenTo(this.model, 'send', this.idle);
            this.listenTo(this.model, 'send:error', this.handleError);

            this.$wait = $('<div class="og_wait" id="keygen">').append(
                $('<i class="fa-key fa icon-key fa-spin icon-spin"/>'),
                $('<span>').text(gt('Generating key, Please wait'))
            ).hide();
        },

        events: {
            'keyup input[name=newogpassword]': 'onPasswordChange',
            'keyup input[name=newogpassword2]': 'onPasswordValidationChange',
            'change input[name=recoverymail]': 'onRecoverymailChange'
        },
        onPasswordChange: function (ev) {
            this.model.set('password', ev.target.value);
        },
        onPasswordValidationChange: function (ev) {
            this.model.set('passwordValidation', ev.target.value);
        },
        onRecoverymailChange: function (ev) {
            this.model.set('email', ev.target.value);
        },

        handleChange: function (model) {
            var $el = this.$el;
            //reset error fields
            $el.find('.has-error')
                .removeClass('has-error')
                .find('.error-msg').empty();

            if (!model.isValid()) {
                model.validationError.forEach(function (error) {
                    $el.find(error.field)
                        .addClass('has-error')
                        .find('.error-msg').text(error.message);
                });
            }
            model.checkStrength();
            var strength = model.get('passwordStrength');
            $el.find('.password')
                .toggleClass('has-success', strength === 'good')
                .toggleClass('has-warning', strength === 'weak');
        },
        handleError: function (error) {
            require (['io.ox/core/notifications'], function (notify) {
                notify.yell('error', error);
            });
        },
        busy: function () {
            this.$wait.show();
        },
        idle: function () {
            this.$wait.hide();
        },

        renderDescription: function (el) {
            el.append($('<div>').append(
                this.model.get('initialSetup') ? $('<p>').text(gt('Please choose the password you will use for %s. You will need to type this password whenever you want to encrypt or decrypt items. Remember it should be different from your login password, and will not change if you change your login password.', window.oxguarddata.productName)) : '',
                $('<p>').text(gt('Please enter a password to protect your new encrypted items.'))
            ));
            return this;
        },
        renderPasswordPrompt: function (el) {
            el.append(
                $('<div>').addClass('form-group password').css('margin-right', '0px').append(
                    $('<label>')
                        .addClass('col-sm-12 col-md-4')
                        .text(gt('Password')),
                    $('<div>').addClass('col-sm-12').append(
                        $('<input id="newogpassword" name="newogpassword" type="password" readonly autocomplete="off" onfocus="this.removeAttribute(\'readonly\');"/>').addClass('form-control')
                    ),
                    $('<div>').addClass('col-sm-12 error-msg')
                ),
                $('<div>').addClass('form-group password-validation').css('margin-right', '0px').append(
                    $('<label>')
                        .addClass('col-sm-12 col-md-4')
                        .text(gt('Confirm')),
                    $('<div>').addClass('col-sm-12').append(
                        $('<input name="newogpassword2" type="password" autcomplete="off" readonly onfocus="this.removeAttribute(\'readonly\');"/>').addClass('form-control')
                    ),
                    $('<div>').addClass('col-sm-12 error-msg')
                )
            );
            window.setTimeout(function () {
                if (_.device('desktop')) {
                    $('#newogpassword').focus();
                } else {
                    $('[type="password"]').removeAttr('readonly');  // Remove the read only but do not focus.  User needs to click for keyboard
                }
            }, 1000);
            return this;
        },
        renderRecoverMailPrompt: function (el) {
            if (window.oxguarddata.settings.noRecovery === true) {
                el.append($('<hr style="padding:10px;"/>'),
                        $('<p class="oxguard_warning">').text(gt('Warning: This password for encryption cannot be restored or recovered in any way.  If forgotten, all encrypted data will be lost')));
                return this;  // If no recovery set, don't add prompt for second password
            }
            if (this.model.get('initialSetup')) {
                el.append(
                    $('<hr style="padding:10px;"/>'),
                    $('<p>').text(gt('Please enter a secondary email address in case you need to reset your password.')),
                        $('<div>').addClass('form-group email').css('margin-right', '0px').append(
                        $('<label>').addClass('col-sm-12 col-md-4').text(gt('Email:')),
                        $('<div>').addClass('col-sm-12').append(
                            $('<input name="recoverymail">').addClass('form-control')
                        ),
                        $('<div>').addClass('col-sm-12 error-msg')
                    )
                );
            }
            return this;
        },
        render: function () {
            this.$el.empty().append(this.$wait);
            return this
                .renderDescription(this.$el)
                .renderPasswordPrompt(this.$el)
                .renderRecoverMailPrompt(this.$el);
        }
    });

    function createKeysWizard(guardOnly) {
        var def = $.Deferred();
        require(['io.ox/core/tk/wizard', 'oxguard/tour/main'], function (Tour) {
            if (guardOnly) {
                Tour.registry.get('default/oxguard/createKeys').get('run')().then(def.resolve, def.reject);
            } else {
                Tour.registry.get('default/oxguard/choice').get('run')().then(def.resolve, def.reject);
            }
        });
        return def;
    }

    /**
     * @returns deferred object resolves to one of:
     *  * "OK" - everything is fine
     *  * "cancel" - user cancelled the dialog
     *  * object - failed with error message stored in responseText attribute
     */
    // Prompt for creating master keys for this user if not yet created.
    function createKeys() {
        var def = $.Deferred(),
            view = new CreateKeysView();

        require (['io.ox/core/notifications', 'io.ox/core/tk/dialogs'], function (notify, dialogs) {

            var dialog = new dialogs.CreateDialog({ width: 450, center: true, enter: 'ok', async: true });
            //#. %s product name
            dialog.header($('<h4>').text(gt('Create %s Security Keys', window.oxguarddata.productName)));
            dialog.getBody().append(view.render().$el);

            dialog
            .addPrimaryButton('ok', gt('OK'), 'ok')
            .addButton('cancel', gt('Cancel'), 'cancel')
            .on('cancel', function () {
                def.reject('cancel');
            })
            .on('ok', function () {
                view.model.once('send', function (error) {
                    if (error) {
                        def.reject({
                            responseText: error
                        });
                        return;
                    }
                    def.resolve('OK');
                });
                view.model.send();
            })
            .show(function () {  // Normalize the two tables to have same column size
                if (!_.device('ios')) {
                    $('input[name="newogpassword"]').focus();
                }
            });
            view.model.on('change', function (model) {
                var el = dialog.getFooter().find('button[data-action=ok]');
                if (model.isValid()) {
                    el.removeAttr('disabled');
                } else {
                    el.attr('disabled', true);
                }
            });
            def.always(function () {
                view.remove();
                view.model.off();
                view.model = null;
                view = null;
                dialog.close();
            });
        });
        return def;
    }

    function validateEmail(email) {
        var re = /^(([^<>()[\]\\.,;:\s@\"]+(\.[^<>()[\]\\.,;:\s@\"]+)*)|(\".+\"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
        return re.test(email);
    }

    return {
        createKeys: createKeys,
        createKeysWizard: createKeysWizard,
        CreateKeysViewModel: CreateKeysViewModel,
        CreateKeysView: CreateKeysView
    };

});
