/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * author Greg Hill <greg.hill@open-xchange.com>
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */
define('oxguard/auth', ['oxguard/oxguard_core', 'oxguard/util',
                        'gettext!oxguard'], function (core, util, gt) {
    'use strict';

    var auth_core = {};

    // Perform auth against Guard server
    // Return auth code if any
    auth_core.authorize = function authorize(prompt, minSingleUse, forceRelogin) {
        var def = $.Deferred();
        ensureSetup().then(function () {
            if (forceRelogin) {  // If we must relogin, show password prompt
                promptPassword (prompt, minSingleUse, def);
            } else { // Else check if we have authentication already in session
                core.checkAuth()
                .done(function (data) {
                    def.resolve(data.auth);
                })
                .fail(function () {
                    promptPassword (prompt, minSingleUse, def);
                });
            }
        });
        return def;
    };

    // Display password prompt to authorize
    function promptPassword (prompt, minSingleUse, def) {
        if (prompt === undefined) {
            prompt = gt('Enter %s security password', window.oxguarddata.productName);
        }
        core.getPassword(prompt, true)  //Get password
        .done(function (passdata) {
            if (passdata.duration > -1 || minSingleUse) {
                core.savePassword(passdata.password, passdata.duration)
                .done(function (data) {
                    def.resolve(data.auth);
                })
                .fail(function (data) {
                    handleFail(data.auth);
                    def.reject(data.auth);
                });
            } else {
                core.auth(ox.user_id, passdata.password)
                .done(function (data) {
                    if (data.auth.length > 20) {
                        def.resolve(data.auth);
                    } else {
                        handleFail(data.auth);
                        def.reject(data.auth);
                    }
                });
            }
        })
        .fail(function () {
            def.reject('cancel');
        });
    }

    function ensureSetup() {

        if (!util.isGuardConfigured()) {
            var def = $.Deferred();
            require (['oxguard/core/createKeys'], function (keys) {
                core.metrics('drive', 'create-keys');
                keys.createKeysWizard(true)
                .then(def.resolve, def.reject);
            });
            return def;
        }

        // If keys were set up by receiving email, possible user hasn't had prompt to create password.
        if (window.oxguarddata.passcode === 'Password Needed') {
            require(['oxguard/core/tempPassword'], function (core) {
                var go = function () {
                    return;
                };
                core.createOxGuardPasswordPrompt(null, go);
            });
            return $.Deferred().reject();
        }

        return $.when();
    }

    function handleFail (auth) {
        require (['io.ox/core/notifications'], function (notifications) {
            switch (auth.toLowerCase()) {
            case 'bad password':
                notifications.yell('error', gt('Bad password'));
                return;
            case 'lockout':
                notifications.yell('error', gt('Account Locked out'));
                return;
            default:
                notifications.yell('error', gt('Unspecified error'));
                return;
            }
        });
    }

    return auth_core;
});
