/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.pgp;

import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.storage.KeyTableStorage;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.pgp.PgpHandler;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.GuardUserSession;

/**
 * Retrieves all Keys for a user
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class GetUserKeysAction extends GuardServletAction {

    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception {

        OxCookie cookie = new OxCookie(request.getCookies());
        Api ap = new Api(cookie, request);
        if (ap.verifyLogin()) {
            ServletUtils.sendHtmlOK(response, getAllKeys(userSession.getUserId(), userSession.getContextId()).toString());
        } else {
            ServletUtils.sendNotAcceptable(response, "Must be logged in");
        }
    }

    private JsonArray getAllKeys(int id, int cid) throws Exception {
        KeyTableStorage keyTableStorage = Services.getService(KeyTableStorage.class);
        List<GuardKeys> keys = keyTableStorage.getKeysForUser(id, cid);
        JsonArray keysData = new JsonArray();
        for (int i = 0; i < keys.size(); i++) {
            GuardKeys key = keys.get(i);
            JsonArray pubKeyData = PgpHandler.getKeyInfo(key.getPGPPublicKeyRing());
            if (key.getPGPSecretKeyRing() != null) {  // If we have private key data, see if associated with each public
                JsonArray privData = PgpHandler.getPrivateInfo(key.getPGPSecretKeyRing());
                for (int j = 0; j < pubKeyData.size(); j++) {
                    JsonObject pubkey = pubKeyData.get(j).getAsJsonObject();
                    String keyid = pubkey.get("KeyID").getAsString();
                    boolean hasPrivate = false;
                    for (int k = 0; k < privData.size(); k++) {
                        if (privData.get(k).getAsJsonObject().get("KeyID").getAsString().equals(keyid)) {
                            hasPrivate = true;
                            continue;
                        }
                    }
                    if (hasPrivate) {
                        pubkey.addProperty("hasPrivate", true);
                    } else {
                        pubkey.addProperty("hasPrivate", false);
                    }
                    if (key.isCurrent()) {
                        pubkey.addProperty("current", true);
                    }
                    pubkey.addProperty("GuardKey", Long.toString(key.getKeyid()));
                }
            } else {  // No private key, just public
                for (int j = 0; j < pubKeyData.size(); j++) {
                    JsonObject pubkey = pubKeyData.get(j).getAsJsonObject();
                    if (key.isCurrent()) {
                        pubkey.addProperty("current", true);
                    }
                    pubkey.addProperty("GuardKey", Long.toString(key.getKeyid()));
                    pubkey.addProperty("hasPrivate", false);

                }
            }

            keysData.add(pubKeyData);
        }
        return (keysData);

    }
}
