/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.keyretrieval;

import java.util.ArrayList;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.exception.OXException;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.common.util.JsonUtil;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.handler.ResponseHandler;
import com.openexchange.guard.inputvalidation.EmailValidator;
import com.openexchange.guard.internal.UserData;
import com.openexchange.guard.keymanagement.commons.DeletedKey;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.export.KeyExportUtil;
import com.openexchange.guard.keymanagement.storage.DeletedKeysStorage;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.session.GuardSessionService;

/**
 * Class for handling retrieving deleted keys
 *
 * @author greg
 *
 */
public class KeyRetriever {

    private static Logger LOG = LoggerFactory.getLogger(KeyRetriever.class);

    /**
     * Initial login for key retrieval. Password verified. Salt verified
     * Keys listed in response
     *
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void login(HttpServletRequest request, HttpServletResponse response, OxCookie cookie) throws Exception {
        JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
        int userid = JsonUtil.getIntFromJson(json, "userid", true);
        int cid = JsonUtil.getIntFromJson(json, "cid", true);
        final String email = new EmailValidator().assertInput(JsonUtil.getStringFromJson(json, "username", true), "username");
        String salt = JsonUtil.getStringFromJson(json, "salt", true);
        String password = JsonUtil.getStringFromJson(json, "password", true);
        JsonObject returnJson = new JsonObject();
        GuardAntiAbuseService antiabuse = Services.getService(GuardAntiAbuseService.class);
        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
        if (antiabuse.isBad("retr-" + email, configService.getIntProperty(GuardProperty.badPasswordCount))) {
            LOG.warn("Account lockout for key retrieval");
            sendBad(response, "lockout");
            return;
        }

        DeletedKeysStorage deletedKeysStorage = Services.getService(DeletedKeysStorage.class);
        ArrayList<DeletedKey> keys = deletedKeysStorage.getAllExposedForEmail(email, userid, cid);
        if (keys.size() == 0) {
            sendBad(response, "not found");
            return;
        }
        if (!keys.get(0).verifyPassword(password)) {
            antiabuse.addBad("retr-" + email);
            LOG.warn("Bad password attempt for key retrieval");
            sendBad(response, "Bad Password");
            return;
        }

        // Check salt matches at least on key
        boolean match = false;
        int i = 0;
        while ((i < keys.size()) && !match) {
            if (keys.get(i++).getSalt().equals(salt.trim())) {
                match = true;
            }
        }

        if (!match) {
            antiabuse.addBad("retr-" + email);
            sendBad(response, "bad salt");
            return;
        }

        // Ok, key password and salt verified, send data

        // Add the authorization data here
        LOG.info("Sending retrieval key list for user " + email);
        JsonObject authdata = new JsonObject();
        GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_GCM);
        GuardSessionService sessionService = Services.getService(GuardSessionService.class);
        String token = sessionService.getToken(cookie.getOxReaderID());
        if (token == null) {
            token = sessionService.newToken(cookie.getOxReaderID(), userid, cid);
        }
        authdata.addProperty("encr_password", password);
        authdata.addProperty("user_id", userid);
        authdata.addProperty("cid", cid);
        String auth = cipherService.encrypt(authdata.toString(), token);
        returnJson.addProperty("auth", auth);
        returnJson.add("keys", getKeyData(keys));
        ServletUtils.sendJsonOK(response, returnJson);
        return;

    }

    /**
     * Retrieve a selected key by index
     * Must have auth for primary key
     *
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void getkeys(HttpServletRequest request, HttpServletResponse response, OxCookie cookie) throws Exception {
        JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
        final String email = new EmailValidator().assertInput(JsonUtil.getStringFromJson(json, "username", true), "username");
        String auth = ServletUtils.getStringParameter(request, "auth", true);
        GuardAntiAbuseService antiabuse = Services.getService(GuardAntiAbuseService.class);
        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
        if (antiabuse.isBad("retr-" + email, configService.getIntProperty(GuardProperty.badPasswordCount))) {
            ServletUtils.sendNotAcceptable(response, "lockout");
            return;
        }
        cookie.setJSESSIONID(cookie.getOxReaderID());
        UserData userdata = new UserData(auth, cookie);
        if (userdata.getUserid() < 0) {
            ServletUtils.sendNotAcceptable(response, "Bad auth");
            antiabuse.addBad("retr-" + email);
            LOG.warn("Bad password attempt for key retrieval");
            return;
        }
        int index = JsonUtil.getIntFromJson(json, "index", true);
        String password = JsonUtil.getStringFromJson(json, "password");
        if (password == null) {
            password = "";
        }
        DeletedKeysStorage deletedKeysStorage = Services.getService(DeletedKeysStorage.class);
        ArrayList<DeletedKey> keys = deletedKeysStorage.getAllExposedForEmail(email, userdata.getUserid(), userdata.getCid());
        if (keys.size() == 0) {
            ServletUtils.sendNotAcceptable(response, "not found");
            return;
        }
        if (index > keys.size()) {
            ServletUtils.sendNotAcceptable(response, "out of range");
            return;
        }
        // Verify first password is correct.
        if (!keys.get(0).verifyPassword(userdata.getEncr_password())) {
            antiabuse.addBad("retr-" + email);
            ServletUtils.sendNotAcceptable(response, "Bad Password");
            LOG.warn("Bad password attempt for key retrieval");
            return;
        }
        String export;
        try {
            DeletedKey deletedKey = keys.get(index);
            GuardKeys key = new GuardKeys(deletedKey.getUserId(), deletedKey.getCid(), deletedKey.getSecretKeyRing(), deletedKey.getSalt());
            export = KeyExportUtil.exportPGPPrivateKey(key, password.equals("") ? userdata.getEncr_password() : password);
        } catch (OXException e) {
            ServletUtils.sendNotAcceptable(response, "bad password");
            antiabuse.addBad("retr-" + email);
            return;
        }
        LOG.info("Exported previously deleted key for " + email);
        JsonObject returnJson = new JsonObject();
        returnJson.add("keydata", getData(keys.get(index)));
        returnJson.addProperty("data", export);
        ServletUtils.sendJsonOK(response, returnJson);
    }

    /**
     * Send a json response back with auth set to resp value
     *
     * @param response
     * @param resp
     */
    private void sendBad(HttpServletResponse response, String resp) {
        JsonObject returnJson = new JsonObject();
        returnJson.addProperty("auth", resp);
        ServletUtils.sendJsonOK(response, returnJson);
        return;
    }

    /**
     * Get key data from arraylist
     *
     * @param keys
     * @return
     */
    private JsonArray getKeyData(ArrayList<DeletedKey> keys) {
        JsonArray keyArray = new JsonArray();
        for (int i = 0; i < keys.size(); i++) {
            JsonObject keyJson = getData(keys.get(i));
            keyJson.addProperty("index", i);
            keyArray.add(keyJson);
        }
        return (keyArray);
    }

    /**
     * Get the data from a deleted key
     *
     * @param key
     * @return
     */
    private JsonObject getData(DeletedKey key) {
        JsonObject keyJson = new JsonObject();
        ArrayList<String> userids = key.getUserIds();
        JsonArray uids = new JsonArray();
        for (String uid : userids) {
            uids.add(new JsonPrimitive(uid));
        }
        keyJson.add("ids", uids);
        ArrayList<String> fingerprints = key.getKeyFingerprints();
        JsonArray fpArray = new JsonArray();
        for (String fp : fingerprints) {
            fpArray.add(new JsonPrimitive(fp));
        }
        keyJson.add("fingerprints", fpArray);
        return (keyJson);
    }

}
