/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgp;

import java.security.SignatureException;
import java.util.Date;
import java.util.Iterator;
import org.apache.commons.net.util.Base64;
import org.bouncycastle.bcpg.attr.ImageAttribute;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPUserAttributeSubpacketVector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.guard.common.util.LongUtil;
import com.openexchange.guard.keymanagement.commons.util.PGPUtil;

public class PgpHandler {

    private static Logger LOG = LoggerFactory.getLogger(PgpHandler.class);

    /**
     * Return JsonArray of info for PGPSecretKeyRing
     *
     * @param skeyring
     * @return
     */
    public static JsonArray getPrivateInfo(PGPSecretKeyRing skeyring) {
        if (skeyring == null) {
            return (null);
        }
        Iterator skeys = skeyring.getSecretKeys();
        JsonArray keyRingData = new JsonArray();
        while (skeys.hasNext()) {
            PGPSecretKey key = (PGPSecretKey) skeys.next();
            JsonObject pubKey = new JsonObject();
            pubKey.addProperty("Key", LongUtil.longToHexStringTruncated(key.getKeyID()));
            pubKey.addProperty("KeyID", LongUtil.longToHexString(key.getKeyID()));
            pubKey.addProperty("Master", key.isMasterKey());
            Iterator it = key.getUserIDs();
            JsonArray ids = new JsonArray();
            while (it.hasNext()) {
                ids.add(new JsonPrimitive((String) it.next()));
            }
            pubKey.add("ids", ids);
            keyRingData.add(pubKey);
        }
        return (keyRingData);
    }

    /**
     * Print the user info for a PGP public key ring with userid escaped for display
     * @param pkey
     * @return
     * @throws PGPException
     * @throws SignatureException
     */
    public static JsonArray getKeyInfo(PGPPublicKeyRing pkey) throws PGPException, SignatureException {
        return getKeyInfo(pkey, true);
    }

    /**
     * Print the user info for a PGP public key ring. Returns as JSON
     *
     * @param pkey
     * @return
     * @throws PGPException
     * @throws SignatureException
     */
    public static JsonArray getKeyInfo(PGPPublicKeyRing pkey, boolean htmlSafe) throws PGPException, SignatureException {
        JsonArray keyRingData = new JsonArray();
        Iterator<PGPPublicKey> keys = pkey.getPublicKeys();
        PGPPublicKey key = null;
        while (keys.hasNext()) {
            key = keys.next();
            JsonObject pubKey = new JsonObject();
            pubKey.addProperty("Key", LongUtil.longToHexStringTruncated(key.getKeyID()));
            pubKey.addProperty("KeyID", LongUtil.longToHexString(key.getKeyID()));
            pubKey.addProperty("KeyLong", Long.toString(key.getKeyID()));
            pubKey.addProperty("Fingerprint", PGPUtil.getFingerPrintInBlocks(key.getFingerprint()));
            pubKey.addProperty("Created", key.getCreationTime().toString());
            pubKey.addProperty("Master", key.isMasterKey());
            pubKey.addProperty("Encryption", PGPUtil.isEncryptionKey(key));
            pubKey.addProperty("Revoked", key.hasRevocation());
            boolean expired = false;
            long validSeconds = key.getValidSeconds();
            if (validSeconds != 0) {  // 0 expiration time means doesn't expire
                Date now = new Date();
                long timeLeft = key.getCreationTime().getTime() + (validSeconds * 1000) - now.getTime();
                if (timeLeft < 0) {
                    expired = true;
                } else {
                    pubKey.addProperty("Expires", timeLeft / 86400000);  // Number of days left
                }
            } else {
                pubKey.addProperty("Expires", "never");
            }
            pubKey.addProperty("Expired", expired);
            Iterator it = key.getUserIDs();
            int i = 0;
            JsonArray ids = new JsonArray();
            while (it.hasNext()) {
                if (htmlSafe) {
                    String htmlEscapedId = org.apache.commons.lang.StringEscapeUtils.escapeHtml((String) it.next());
                    ids.add(new JsonPrimitive(htmlEscapedId));
                } else {
                    ids.add(new JsonPrimitive((String) it.next()));
                }
            }
            pubKey.add("ids", ids);
            Iterator<PGPUserAttributeSubpacketVector> im = key.getUserAttributes();
            int imagecount = 0;
            JsonArray images = new JsonArray();
            while (im.hasNext()) {
                PGPUserAttributeSubpacketVector vector = im.next();
                ImageAttribute image = vector.getImageAttribute();
                if (image != null) {
                    JsonObject imageJson = new JsonObject();
                    imageJson.add("type", new JsonPrimitive(image.getType()));
                    if (image.getType() == 1) {
                        imageJson.addProperty("data", "data:image/jpg;base64," + Base64.encodeBase64String(image.getImageData()));
                    }
                    imageJson.add("index", new JsonPrimitive(imagecount));
                    imageJson.addProperty("keyid", Long.toString(key.getKeyID()));
                    imagecount++;
                    images.add(imageJson);
                }

            }
            pubKey.add("images", images);
            it = key.getSignatures();
            i = 0;
            while (it.hasNext()) {
                PGPSignature sig = (PGPSignature) it.next();
                String sigid = LongUtil.longToHexStringTruncated(sig.getKeyID());
                //	if (!sigid.equals(self)) {
                pubKey.addProperty(i++ + "-" + PGPUtil.getSignatureType(sig.getSignatureType()) + "- Signed by", sigid);
                //	}
            }
            keyRingData.add(pubKey);
        }
        return (keyRingData);

    }
}
