/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgp;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.security.NoSuchProviderException;
import java.security.SecureRandom;
import java.security.SignatureException;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.bcpg.BCPGOutputStream;
import org.bouncycastle.openpgp.PGPCompressedData;
import org.bouncycastle.openpgp.PGPCompressedDataGenerator;
import org.bouncycastle.openpgp.PGPEncryptedData;
import org.bouncycastle.openpgp.PGPEncryptedDataGenerator;
import org.bouncycastle.openpgp.PGPEncryptedDataList;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPLiteralData;
import org.bouncycastle.openpgp.PGPLiteralDataGenerator;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.bouncycastle.openpgp.PGPOnePassSignature;
import org.bouncycastle.openpgp.PGPOnePassSignatureList;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyEncryptedData;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureGenerator;
import org.bouncycastle.openpgp.PGPSignatureList;
import org.bouncycastle.openpgp.PGPSignatureSubpacketGenerator;
import org.bouncycastle.openpgp.PGPUtil;
import org.bouncycastle.openpgp.operator.bc.BcKeyFingerprintCalculator;
import org.bouncycastle.openpgp.operator.bc.BcPGPDataEncryptorBuilder;
import org.bouncycastle.openpgp.operator.bc.BcPublicKeyKeyEncryptionMethodGenerator;
import org.bouncycastle.openpgp.operator.jcajce.JcaPGPContentSignerBuilder;
import org.bouncycastle.openpgp.operator.jcajce.JcaPGPContentVerifierBuilderProvider;
import org.bouncycastle.openpgp.operator.jcajce.JcePublicKeyDataDecryptorFactoryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.common.util.LongUtil;
import com.openexchange.guard.exceptions.GuardAuthExceptionCodes;
import com.openexchange.guard.hkpclient.services.HKPClientService;
import com.openexchange.guard.hkpclient.services.RemoteKeyResult;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.keys.PGPKeyService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.pgp.exceptions.GuardPGPExceptionCodes;
import com.openexchange.guard.session.GuardSessionService;

public class PGPUtils {

    private static Logger LOG = LoggerFactory.getLogger(PGPUtils.class);

    private static final int bufftest = 256;

    /**
     * Decode PGP item and return decoded in PGPResult
     *
     * @param in
     * @param userid
     * @param cid
     * @param password
     * @return PGP Result with decoded in result.decoded
     * @throws Exception
     */
    public static PGPResult decryptFile(InputStream in, int userid, int cid, String password) throws Exception {
        ByteArrayOutputStream fOut = new ByteArrayOutputStream();
        PGPResult result = decryptFile(in, fOut, userid, cid, password, true, null);
        result.setDecoded(fOut.toByteArray());
        return (result);
    }

    /**
     * Decode PGP item and return decoded in PGPResult
     * @param in
     * @param fOut
     * @param userid
     * @param cid
     * @param password
     * @param close
     * @return
     * @throws IOException
     * @throws OXException
     * @throws PGPException
     */
    public static PGPResult decryptFile(InputStream in, OutputStream fOut, int userid, int cid, String password, boolean close) throws IOException, OXException, PGPException {
        return decryptFile (in, fOut, userid, cid, password, close, null);
    }

    /**
     * Decrypt PGP Content
     *
     * @param in The input stream to decrypt
     * @param fOutT he output stream writing the decrypted data to
     * @param userid The user's id
     * @param cid
     * @param password The user's password
     * @param close Output stream should be closed and flushed
     * @param The From address of sender if signature attached
     * @return PGPResult The result
     * @throws IOException
     * @throws OXException
     * @throws PGPException
     */
    public static PGPResult decryptFile(InputStream in, OutputStream fOut, int userid, int cid, String password, boolean close, String fromAddress) throws IOException, OXException, PGPException {

        if (password == null || password.isEmpty()) {
            LOG.info("Bad password");
            throw GuardPGPExceptionCodes.BAD_PASSWORD_ERROR.create();
        }
        PGPResult result = new PGPResult();
        in = PGPUtil.getDecoderStream(in);
        PGPObjectFactory pgpF = new PGPObjectFactory(in, new BcKeyFingerprintCalculator());
        PGPEncryptedDataList enc;

        //reading first part of the stream
        Object o = pgpF.nextObject();
        if (o == null) {
            result.setError("No Encrypted items");
            return (result);
        }

        //the first object might be a PGP marker packet.
        if (o instanceof PGPEncryptedDataList) {
            enc = (PGPEncryptedDataList) o;
        } else {
            enc = (PGPEncryptedDataList) pgpF.nextObject();
        }

        // find the secret key
        Iterator<?> it = enc.getEncryptedDataObjects();
        PGPPrivateKey sKey = null;
        PGPPublicKeyEncryptedData pbe = null;
        StringBuilder missing = new StringBuilder();
        while (sKey == null && it.hasNext()) {
            pbe = (PGPPublicKeyEncryptedData) it.next();
            GuardKeys keys = PGPPublicHandler.getKeyForPGP(userid, cid, pbe.getKeyID(), true);
            if (keys != null) {
                //For upgrade from prior system.  Eventually can remove
                if (keys.getVersion() < 0) {  // Look if we need upgraded UUID
                    try {
                        PGPKeyService pgpKeyService = Services.getService(PGPKeyService.class);
                        pgpKeyService.checkBadUID(keys, password);
                    } catch (Exception e) {
                        LOG.info("Bad password");
                        throw GuardPGPExceptionCodes.BAD_PASSWORD_ERROR.create();
                    }
                }
                PGPSecretKeyRing PGPSecretKR = keys.getPGPSecretKeyRing();
                if (PGPSecretKR == null) {
                    result.setError("No Private Key associated with the Public Key");
                    return (result);
                }
                PGPSecretKey protectedSKey = PGPSecretKR.getSecretKey(pbe.getKeyID());
                if (protectedSKey == null) {
                    result.setError("No Private Key associated with the Public Key");
                    return (result);
                }
                try {
                    sKey = com.openexchange.guard.keymanagement.commons.util.PGPUtil.decodePrivate(protectedSKey, password, keys.getSalt());
                } catch (org.bouncycastle.openpgp.PGPException ex) {
                    LOG.info("Bad password");
                    throw GuardPGPExceptionCodes.BAD_PASSWORD_ERROR.create();
                }
            } else {
                missing.append(LongUtil.longToHexStringTruncated(pbe.getKeyID()) + " ");
            }
        }
        if (sKey == null) {
            result.setError("No secret key : " + missing.toString());
            LOG.debug("No secret key");
            return (result);
        }

        InputStream clear = pbe.getDataStream(new JcePublicKeyDataDecryptorFactoryBuilder().setProvider("BC").build(sKey));
        PGPObjectFactory plainFact = new PGPObjectFactory(clear, new BcKeyFingerprintCalculator());
        PGPOnePassSignatureList onePassSignatureList = null;
        PGPOnePassSignature ops = null;
        PGPPublicKey publicKey = null;

        //Parsing messages
        Object message = plainFact.nextObject();
        boolean signatureInitialized = false;
        while (message != null) {
            if (message instanceof PGPCompressedData) {
                PGPCompressedData compressedData = (PGPCompressedData) message;
                plainFact = new PGPObjectFactory(compressedData.getDataStream(), new BcKeyFingerprintCalculator());
                message = plainFact.nextObject();
            }

            if (message instanceof PGPLiteralData) {
                //processing the message
                PGPLiteralData pgpLiteralData = (PGPLiteralData) message;
                InputStream literalInputStream = pgpLiteralData.getInputStream();
                byte[] buffer = new byte[bufftest];
                int len = 0;
                while ((len = literalInputStream.read(buffer)) > -1) {
                    if (result.isSignature() && signatureInitialized) {
                        //If we have a signature we are going to update the OPS for verifying the signature
                        ops.update(buffer, 0, len);
                    }
                    ;
                    //Writing the decrypted message to the output stream
                    fOut.write(buffer, 0, len);
                }
                message = plainFact.nextObject();
            } else if (message instanceof PGPOnePassSignatureList) {
                //Initializing the signature
                onePassSignatureList = (PGPOnePassSignatureList) message;
                ops = onePassSignatureList.get(0);
                PGPPublicKeyRing pgpRing = null;

                // may not work due to static method
                PGPKeyService pgpKeyService = Services.getService(PGPKeyService.class);
                pgpRing = pgpKeyService.getPublicKeyRingById(ops.getKeyID(), userid, cid);
                if (pgpRing == null) {
                    pgpRing = checkPublicKeys(ops.getKeyID(), fromAddress, result);
                }
                result.setSignature(true);
                if (pgpRing != null) {
                    publicKey = pgpRing.getPublicKey(ops.getKeyID());
                    ops.init(new JcaPGPContentVerifierBuilderProvider().setProvider("BC"), publicKey);
                    signatureInitialized = true;
                } else {
                    result.setMissingPublicKey(true);
                    result.setVerified(false);
                }
                message = plainFact.nextObject();

            } else if (message instanceof PGPSignatureList) {
                //Verify signatures
                PGPSignatureList signatureList = (PGPSignatureList) message;
                if (result.isSignature() && signatureInitialized) {
                    for (int i = 0; i < onePassSignatureList.size(); i++) {
                        PGPSignature signature = signatureList.get(i);
                        try {
                            if (ops.verify(signature)) {
                                Iterator<?> userIds = publicKey.getUserIDs();
                                while (userIds.hasNext()) {
                                    String userId = (String) userIds.next();
                                    LOG.debug("Signed by " + userId);
                                }
                                result.setVerified(true);
                                LOG.info("Signature verified");
                            }
                        } catch (Exception e) {
                            result.setVerified(false);
                            LOG.info("Unable to verify signature");
                        }
                    }
                }
                message = plainFact.nextObject();
            } else {
                result.setError("Unk type");
                break;
            }
        }
        return result;
    }

    /**
     * Verify signature in inline signature
     *
     * @param in
     * @param data
     * @param userid
     * @param cid
     * @return
     * @throws IOException
     * @throws SignatureException
     * @throws PGPException
     * @throws OXException
     */
    public PGPResult verifyInlineSignature(InputStream in, String data, int userid, int cid, String fromAddress) throws IOException, SignatureException, PGPException, OXException {

        com.openexchange.guard.pgp.PGPResult result = new com.openexchange.guard.pgp.PGPResult();
        in = PGPUtil.getDecoderStream(in);

        PGPObjectFactory pgpF = new PGPObjectFactory(in, new BcKeyFingerprintCalculator());

        Object o = pgpF.nextObject();
        if (o == null) {
            result.setError("No Encrypted items");
            return (result);
        }
        //
        // the first object might be a PGP marker packet.
        //
        if (o instanceof PGPSignatureList) {
            result.setSignature(true);
            PGPSignature signature = ((PGPSignatureList) o).get(0);
            PGPKeyService pgpKeyService = Services.getService(PGPKeyService.class);
            PGPPublicKeyRing pgpRing = pgpKeyService.getPublicKeyRingById(signature.getKeyID(), userid, cid);
            if (pgpRing == null) {
                pgpRing = checkPublicKeys(signature.getKeyID(), fromAddress, result);
                if (pgpRing == null) {
                    LOG.info("Unable to find pgpRing for sender");
                    result.setError("No public key");
                    result.setVerified(false);
                    result.setMissingPublicKey(true);
                    return (result);
                }
            }
            PGPPublicKey publicKey = pgpRing.getPublicKey(signature.getKeyID());
            signature.init(new JcaPGPContentVerifierBuilderProvider().setProvider("BC"), publicKey);
            signature.update(data.getBytes("UTF-8"));
            if (signature.verify()) {
                result.setVerified(true);
            } else {
                result.setVerified(false);
            }

        } else {
            result.setSignature(false);
            // no signature
        }
        return (result);
    }

    /**
     * Verify seperate signature
     *
     * @param data
     * @param signature
     * @param userid
     * @param cid
     * @return
     * @throws IOException
     * @throws PGPException
     * @throws OXException
     * @throws Exception
     */
    public PGPResult verifySignature(String data, String signature, int userid, int cid, String fromAddress) throws IOException, PGPException, OXException {
        PGPResult result = new PGPResult();
        InputStream datastream = new ByteArrayInputStream(signature.getBytes(StandardCharsets.UTF_8));
        InputStream in = PGPUtil.getDecoderStream(datastream);
        PGPObjectFactory factory = new PGPObjectFactory(in, new BcKeyFingerprintCalculator());
        PGPSignatureList list = null;
        Object o = factory.nextObject();
        if (o == null) {
            LOG.error("No signature found");
            result.setError("No signature found");
            result.setSignature(false);
            return (result);
        }
        if (o instanceof PGPCompressedData) {
            PGPCompressedData c1 = (PGPCompressedData) o;
            factory = new PGPObjectFactory(c1.getDataStream(), new BcKeyFingerprintCalculator());
            list = (PGPSignatureList) factory.nextObject();
        } else {
            list = (PGPSignatureList) o;
        }

        InputStream sigfile = new ByteArrayInputStream(data.getBytes(StandardCharsets.UTF_8));
        int ch;
        if (list.size() > 0) {
            result.setSignature(true);
        } else {
            result.setError("No signatures");
            return (result);
        }
        PGPSignature sig = list.get(0);
        LOG.debug("Verifying signature for key " + sig.getKeyID());
        PGPKeyService pgpKeyService = Services.getService(PGPKeyService.class);
        PGPPublicKeyRing pgpRing = pgpKeyService.getPublicKeyRingById(sig.getKeyID(), userid, cid);
        if (pgpRing == null) {
            pgpRing = checkPublicKeys(sig.getKeyID(), fromAddress, result);
            if (pgpRing == null) {
                LOG.info("Unable to find pgpRing for sender");
                result.setError("No public key");
                result.setVerified(false);
                result.setMissingPublicKey(true);
                return (result);
            }
        }
        //	PGPPublicKeyRing pgpRing = gkey.pubring;

        PGPPublicKey key = pgpRing.getPublicKey(sig.getKeyID());
        if (key == null) {
            LOG.info("Unable to find pgp public key for sender");
            result.setError("No public key");
            result.setVerified(false);
            result.setMissingPublicKey(true);
            return (result);
        }
        sig.init(new JcaPGPContentVerifierBuilderProvider(), key);
        while ((ch = sigfile.read()) >= 0) {
            sig.update((byte) ch);
        }
        sigfile.close();
        in.close();
        datastream.close();
        if (sig.verify()) {
            result.setVerified(true);
        } else {
            result.setVerified(false);
        }
        return (result);

    }

    private static PGPPublicKeyRing checkPublicKeys (Long id, String fromEmail, PGPResult result) throws OXException {
        HKPClientService hkpClient = Services.getService(HKPClientService.class);
        RemoteKeyResult remote = hkpClient.find(null, fromEmail, id);
        if (remote != null && remote.hasResult()) {
            if (remote.getSource().isTrusted()) {
                result.setSource(remote.getSource());
                return remote.getRing();
            }
        }
        return null;

    }

    /**
     * Encrypt byte[] data
     *
     * @param out
     * @param data
     * @param fileName
     * @param recipients
     * @param armor
     * @param withIntegrityCheck
     * @throws IOException
     * @throws NoSuchProviderException
     * @throws PGPException
     * @throws OXException
     */
    public static void encrypt(OutputStream out, byte[] data, String fileName, List<RecipKey> recipients, boolean armor, boolean withIntegrityCheck) throws IOException, NoSuchProviderException, PGPException, OXException {
        ByteArrayInputStream in = new ByteArrayInputStream(data);
        encrypt(out, in, fileName, recipients, armor, withIntegrityCheck);
    }

    /**
     * Encrypt a PGP Item
     *
     * @param out
     * @param in
     * @param fileName
     * @param recipients List of recipients
     * @param armor Output is armor stream
     * @param withIntegrityCheck
     * @throws IOException
     * @throws NoSuchProviderException
     * @throws PGPException
     * @throws OXException
     */
    public static void encrypt(OutputStream out, InputStream in, String fileName, List<RecipKey> recipients, boolean armor, boolean withIntegrityCheck) throws IOException, NoSuchProviderException, PGPException, OXException {
        if (armor) {
            out = new ArmoredOutputStream(out);
        }

        BcPGPDataEncryptorBuilder builder = new BcPGPDataEncryptorBuilder(PGPEncryptedData.AES_256);
        builder.setSecureRandom(new SecureRandom());
        builder.setWithIntegrityPacket(true);
        PGPEncryptedDataGenerator encryptedDataGenerator = new PGPEncryptedDataGenerator(builder);
        for (int i = 0; i < recipients.size(); i++) {
            if (recipients.get(i).getPGPPublicKeyRing() != null) {
                PGPPublicKey pubkey = getCryptKey(recipients.get(i).getPGPPublicKeyRing());
                if (pubkey == null) {  // If no valid key found, quit
                    throw GuardPGPExceptionCodes.NO_VALID_KEY.create();
                }
                BcPublicKeyKeyEncryptionMethodGenerator encKeyGen = new BcPublicKeyKeyEncryptionMethodGenerator(pubkey);
                encryptedDataGenerator.addMethod(encKeyGen);
            }
        }

        OutputStream encryptedOut = encryptedDataGenerator.open(out, new byte[4028]);

        PGPCompressedDataGenerator cGen = new PGPCompressedDataGenerator(PGPCompressedData.ZLIB);

        BCPGOutputStream bOut = new BCPGOutputStream(cGen.open(encryptedOut));

        PGPLiteralDataGenerator lGen = new PGPLiteralDataGenerator();
        OutputStream lOut = lGen.open(bOut, PGPLiteralData.BINARY, "encrypted.asc", new Date(), new byte[256]);

        byte[] buffer = new byte[bufftest];
        int len = 0;
        while ((len = in.read(buffer)) > -1) {
            lOut.write(buffer, 0, len);
        }

        lGen.close();
        cGen.close();
        encryptedOut.close();
        if (armor) {
            out.close();
        }

    }

    /**
     * Return encryption key. Will return master if no other found
     *
     * @param ring
     * @return
     */
    private static PGPPublicKey getCryptKey(PGPPublicKeyRing ring) {
        PGPPublicKey found = null;
        Iterator it = ring.getPublicKeys();
        while (it.hasNext()) {
            PGPPublicKey key = (PGPPublicKey) it.next();
            if (com.openexchange.guard.keymanagement.commons.util.PGPUtil.isEncryptionKey(key)) {
                if (key.isMasterKey() && !com.openexchange.guard.keymanagement.commons.util.PGPUtil.isExpired(key) && !key.hasRevocation()) {  // If master key, we will use only if we don't have another encryption key
                    if (found == null) {
                        found = key;
                    }
                } else {
                    if (!key.hasRevocation()) {
                        if (!com.openexchange.guard.keymanagement.commons.util.PGPUtil.isExpired(key)) {
                            return (key);
                        }
                    }
                }
            }
        }
        return found;
    }

    public static void signEncryptFile(OutputStream out, byte[] data, String fileName, List<RecipKey> recipients, GuardKeys keys, String password, boolean armor, boolean withIntegrityCheck) throws Exception {
        ByteArrayInputStream in = new ByteArrayInputStream(data);
        signEncryptFile(out, in, fileName, recipients, keys, password, armor, withIntegrityCheck);
    }

    /**
     * Encrypt and sign PGP item with onepass signature
     *
     * @param out
     * @param dataStream
     * @param fileName
     * @param recipients
     * @param keys
     * @param password
     * @param armor
     * @param withIntegrityCheck
     * @throws Exception
     */
    public static void signEncryptFile(OutputStream out, InputStream dataStream, String fileName, List<RecipKey> recipients, GuardKeys keys, String password, boolean armor, boolean withIntegrityCheck) throws Exception {
        if (armor) {
            out = new ArmoredOutputStream(out);
        }

        PGPKeyService pgpKeyService = Services.getService(PGPKeyService.class);
        PGPPrivateKey pgpSignKey;
        try {
            pgpSignKey = pgpKeyService.getPGPSignKey(keys.getPGPSecretKeyRing(), password, keys.getSalt());
        } catch (OXException ex) {
            throw GuardAuthExceptionCodes.BAD_PASSWORD.create();
        }
        if (pgpSignKey == null) {
            throw GuardPGPExceptionCodes.UNABLE_TO_DECODE.create(keys.getUserid(), keys.getContextid());
        }
        PGPPublicKey pubkey = keys.getPGPPublicKeyRing().getPublicKey(pgpSignKey.getKeyID());  // Get the public key specific to the signing key
        PGPSignatureGenerator sGen = new PGPSignatureGenerator(new JcaPGPContentSignerBuilder(pubkey.getAlgorithm(), PGPUtil.SHA512).setProvider("BC"));

        sGen.init(PGPSignature.BINARY_DOCUMENT, pgpSignKey);

        Iterator it = pubkey.getUserIDs();
        if (it.hasNext()) {
            PGPSignatureSubpacketGenerator spGen = new PGPSignatureSubpacketGenerator();

            spGen.setSignerUserID(false, (String) it.next());
            sGen.setHashedSubpackets(spGen.generate());
        }

        BcPGPDataEncryptorBuilder builder = new BcPGPDataEncryptorBuilder(PGPEncryptedData.AES_256);
        builder.setSecureRandom(new SecureRandom());
        builder.setWithIntegrityPacket(true);
        PGPEncryptedDataGenerator encryptedDataGenerator = new PGPEncryptedDataGenerator(builder);
        for (int i = 0; i < recipients.size(); i++) {
            BcPublicKeyKeyEncryptionMethodGenerator encKeyGen = new BcPublicKeyKeyEncryptionMethodGenerator(getCryptKey(recipients.get(i).getPGPPublicKeyRing()));
            encryptedDataGenerator.addMethod(encKeyGen);
        }
        OutputStream encryptedOut = encryptedDataGenerator.open(out, new byte[4028]);

        PGPCompressedDataGenerator cGen = new PGPCompressedDataGenerator(PGPCompressedData.ZLIB);

        BCPGOutputStream bOut = new BCPGOutputStream(cGen.open(encryptedOut));

        sGen.generateOnePassVersion(false).encode(bOut);

        PGPLiteralDataGenerator lGen = new PGPLiteralDataGenerator();
        OutputStream lOut = lGen.open(bOut, PGPLiteralData.BINARY, "encrypted.asc", new Date(), new byte[256]);

        byte[] buffer = new byte[bufftest];
        int len = 0;
        while ((len = dataStream.read(buffer)) > -1) {
            lOut.write(buffer, 0, len);
            sGen.update(buffer, 0, len);
        }

        lGen.close();

        sGen.generate().encode(bOut);

        cGen.close();
        encryptedOut.close();
        if (armor) {
            out.close();
        }
    }

    /**
     * Create a detached signature of inputstream
     *
     * @param in
     * @param out
     * @param key
     * @param password
     * @param armor
     * @throws IOException
     * @throws SignatureException
     * @throws PGPException
     * @throws OXException
     */
    public void createDetachedSignature(InputStream in, OutputStream out, GuardKeys key, String password, boolean armor) throws IOException, SignatureException, PGPException, OXException {

        if (armor) {
            out = new ArmoredOutputStream(out);
        }

        int algorithm = com.openexchange.guard.keymanagement.commons.util.PGPUtil.getSigningKey(key.getPGPSecretKeyRing()).getPublicKey().getAlgorithm();
        PGPSignatureGenerator sgen = new PGPSignatureGenerator(new JcaPGPContentSignerBuilder(algorithm, PGPUtil.SHA512).setProvider("BC"));
        PGPKeyService pgpKeyService = Services.getService(PGPKeyService.class);
        PGPPrivateKey pgpSignKey;
        try {
            pgpSignKey = pgpKeyService.getPGPSignKey(key.getPGPSecretKeyRing(), password, key.getSalt());
        } catch (OXException ex) {
            throw GuardAuthExceptionCodes.BAD_PASSWORD.create();
        }

        sgen.init(PGPSignature.BINARY_DOCUMENT, pgpSignKey);

        BCPGOutputStream bout = new BCPGOutputStream(out);
        byte[] buffer = new byte[bufftest];
        int len = 0;
        while ((len = in.read(buffer)) > -1) {
            sgen.update(buffer, 0, len);
        }
        sgen.generate().encode(bout);
        out.close();

    }

    public static String encodeEPass(String password, String jSessionId, int userid, int cid) throws Exception {
        GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
        GuardSessionService sessionService = Services.getService(GuardSessionService.class);
        String token = sessionService.getToken(jSessionId);
        if (token == null) {
            token = sessionService.newToken(jSessionId, userid, cid);
        }
        return (cipherService.encrypt(password, token));
    }

    public static String decodeEPass(String epass, String sessionId) throws OXException {
        GuardSessionService sessionService = Services.getService(GuardSessionService.class);
        String token = sessionService.getToken(sessionId);
        GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
        return (cipherService.decrypt(epass, token));
    }
}
