/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.mailvelope;

import java.util.ArrayList;
import java.util.Iterator;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.bouncycastle.openpgp.PGPKeyRing;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonObject;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.email.storage.ogEmail.Email;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.guestupgrade.GuestUpgradeService;
import com.openexchange.guard.handler.ResponseHandler;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.util.PGPMergeUtil;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.storage.KeyTableStorage;
import com.openexchange.guard.keymanagement.storage.PGPKeysStorage;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.pgp.PGPPrivateKeyHandler;
import com.openexchange.guard.pgp.PGPPublicHandler;

public class PublicKeyImporter {

    private static Logger logger = LoggerFactory.getLogger(PublicKeyImporter.class);

    /**
     * Import Public key from mailvelope
     * Requires uid to be the primary for the account
     *
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void importPublicKey(HttpServletRequest request, HttpServletResponse response, int userid, int cid, OxCookie cookie) throws Exception {
        JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
        // Verify logged in
        Api ap = new Api(cookie, request);
        if (!ap.verifyLogin()) {
            ServletUtils.sendError(response, "Must be logged in");
            return;
        }
        ArrayList<String> emails = ap.getUserEmails();
        String data = null;
        String email = null;
        try {
            email = json.get("email").getAsString();
            if (!emails.contains(email)) {
                ServletUtils.sendError(response, "Email not in account");
                return;
            }
            /////   Restrict to primary email address here ?
            if (!email.toLowerCase().equals(ap.getPrimary().toLowerCase())) {
                ServletUtils.sendError(response, "Email not the primary");
                return;
            }

            if (cid != json.get("cid").getAsInt()) {
                ServletUtils.sendError(response, "Bad login data");
                return;
            }
            data = json.get("keydata").getAsString();
        } catch (Exception e) {
            ServletUtils.sendError(response, "Bad data");
            logger.error("Error getting mailvelope public key json ", e);
            return;
        }
        if (!data.contains("PGP PUBLIC")) {
            ServletUtils.sendError(response, "Bad key data");
            return;
        }
        // Create new key with the data
        GuardKeys newkey = new GuardKeys();
        newkey.setContextid(cid);
        newkey.setUserid(userid);
        newkey.setEmail(email);
        newkey.setPGPKeyRingFromAsc(data);

        // Make sure the key has the email in userids
        if (!hasUserID(newkey.getPGPPublicKeyRing(), email)) {
            logger.debug("User email address not found in the uploaded public keys");
            ServletUtils.sendError(response, "Bad UID");
            return;
        }

        PGPPrivateKeyHandler pk = new PGPPrivateKeyHandler();
        Long dupl;
        // Now, we need to check to make sure we don't already have this key
        boolean duplicate = false;
        if ((dupl = pk.isDuplicate(newkey)) != 0) {  // If public key already registered, possible duplicate
            duplicate = true;
        }
        if (duplicate) {
            //////////// HANDLE MERGE/DUPLICATE public keys
            GuardKeys existing = PGPPublicHandler.getKeyForPGP(userid, cid, dupl, true);
            if (newkey.getPGPSecretKey() != null) {
                ServletUtils.sendError(response, "Private key exists, merge not supported through mailvelope");
                return;
            }
            PGPPublicKeyRing mergedKeyRings = PGPMergeUtil.mergeKeyRings(newkey.getPGPPublicKeyRing(), existing.getPGPPublicKeyRing());
            if (mergedKeyRings == null) {
                logger.error("Problem handling duplicate public key");
                ServletUtils.sendError(response, "duplicate error");
                return;
            }
            else {
                newkey.setPGPPublicKeyRing(mergedKeyRings);
                PGPKeysStorage keysStorage = Services.getService(PGPKeysStorage.class);
                keysStorage.addPublicKeyIndex(existing.getContextid(), existing.getKeyid(), existing.getEmail(), existing.getPGPPublicKeyRing());
            }
            KeyTableStorage ks = Services.getService(KeyTableStorage.class);
            ks.updatePublicKey(newkey);

        } else {
            EmailStorage eog = Services.getService(EmailStorage.class);
            Email lookup = eog.getByEmail(email);
            if (lookup != null) {
                if (lookup.getContextId() < 0) {
                    GuestUpgradeService guestUpgradeService = Services.getService(GuestUpgradeService.class);
                    guestUpgradeService.upgrade(lookup.getUserId(), lookup.getContextId(), userid, cid);
                }
            } else {
                eog.insertOrUpdate(email, cid, userid, 0);
            }
            GuardKeyService ks = Services.getService(GuardKeyService.class);
            ks.addNewKey(newkey);
            ServletUtils.sendHtmlOK(response, "OK");
        }
    }

    /**
     * Verify has the email in userid in the ring
     *
     * @param ring
     * @param email
     * @return
     */
    private boolean hasUserID(PGPKeyRing ring, String email) {
        Iterator<String> uidIterator = ring.getPublicKey().getUserIDs();

        while (uidIterator.hasNext()) {
            String uid = uidIterator.next().toLowerCase();
            if (uid.contains(email)) {
                return true;
            }
        }

        return (false);
    }

}
